"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FilterGroup = exports.EventAction = exports.Source = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const source_types_1 = require("./source-types");
/**
 * Source provider definition for a CodeBuild Project.
 */
class Source {
    constructor(props) {
        this.badgeSupported = false;
        this.identifier = props.identifier;
    }
    static s3(props) {
        return new S3Source(props);
    }
    static codeCommit(props) {
        return new CodeCommitSource(props);
    }
    static gitHub(props) {
        return new GitHubSource(props);
    }
    static gitHubEnterprise(props) {
        return new GitHubEnterpriseSource(props);
    }
    static bitBucket(props) {
        return new BitBucketSource(props);
    }
    /**
     * Called by the project when the source is added so that the source can perform
     * binding operations on the source. For example, it can grant permissions to the
     * code build project to read from the S3 bucket.
     */
    bind(_scope, _project) {
        return {
            sourceProperty: {
                sourceIdentifier: this.identifier,
                type: this.type,
            },
        };
    }
}
exports.Source = Source;
/**
 * A common superclass of all build sources that are backed by Git.
 */
class GitSource extends Source {
    constructor(props) {
        super(props);
        this.cloneDepth = props.cloneDepth;
        this.branchOrRef = props.branchOrRef;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceVersion: this.branchOrRef,
            sourceProperty: {
                ...superConfig.sourceProperty,
                gitCloneDepth: this.cloneDepth,
            },
        };
    }
}
/**
 * The types of webhook event actions.
 */
var EventAction;
(function (EventAction) {
    /**
     * A push (of a branch, or a tag) to the repository.
     */
    EventAction["PUSH"] = "PUSH";
    /**
     * Creating a Pull Request.
     */
    EventAction["PULL_REQUEST_CREATED"] = "PULL_REQUEST_CREATED";
    /**
     * Updating a Pull Request.
     */
    EventAction["PULL_REQUEST_UPDATED"] = "PULL_REQUEST_UPDATED";
    /**
     * Merging a Pull Request.
     */
    EventAction["PULL_REQUEST_MERGED"] = "PULL_REQUEST_MERGED";
    /**
     * Re-opening a previously closed Pull Request.
     * Note that this event is only supported for GitHub and GitHubEnterprise sources.
     */
    EventAction["PULL_REQUEST_REOPENED"] = "PULL_REQUEST_REOPENED";
})(EventAction = exports.EventAction || (exports.EventAction = {}));
const FILE_PATH_WEBHOOK_COND = 'FILE_PATH';
/**
 * An object that represents a group of filter conditions for a webhook.
 * Every condition in a given FilterGroup must be true in order for the whole group to be true.
 * You construct instances of it by calling the {@link #inEventOf} static factory method,
 * and then calling various `andXyz` instance methods to create modified instances of it
 * (this class is immutable).
 *
 * You pass instances of this class to the `webhookFilters` property when constructing a source.
 */
class FilterGroup {
    constructor(actions, filters) {
        if (actions.size === 0) {
            throw new Error('A filter group must contain at least one event action');
        }
        this.actions = actions;
        this.filters = filters;
    }
    /**
     * Creates a new event FilterGroup that triggers on any of the provided actions.
     *
     * @param actions the actions to trigger the webhook on
     */
    static inEventOf(...actions) {
        return new FilterGroup(new Set(actions), []);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIs(branchName) {
        return this.addHeadBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIsNot(branchName) {
        return this.addHeadBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIs(tagName) {
        return this.addHeadTagFilter(tagName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIsNot(tagName) {
        return this.addHeadTagFilter(tagName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIs(pattern) {
        return this.addHeadRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIsNot(pattern) {
        return this.addHeadRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIs(pattern) {
        return this.addActorAccountId(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must not match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIsNot(pattern) {
        return this.addActorAccountId(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIs(branchName) {
        return this.addBaseBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIsNot(branchName) {
        return this.addBaseBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIs(pattern) {
        return this.addBaseRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIsNot(pattern) {
        return this.addBaseRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIs(pattern) {
        return this.addFilePathFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must not affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIsNot(pattern) {
        return this.addFilePathFilter(pattern, false);
    }
    /** @internal */
    get _actions() {
        return set2Array(this.actions);
    }
    /** @internal */
    get _filters() {
        return this.filters.slice();
    }
    /** @internal */
    _toJson() {
        const eventFilter = {
            type: 'EVENT',
            pattern: set2Array(this.actions).join(', '),
        };
        return [eventFilter].concat(this.filters);
    }
    addHeadBranchFilter(branchName, include) {
        return this.addHeadRefFilter(`refs/heads/${branchName}`, include);
    }
    addHeadTagFilter(tagName, include) {
        return this.addHeadRefFilter(`refs/tags/${tagName}`, include);
    }
    addHeadRefFilter(refName, include) {
        return this.addFilter('HEAD_REF', refName, include);
    }
    addActorAccountId(accountId, include) {
        return this.addFilter('ACTOR_ACCOUNT_ID', accountId, include);
    }
    addBaseBranchFilter(branchName, include) {
        return this.addBaseRefFilter(`refs/heads/${branchName}`, include);
    }
    addBaseRefFilter(refName, include) {
        if (this.actions.has(EventAction.PUSH)) {
            throw new Error('A base reference condition cannot be added if a Group contains a PUSH event action');
        }
        return this.addFilter('BASE_REF', refName, include);
    }
    addFilePathFilter(pattern, include) {
        return this.addFilter(FILE_PATH_WEBHOOK_COND, pattern, include);
    }
    addFilter(type, pattern, include) {
        return new FilterGroup(this.actions, this.filters.concat([{
                type,
                pattern,
                excludeMatchedPattern: include ? undefined : true,
            }]));
    }
}
exports.FilterGroup = FilterGroup;
/**
 * A common superclass of all third-party build sources that are backed by Git.
 */
class ThirdPartyGitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.webhook = props.webhook;
        this.reportBuildStatus = props.reportBuildStatus === undefined ? true : props.reportBuildStatus;
        this.webhookFilters = props.webhookFilters || [];
    }
    bind(_scope, _project) {
        const anyFilterGroupsProvided = this.webhookFilters.length > 0;
        const webhook = this.webhook === undefined ? (anyFilterGroupsProvided ? true : undefined) : this.webhook;
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                reportBuildStatus: this.reportBuildStatus,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: webhook === undefined ? undefined : {
                webhook,
                filterGroups: anyFilterGroupsProvided ? this.webhookFilters.map(fg => fg._toJson()) : undefined,
            },
        };
    }
}
/**
 * CodeCommit Source definition for a CodeBuild project.
 */
class CodeCommitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.type = source_types_1.CODECOMMIT_SOURCE_TYPE;
        this.repo = props.repository;
    }
    bind(_scope, project) {
        // https://docs.aws.amazon.com/codebuild/latest/userguide/setting-up.html
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codecommit:GitPull'],
            resources: [this.repo.repositoryArn],
        }));
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.repo.repositoryCloneUrlHttp,
            },
            sourceVersion: superConfig.sourceVersion,
        };
    }
}
/**
 * S3 bucket definition for a CodeBuild project.
 */
class S3Source extends Source {
    constructor(props) {
        super(props);
        this.type = source_types_1.S3_SOURCE_TYPE;
        this.bucket = props.bucket;
        this.path = props.path;
        this.version = props.version;
    }
    bind(_scope, project) {
        this.bucket.grantRead(project);
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: `${this.bucket.bucketName}/${this.path}`,
            },
            sourceVersion: this.version,
        };
    }
}
/**
 * GitHub Source definition for a CodeBuild project.
 */
class GitHubSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_SOURCE_TYPE;
        this.httpsCloneUrl = `https://github.com/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, project) {
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Enterprise Source definition for a CodeBuild project.
 */
class GitHubEnterpriseSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_ENTERPRISE_SOURCE_TYPE;
        this.httpsCloneUrl = props.httpsCloneUrl;
        this.ignoreSslErrors = props.ignoreSslErrors;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                insecureSsl: this.ignoreSslErrors,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * BitBucket Source definition for a CodeBuild project.
 */
class BitBucketSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.BITBUCKET_SOURCE_TYPE;
        this.httpsCloneUrl = `https://bitbucket.org/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, _project) {
        // BitBucket sources don't support the PULL_REQUEST_REOPENED event action
        if (this.anyWebhookFilterContainsPrReopenedEventAction()) {
            throw new Error('BitBucket sources do not support the PULL_REQUEST_REOPENED webhook event action');
        }
        // they also don't support file path conditions
        if (this.anyWebhookFilterContainsFilePathConditions()) {
            throw new Error('BitBucket sources do not support file path conditions for webhook filters');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    anyWebhookFilterContainsPrReopenedEventAction() {
        return this.webhookFilters.findIndex(fg => {
            return fg._actions.findIndex(a => a === EventAction.PULL_REQUEST_REOPENED) !== -1;
        }) !== -1;
    }
    anyWebhookFilterContainsFilePathConditions() {
        return this.webhookFilters.findIndex(fg => {
            return fg._filters.findIndex(f => f.type === FILE_PATH_WEBHOOK_COND) !== -1;
        }) !== -1;
    }
}
function set2Array(set) {
    const ret = [];
    set.forEach(el => ret.push(el));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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