"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegendPosition = exports.Color = exports.Shading = exports.SingleValueWidget = exports.GraphWidget = exports.AlarmWidget = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rendering_1 = require("./private/rendering");
const widget_1 = require("./widget");
/**
 * Display the metric associated with an alarm, including the alarm line
 */
class AlarmWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    annotations: {
                        alarms: [this.props.alarm.alarmArn],
                    },
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                    },
                },
            }];
    }
}
exports.AlarmWidget = AlarmWidget;
/**
 * A dashboard widget that displays metrics
 */
class GraphWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        const horizontalAnnotations = [
            ...(this.props.leftAnnotations || []).map(mapAnnotation('left')),
            ...(this.props.rightAnnotations || []).map(mapAnnotation('right')),
        ];
        const metrics = rendering_1.allMetricsGraphJson(this.props.left || [], this.props.right || []);
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    stacked: this.props.stacked,
                    metrics: metrics.length > 0 ? metrics : undefined,
                    annotations: horizontalAnnotations.length > 0 ? { horizontal: horizontalAnnotations } : undefined,
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                        right: this.props.rightYAxis !== undefined ? this.props.rightYAxis : undefined,
                    },
                    legend: this.props.legendPosition !== undefined ? { position: this.props.legendPosition } : undefined,
                    liveData: this.props.liveData,
                },
            }];
    }
}
exports.GraphWidget = GraphWidget;
/**
 * A dashboard widget that displays the most recent value for every metric
 */
class SingleValueWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 3);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'singleValue',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    metrics: rendering_1.allMetricsGraphJson(this.props.metrics, []),
                    setPeriodToTimeRange: this.props.setPeriodToTimeRange,
                },
            }];
    }
}
exports.SingleValueWidget = SingleValueWidget;
/**
 * Fill shading options that will be used with an annotation
 */
var Shading;
(function (Shading) {
    /**
     * Don't add shading
     */
    Shading["NONE"] = "none";
    /**
     * Add shading above the annotation
     */
    Shading["ABOVE"] = "above";
    /**
     * Add shading below the annotation
     */
    Shading["BELOW"] = "below";
})(Shading = exports.Shading || (exports.Shading = {}));
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 */
class Color {
}
exports.Color = Color;
/** blue - hex #1f77b4 */
Color.BLUE = '#1f77b4';
/** brown - hex #8c564b */
Color.BROWN = '#8c564b';
/** green - hex #2ca02c */
Color.GREEN = '#2ca02c';
/** grey - hex #7f7f7f */
Color.GREY = '#7f7f7f';
/** orange - hex #ff7f0e */
Color.ORANGE = '#ff7f0e';
/** pink - hex #e377c2 */
Color.PINK = '#e377c2';
/** purple - hex #9467bd */
Color.PURPLE = '#9467bd';
/** red - hex #d62728 */
Color.RED = '#d62728';
/**
 * The position of the legend on a GraphWidget.
 */
var LegendPosition;
(function (LegendPosition) {
    /**
     * Legend appears below the graph (default).
     */
    LegendPosition["BOTTOM"] = "bottom";
    /**
     * Add shading above the annotation
     */
    LegendPosition["RIGHT"] = "right";
    /**
     * Add shading below the annotation
     */
    LegendPosition["HIDDEN"] = "hidden";
})(LegendPosition = exports.LegendPosition || (exports.LegendPosition = {}));
function mapAnnotation(yAxis) {
    return (a) => {
        return { ...a, yAxis };
    };
}
//# sourceMappingURL=data:application/json;base64,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