"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaEdgeEventType = exports.CachedMethods = exports.AllowedMethods = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.HttpVersion = exports.Distribution = void 0;
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        this.additionalBehaviors = [];
        this.boundOrigins = [];
        this.originGroups = [];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`Distribution certificates must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
            if (((_a = props.domainNames) !== null && _a !== void 0 ? _a : []).length === 0) {
                throw new Error('Must specify at least one domain name to use a certificate with a distribution');
            }
        }
        const originId = this.addOrigin(props.defaultBehavior.origin);
        this.defaultBehavior = new cache_behavior_1.CacheBehavior(originId, { pathPattern: '*', ...props.defaultBehavior });
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_b = props.errorResponses) !== null && _b !== void 0 ? _b : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'Resource', {
            distributionConfig: {
                enabled: (_c = props.enabled) !== null && _c !== void 0 ? _c : true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                originGroups: core_1.Lazy.anyValue({ produce: () => this.renderOriginGroups() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                aliases: props.domainNames,
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                comment: props.comment,
                customErrorResponses: this.renderErrorResponses(),
                defaultRootObject: props.defaultRootObject,
                httpVersion: (_d = props.httpVersion) !== null && _d !== void 0 ? _d : HttpVersion.HTTP2,
                ipv6Enabled: (_e = props.enableIpv6) !== null && _e !== void 0 ? _e : true,
                logging: this.renderLogging(props),
                priceClass: (_f = props.priceClass) !== null && _f !== void 0 ? _f : undefined,
                restrictions: this.renderRestrictions(props.geoRestriction),
                viewerCertificate: this.certificate ? this.renderViewerCertificate(this.certificate) : undefined,
                webAclId: props.webAclId,
            },
        });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param origin the origin to use for this behavior
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        const originId = this.addOrigin(origin);
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior(originId, { pathPattern, ...behaviorOptions }));
    }
    addOrigin(origin, isFailoverOrigin = false) {
        var _a;
        const existingOrigin = this.boundOrigins.find(boundOrigin => boundOrigin.origin === origin);
        if (existingOrigin) {
            return (_a = existingOrigin.originGroupId) !== null && _a !== void 0 ? _a : existingOrigin.originId;
        }
        else {
            const originIndex = this.boundOrigins.length + 1;
            const scope = new core_1.Construct(this, `Origin${originIndex}`);
            const originId = scope.node.uniqueId;
            const originBindConfig = origin.bind(scope, { originId });
            if (!originBindConfig.failoverConfig) {
                this.boundOrigins.push({ origin, originId, ...originBindConfig });
            }
            else {
                if (isFailoverOrigin) {
                    throw new Error('An Origin cannot use an Origin with its own failover configuration as its fallback origin!');
                }
                const groupIndex = this.originGroups.length + 1;
                const originGroupId = new core_1.Construct(this, `OriginGroup${groupIndex}`).node.uniqueId;
                this.boundOrigins.push({ origin, originId, originGroupId, ...originBindConfig });
                const failoverOriginId = this.addOrigin(originBindConfig.failoverConfig.failoverOrigin, true);
                this.addOriginGroup(originGroupId, originBindConfig.failoverConfig.statusCodes, originId, failoverOriginId);
                return originGroupId;
            }
            return originId;
        }
    }
    addOriginGroup(originGroupId, statusCodes, originId, failoverOriginId) {
        statusCodes = statusCodes !== null && statusCodes !== void 0 ? statusCodes : [500, 502, 503, 504];
        if (statusCodes.length === 0) {
            throw new Error('fallbackStatusCodes cannot be empty');
        }
        this.originGroups.push({
            failoverCriteria: {
                statusCodes: {
                    items: statusCodes,
                    quantity: statusCodes.length,
                },
            },
            id: originGroupId,
            members: {
                items: [
                    { originId },
                    { originId: failoverOriginId },
                ],
                quantity: 2,
            },
        });
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.boundOrigins.forEach(boundOrigin => {
            if (boundOrigin.originProperty) {
                renderedOrigins.push(boundOrigin.originProperty);
            }
        });
        return renderedOrigins;
    }
    renderOriginGroups() {
        return this.originGroups.length === 0
            ? undefined
            : {
                items: this.originGroups,
                quantity: this.originGroups.length,
            };
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
    renderLogging(props) {
        var _a;
        if (!props.enableLogging && !props.logBucket) {
            return undefined;
        }
        if (props.enableLogging === false && props.logBucket) {
            throw new Error('Explicitly disabled logging but provided a logging bucket.');
        }
        const bucket = (_a = props.logBucket) !== null && _a !== void 0 ? _a : new s3.Bucket(this, 'LoggingBucket');
        return {
            bucket: bucket.bucketRegionalDomainName,
            includeCookies: props.logIncludesCookies,
            prefix: props.logFilePrefix,
        };
    }
    renderRestrictions(geoRestriction) {
        return geoRestriction ? {
            geoRestriction: {
                restrictionType: geoRestriction.restrictionType,
                locations: geoRestriction.locations,
            },
        } : undefined;
    }
    renderViewerCertificate(certificate) {
        return {
            acmCertificateArn: certificate.certificateArn,
            sslSupportMethod: SSLMethod.SNI,
            minimumProtocolVersion: SecurityPolicyProtocol.TLS_V1_2_2019,
        };
    }
}
exports.Distribution = Distribution;
/** Maximum HTTP version to support */
var HttpVersion;
(function (HttpVersion) {
    /** HTTP 1.1 */
    HttpVersion["HTTP1_1"] = "http1.1";
    /** HTTP 2 */
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
    SecurityPolicyProtocol["TLS_V1_2_2019"] = "TLSv1.2_2019";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
/**
 * The HTTP methods that the Behavior will cache requests on.
 */
class CachedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.CachedMethods = CachedMethods;
/** HEAD and GET */
CachedMethods.CACHE_GET_HEAD = new CachedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
CachedMethods.CACHE_GET_HEAD_OPTIONS = new CachedMethods(['GET', 'HEAD', 'OPTIONS']);
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
//# sourceMappingURL=data:application/json;base64,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