"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackChannelConfiguration = exports.LoggingLevel = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const chatbot_generated_1 = require("./chatbot.generated");
/**
 * Logging levels include ERROR, INFO, or NONE.
 */
var LoggingLevel;
(function (LoggingLevel) {
    /**
     * ERROR
     */
    LoggingLevel["ERROR"] = "ERROR";
    /**
     * INFO
     */
    LoggingLevel["INFO"] = "INFO";
    /**
     * NONE
     */
    LoggingLevel["NONE"] = "NONE";
})(LoggingLevel = exports.LoggingLevel || (exports.LoggingLevel = {}));
/**
 * Either a new or imported Slack channel configuration
 */
class SlackChannelConfigurationBase extends cdk.Resource {
    /**
     * Adds extra permission to iam-role of Slack channel configuration
     * @param statement
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(statement);
    }
}
/**
 * A new Slack channel configuration
 */
class SlackChannelConfiguration extends SlackChannelConfigurationBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.slackChannelConfigurationName,
        });
        this.role = props.role || new iam.Role(this, 'ConfigurationRole', {
            assumedBy: new iam.ServicePrincipal('chatbot.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const configuration = new chatbot_generated_1.CfnSlackChannelConfiguration(this, 'Resource', {
            configurationName: props.slackChannelConfigurationName,
            iamRoleArn: this.role.roleArn,
            slackWorkspaceId: props.slackWorkspaceId,
            slackChannelId: props.slackChannelId,
            snsTopicArns: (_a = props.notificationTopics) === null || _a === void 0 ? void 0 : _a.map(topic => topic.topicArn),
            loggingLevel: (_b = props.loggingLevel) === null || _b === void 0 ? void 0 : _b.toString(),
        });
        this.slackChannelConfigurationArn = configuration.ref;
        this.slackChannelConfigurationName = props.slackChannelConfigurationName;
    }
    /**
     * Import an existing Slack channel configuration provided an ARN
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param slackChannelConfigurationArn configuration ARN (i.e. arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack)
     *
     * @returns a reference to the existing Slack channel configuration
     */
    static fromSlackChannelConfigurationArn(scope, id, slackChannelConfigurationArn) {
        const re = /^slack-channel\//;
        const resourceName = cdk.Stack.of(scope).parseArn(slackChannelConfigurationArn).resourceName;
        if (!re.test(resourceName)) {
            throw new Error('The ARN of a Slack integration must be in the form: arn:aws:chatbot:{region}:{account}:chat-configuration/slack-channel/{slackChannelName}');
        }
        class Import extends SlackChannelConfigurationBase {
            constructor(s, i) {
                super(s, i);
                /**
                 * @attribute
                 */
                this.slackChannelConfigurationArn = slackChannelConfigurationArn;
                this.role = undefined;
                /**
                 * Returns a name of Slack channel configuration
                 *
                 * NOTE:
                 * For example: arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack
                 * The ArnComponents API will return `slack-channel/my-slack`
                 * It need to handle that to gets a correct name.`my-slack`
                 */
                this.slackChannelConfigurationName = resourceName.substring('slack-channel/'.length);
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
}
exports.SlackChannelConfiguration = SlackChannelConfiguration;
//# sourceMappingURL=data:application/json;base64,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