"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobQueue = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const batch_generated_1 = require("./batch.generated");
/**
 * Batch Job Queue.
 *
 * Defines a batch job queue to define how submitted batch jobs
 * should be ran based on specified batch compute environments.
 */
class JobQueue extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobQueueName,
        });
        if (props.computeEnvironments.length === 0) {
            throw new Error('computeEnvironments must be non-empty');
        }
        const jobQueue = new batch_generated_1.CfnJobQueue(this, 'Resource', {
            computeEnvironmentOrder: props.computeEnvironments.map(cp => ({
                computeEnvironment: cp.computeEnvironment.computeEnvironmentArn,
                order: cp.order,
            })),
            jobQueueName: this.physicalName,
            priority: props.priority || 1,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
        });
        this.jobQueueArn = this.getResourceArnAttribute(jobQueue.ref, {
            service: 'batch',
            resource: 'job-queue',
            resourceName: this.physicalName,
        });
        this.jobQueueName = this.getResourceNameAttribute(jobQueue.ref);
    }
    /**
     * Fetches an existing batch job queue by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobQueueArn
     */
    static fromJobQueueArn(scope, id, jobQueueArn) {
        const stack = core_1.Stack.of(scope);
        const jobQueueName = stack.parseArn(jobQueueArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobQueueArn = jobQueueArn;
                this.jobQueueName = jobQueueName;
            }
        }
        return new Import(scope, id);
    }
}
exports.JobQueue = JobQueue;
//# sourceMappingURL=data:application/json;base64,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