"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const fs_1 = require("fs");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const private_1 = require("./private");
const schema_intermediate_1 = require("./schema-intermediate");
;
/**
 * The Schema for a GraphQL Api
 *
 * If no options are configured, schema will be generated
 * code-first.
 */
class Schema {
    constructor(options) {
        if (options === null || options === void 0 ? void 0 : options.filePath) {
            this.mode = private_1.SchemaMode.FILE;
            this.definition = fs_1.readFileSync(options.filePath).toString('utf-8');
        }
        else {
            this.mode = private_1.SchemaMode.CODE;
            this.definition = '';
        }
    }
    /**
     * Generate a Schema from file
     *
     * @returns `SchemaAsset` with immutable schema defintion
     * @param filePath the file path of the schema file
     */
    static fromAsset(filePath) {
        return new Schema({ filePath });
    }
    /**
     * Called when the GraphQL Api is initialized to allow this object to bind
     * to the stack.
     *
     * @param api The binding GraphQL Api
     */
    bind(api) {
        if (!this.schema) {
            this.schema = new appsync_generated_1.CfnGraphQLSchema(api, 'Schema', {
                apiId: api.apiId,
                definition: core_1.Lazy.stringValue({ produce: () => `${this.declareSchema()}${this.definition}` }),
            });
        }
        return this.schema;
    }
    /**
     * Escape hatch to add to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition, delimiter) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter !== null && delimiter !== void 0 ? delimiter : '';
        this.definition = `${this.definition}${sep}${addition}\n`;
    }
    /**
     * Add a query field to the schema's Query. If one isn't set by
     * the user, CDK will create an Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add query. Schema definition mode must be ${private_1.SchemaMode.CODE} Received: ${this.mode}`);
        }
        if (!this.query) {
            this.query = new schema_intermediate_1.ObjectType('Query', { definition: {} });
            this.addType(this.query);
        }
        ;
        this.query.addField({ fieldName, field });
        return this.query;
    }
    /**
     * Add a mutation field to the schema's Mutation. If one isn't set by
     * the user, CDK will create an Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add mutation. Schema definition mode must be ${private_1.SchemaMode.CODE} Received: ${this.mode}`);
        }
        if (!this.mutation) {
            this.mutation = new schema_intermediate_1.ObjectType('Mutation', { definition: {} });
            this.addType(this.mutation);
        }
        ;
        this.mutation.addField({ fieldName, field });
        return this.mutation;
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.addToSchema(core_1.Lazy.stringValue({ produce: () => type.toString() }));
        return type;
    }
    /**
     * Set the root types of this schema if they are defined.
     *
     * For example:
     * schema {
     *   query: Query
     *   mutation: Mutation
     *   subscription: Subscription
     * }
     */
    declareSchema() {
        if (!this.query && !this.mutation && !this.subscription) {
            return '';
        }
        const list = ['query', 'mutation', 'subscription'];
        return private_1.shapeAddition({
            prefix: 'schema',
            fields: list.map((key) => { var _a; return this[key] ? `${key}: ${(_a = this[key]) === null || _a === void 0 ? void 0 : _a.name}` : ''; })
                .filter((field) => field != ''),
        }) + '\n';
    }
}
exports.Schema = Schema;
//# sourceMappingURL=data:application/json;base64,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