import { CfnGraphQLSchema } from './appsync.generated';
import { GraphqlApi } from './graphqlapi';
import { IIntermediateType } from './schema-base';
import { ResolvableField } from './schema-field';
import { ObjectType } from './schema-intermediate';
/**
 * The options for configuring a schema
 *
 * If no options are specified, then the schema will
 * be generated code-first.
 */
export interface SchemaOptions {
    /**
     * The file path for the schema. When this option is
     * configured, then the schema will be generated from an
     * existing file from disk.
     *
     * @default - schema not configured through disk asset
     */
    readonly filePath?: string;
}
/**
 * The Schema for a GraphQL Api
 *
 * If no options are configured, schema will be generated
 * code-first.
 */
export declare class Schema {
    /**
     * Generate a Schema from file
     *
     * @returns `SchemaAsset` with immutable schema defintion
     * @param filePath the file path of the schema file
     */
    static fromAsset(filePath: string): Schema;
    /**
     * The definition for this schema
     */
    definition: string;
    private query?;
    private mutation?;
    private subscription?;
    private schema?;
    private mode;
    constructor(options?: SchemaOptions);
    /**
     * Called when the GraphQL Api is initialized to allow this object to bind
     * to the stack.
     *
     * @param api The binding GraphQL Api
     */
    bind(api: GraphqlApi): CfnGraphQLSchema;
    /**
     * Escape hatch to add to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition: string, delimiter?: string): void;
    /**
     * Add a query field to the schema's Query. If one isn't set by
     * the user, CDK will create an Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName: string, field: ResolvableField): ObjectType;
    /**
     * Add a mutation field to the schema's Mutation. If one isn't set by
     * the user, CDK will create an Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName: string, field: ResolvableField): ObjectType;
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type: IIntermediateType): IIntermediateType;
    /**
     * Set the root types of this schema if they are defined.
     *
     * For example:
     * schema {
     *   query: Query
     *   mutation: Mutation
     *   subscription: Subscription
     * }
     */
    private declareSchema;
}
