"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputType = exports.ObjectType = exports.InterfaceType = void 0;
const private_1 = require("./private");
const schema_field_1 = require("./schema-field");
/**
 * Interface Types are abstract types that includes a certain set of fields
 * that other types must include if they implement the interface.
 *
 * @experimental
 */
class InterfaceType {
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * Create an GraphQL Type representing this Intermediate Type
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'interface',
            name: this.name,
            fields: Object.keys(this.definition).map((key) => `${key}${this.definition[key].argsToString()}: ${this.definition[key].toString()}`),
        });
    }
    /**
     * Add a field to this Interface Type.
     *
     * Interface Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Interface Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
}
exports.InterfaceType = InterfaceType;
/**
 * Object Types are types declared by you.
 *
 * @experimental
 */
class ObjectType extends InterfaceType {
    constructor(name, props) {
        var _a, _b;
        const options = {
            definition: (_b = (_a = props.interfaceTypes) === null || _a === void 0 ? void 0 : _a.reduce((def, interfaceType) => {
                return Object.assign({}, def, interfaceType.definition);
            }, props.definition)) !== null && _b !== void 0 ? _b : props.definition,
        };
        super(name, options);
        this.interfaceTypes = props.interfaceTypes;
        this.directives = props.directives;
        this.resolvers = [];
        Object.keys(this.definition).forEach((fieldName) => {
            const field = this.definition[fieldName];
            this.generateResolver(fieldName, field.fieldOptions);
        });
    }
    /**
     * Add a field to this Object Type.
     *
     * Object Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Object Types must have both fieldName and field options.');
        }
        this.generateResolver(options.fieldName, options.field.fieldOptions);
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'type',
            name: this.name,
            interfaceTypes: this.interfaceTypes,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => `${key}${this.definition[key].argsToString()}: ${this.definition[key].toString()}`),
        });
    }
    /**
     * Generate the resolvers linked to this Object Type
     */
    generateResolver(fieldName, options) {
        if (options === null || options === void 0 ? void 0 : options.dataSource) {
            if (!this.resolvers) {
                this.resolvers = [];
            }
            this.resolvers.push(options.dataSource.createResolver({
                typeName: this.name,
                fieldName: fieldName,
                pipelineConfig: options.pipelineConfig,
                requestMappingTemplate: options.requestMappingTemplate,
                responseMappingTemplate: options.responseMappingTemplate,
            }));
        }
    }
}
exports.ObjectType = ObjectType;
/**
 * Input Types are abstract types that define complex objects.
 * They are used in arguments to represent
 *
 * @experimental
 */
class InputType {
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * Create an GraphQL Type representing this Input Type
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this input type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'input',
            name: this.name,
            fields: Object.keys(this.definition).map((key) => `${key}${this.definition[key].argsToString()}: ${this.definition[key].toString()}`),
        });
    }
    /**
     * Add a field to this Input Type.
     *
     * Input Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Input Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
}
exports.InputType = InputType;
//# sourceMappingURL=data:application/json;base64,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