"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Between = exports.BinaryCondition = exports.BeginsWith = exports.BaseKeyCondition = exports.shapeAddition = exports.SchemaMode = void 0;
/**
 * Utility enum for Schema class
 */
var SchemaMode;
(function (SchemaMode) {
    SchemaMode["FILE"] = "FILE";
    SchemaMode["CODE"] = "CODE";
})(SchemaMode = exports.SchemaMode || (exports.SchemaMode = {}));
;
/**
 * Generates an addition to the schema
 *
 * @param options the options to produced a stringfied addition
 *
 * @returns the following shape:
 *
 * ```
 * prefix name interfaces directives {
 *   field
 *   field
 *   ...
 * }
 * ```
 */
function shapeAddition(options) {
    const typeName = () => { return options.name ? ` ${options.name}` : ''; };
    const interfaces = generateInterfaces(options.interfaceTypes);
    const directives = generateDirectives(options.directives);
    return options.fields.reduce((acc, field) => `${acc}  ${field}\n`, `${options.prefix}${typeName()}${interfaces}${directives} {\n`) + '}';
}
exports.shapeAddition = shapeAddition;
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
exports.BaseKeyCondition = BaseKeyCondition;
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BeginsWith = BeginsWith;
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BinaryCondition = BinaryCondition;
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
exports.Between = Between;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility function to generate interfaces for object types
 *
 * @param interfaceTypes the interfaces this object type implements
 */
function generateInterfaces(interfaceTypes) {
    if (!interfaceTypes || interfaceTypes.length === 0)
        return '';
    return interfaceTypes.reduce((acc, interfaceType) => `${acc} ${interfaceType.name},`, ' implements').slice(0, -1);
}
/**
 * Utility function to generate directives
 *
 * @param directives the directives of a given type
 * @param delimiter the separator betweeen directives (by default we will add a space)
 */
function generateDirectives(directives, delimiter) {
    if (!directives || directives.length === 0)
        return '';
    return directives.reduce((acc, directive) => `${acc}${directive.statement}${delimiter !== null && delimiter !== void 0 ? delimiter : ' '}`, ' ').slice(0, -1);
}
//# sourceMappingURL=data:application/json;base64,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