"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            sep: '/',
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphqlApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        const defaultMode = (_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) !== null && _b !== void 0 ? _b : { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = (_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.additionalAuthorizationModes) !== null && _d !== void 0 ? _d : [];
        const modes = [defaultMode, ...additionalModes];
        this.validateAuthorizationProps(modes);
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphqlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = (_e = props.schema) !== null && _e !== void 0 ? _e : new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = (_f = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })) === null || _f === void 0 ? void 0 : _f.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        var _a;
        const arn = (_a = attrs.graphqlApiArn) !== null && _a !== void 0 ? _a : core_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        if (!config)
            return undefined;
        const role = new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        });
        return {
            cloudWatchLogsRoleArn: role.roleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: config.fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || UserPoolDefaultAction.ALLOW,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        let expires;
        if (config === null || config === void 0 ? void 0 : config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        return new appsync_generated_1.CfnApiKey(this, `${(config === null || config === void 0 ? void 0 : config.name) || 'Default'}ApiKey`, {
            expires,
            description: config === null || config === void 0 ? void 0 : config.description,
            apiId: this.apiId,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type) {
        return this.schema.addType(type);
    }
    /**
     * Add a query field to the schema's Query. If one isn't set by
     * the user, CDK will create an Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        return this.schema.addQuery(fieldName, field);
    }
    /**
     * Add a mutation field to the schema's Mutation. If one isn't set by
     * the user, CDK will create an Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        return this.schema.addMutation(fieldName, field);
    }
}
exports.GraphqlApi = GraphqlApi;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ3JhcGhxbGFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImdyYXBocWxhcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsMkNBQXlGLENBQUMsbURBQW1EO0FBQzdJLHFDQUFrRixDQUFDLGdEQUFnRDtBQUNuSSwyREFBaUY7QUFDakYsdURBQWdFO0FBQ2hFLHFDQUFrQztBQUlsQzs7R0FFRztBQUNILElBQVksaUJBaUJYO0FBakJELFdBQVksaUJBQWlCO0lBQ3pCOztPQUVHO0lBQ0gsd0NBQW1CLENBQUE7SUFDbkI7O09BRUc7SUFDSCxvQ0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCw0REFBdUMsQ0FBQTtJQUN2Qzs7T0FFRztJQUNILDRDQUF1QixDQUFBO0FBQzNCLENBQUMsRUFqQlcsaUJBQWlCLEdBQWpCLHlCQUFpQixLQUFqQix5QkFBaUIsUUFpQjVCO0FBNkJEOztHQUVHO0FBQ0gsSUFBWSxxQkFTWDtBQVRELFdBQVkscUJBQXFCO0lBQzdCOztPQUVHO0lBQ0gsd0NBQWUsQ0FBQTtJQUNmOztPQUVHO0lBQ0gsc0NBQWEsQ0FBQTtBQUNqQixDQUFDLEVBVFcscUJBQXFCLEdBQXJCLDZCQUFxQixLQUFyQiw2QkFBcUIsUUFTaEM7QUF5RkQ7O0dBRUc7QUFDSCxJQUFZLGFBYVg7QUFiRCxXQUFZLGFBQWE7SUFDckI7O09BRUc7SUFDSCw4QkFBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCxnQ0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCw0QkFBVyxDQUFBO0FBQ2YsQ0FBQyxFQWJXLGFBQWEsR0FBYixxQkFBYSxLQUFiLHFCQUFhLFFBYXhCO0FBdUREOztHQUVHO0FBQ0gsTUFBYSxXQUFXO0lBaUNwQixZQUFvQixJQUFjO1FBQzlCLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDO0lBQ3JCLENBQUM7SUFsQ0Q7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLElBQWM7UUFDbEMsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNuQixNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF5QyxDQUFDLENBQUM7U0FDOUQ7UUFDRCxPQUFPLElBQUksV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFDRDs7Ozs7OztPQU9HO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFZLEVBQUUsR0FBRyxNQUFnQjtRQUNsRCxNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxTQUFTLElBQUksV0FBVyxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsQ0FBQztRQUMxRyxPQUFPLElBQUksV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFDRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxHQUFHO1FBQ2IsT0FBTyxJQUFJLFdBQVcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUtEOzs7O09BSUc7SUFDSSxZQUFZLENBQUMsR0FBZTtRQUMvQixPQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxZQUFLLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsU0FBUztZQUNsQixRQUFRLEVBQUUsUUFBUSxHQUFHLENBQUMsS0FBSyxFQUFFO1lBQzdCLEdBQUcsRUFBRSxHQUFHO1lBQ1IsWUFBWSxFQUFFLEdBQUcsR0FBRyxFQUFFO1NBQ3pCLENBQUMsQ0FBQyxDQUFDO0lBQ1IsQ0FBQztDQUNKO0FBakRELGtDQWlEQztBQWdCRDs7OztHQUlHO0FBQ0gsTUFBYSxVQUFXLFNBQVEsZ0NBQWM7SUFzRDFDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0I7O1FBQzVELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsTUFBTSxXQUFXLGVBQUcsS0FBSyxDQUFDLG1CQUFtQiwwQ0FBRSxvQkFBb0IsbUNBQy9ELEVBQUUsaUJBQWlCLEVBQUUsaUJBQWlCLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDckQsTUFBTSxlQUFlLGVBQUcsS0FBSyxDQUFDLG1CQUFtQiwwQ0FBRSw0QkFBNEIsbUNBQUksRUFBRSxDQUFDO1FBQ3RGLE1BQU0sS0FBSyxHQUFHLENBQUMsV0FBVyxFQUFFLEdBQUcsZUFBZSxDQUFDLENBQUM7UUFDaEQsSUFBSSxDQUFDLDBCQUEwQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxpQ0FBYSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDM0MsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLGtCQUFrQixFQUFFLFdBQVcsQ0FBQyxpQkFBaUI7WUFDakQsU0FBUyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUMvQyxtQkFBbUIsRUFBRSxJQUFJLENBQUMsd0JBQXdCLENBQUMsV0FBVyxDQUFDLG1CQUFtQixDQUFDO1lBQ25GLGNBQWMsRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQztZQUNwRSxpQ0FBaUMsRUFBRSxJQUFJLENBQUMsaUNBQWlDLENBQUMsZUFBZSxDQUFDO1lBQzFGLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztTQUNqQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDO1FBQ2hDLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUM7UUFDNUIsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQztRQUMxQyxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDO1FBQzFCLElBQUksQ0FBQyxNQUFNLFNBQUcsS0FBSyxDQUFDLE1BQU0sbUNBQUksSUFBSSxlQUFNLEVBQUUsQ0FBQztRQUMzQyxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdDLElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQzVFLE1BQU0sTUFBTSxTQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUF1QixFQUFFLEVBQUU7Z0JBQ2xELE9BQU8sSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDO1lBQ3JGLENBQUMsQ0FBQywwQ0FBRSxZQUFZLENBQUM7WUFDakIsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ2hELElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUN0RCxJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsVUFBVSxDQUFDO1NBQ2hEO0lBQ0wsQ0FBQztJQW5GRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsd0JBQXdCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMkI7O1FBQzVGLE1BQU0sR0FBRyxTQUFHLEtBQUssQ0FBQyxhQUFhLG1DQUFJLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ3pELE9BQU8sRUFBRSxTQUFTO1lBQ2xCLFFBQVEsRUFBRSxRQUFRLEtBQUssQ0FBQyxZQUFZLEVBQUU7U0FDekMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxNQUFPLFNBQVEsZ0NBQWM7WUFHL0IsWUFBWSxDQUFZLEVBQUUsQ0FBUztnQkFDL0IsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFIQSxVQUFLLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztnQkFDM0IsUUFBRyxHQUFHLEdBQUcsQ0FBQztZQUcxQixDQUFDO1NBQ0o7UUFDRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBZ0VEOzs7Ozs7O09BT0c7SUFDSSxLQUFLLENBQUMsT0FBbUIsRUFBRSxTQUFzQixFQUFFLEdBQUcsT0FBaUI7UUFDMUUsT0FBTyxlQUFLLENBQUMsY0FBYyxDQUFDO1lBQ3hCLE9BQU87WUFDUCxPQUFPO1lBQ1AsWUFBWSxFQUFFLFNBQVMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDO1lBQzFDLEtBQUssRUFBRSxJQUFJO1NBQ2QsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLGFBQWEsQ0FBQyxPQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDekQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFDN0YsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLFVBQVUsQ0FBQyxPQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDdEQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFDMUYsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQixDQUFDLE9BQW1CLEVBQUUsR0FBRyxNQUFnQjtRQUM3RCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLFdBQVcsQ0FBQyxNQUFNLENBQUMsY0FBYyxFQUFFLEdBQUcsTUFBTSxDQUFDLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztJQUNqRyxDQUFDO0lBQ08sMEJBQTBCLENBQUMsS0FBMEI7UUFDekQsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFO1lBQ2YsSUFBSSxJQUFJLENBQUMsaUJBQWlCLEtBQUssaUJBQWlCLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLG1CQUFtQixFQUFFO2dCQUNoRixNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7YUFDekQ7WUFDRCxJQUFJLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxTQUFTLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFO2dCQUNoRixNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7YUFDekQ7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUNILElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDekYsTUFBTSxJQUFJLEtBQUssQ0FBQyxtSEFBbUgsQ0FBQyxDQUFDO1NBQ3hJO1FBQ0QsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLEtBQUssaUJBQWlCLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNyRixNQUFNLElBQUksS0FBSyxDQUFDLCtHQUErRyxDQUFDLENBQUM7U0FDcEk7SUFDTCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLG1CQUFtQixDQUFDLFNBQXNCO1FBQzdDLFNBQVMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2hCLENBQUM7SUFDTyxjQUFjLENBQUMsTUFBa0I7UUFDckMsSUFBSSxDQUFDLE1BQU07WUFDUCxPQUFPLFNBQVMsQ0FBQztRQUNyQixNQUFNLElBQUksR0FBRyxJQUFJLGNBQUksQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQ3ZDLFNBQVMsRUFBRSxJQUFJLDBCQUFnQixDQUFDLHVCQUF1QixDQUFDO1lBQ3hELGVBQWUsRUFBRTtnQkFDYix1QkFBYSxDQUFDLHdCQUF3QixDQUFDLDZDQUE2QyxDQUFDO2FBQ3hGO1NBQ0osQ0FBQyxDQUFDO1FBQ0gsT0FBTztZQUNILHFCQUFxQixFQUFFLElBQUksQ0FBQyxPQUFPO1lBQ25DLHFCQUFxQixFQUFFLE1BQU0sQ0FBQyxxQkFBcUI7WUFDbkQsYUFBYSxFQUFFLE1BQU0sQ0FBQyxhQUFhO1NBQ3RDLENBQUM7SUFDTixDQUFDO0lBQ08sd0JBQXdCLENBQUMsTUFBNEI7UUFDekQsSUFBSSxDQUFDLE1BQU07WUFDUCxPQUFPLFNBQVMsQ0FBQztRQUNyQixPQUFPO1lBQ0gsT0FBTyxFQUFFLE1BQU0sQ0FBQyxtQkFBbUI7WUFDbkMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRO1lBQ3pCLE1BQU0sRUFBRSxNQUFNLENBQUMsb0JBQW9CO1lBQ25DLE1BQU0sRUFBRSxNQUFNLENBQUMsWUFBWTtTQUM5QixDQUFDO0lBQ04sQ0FBQztJQUNPLG1CQUFtQixDQUFDLE1BQXVCO1FBQy9DLElBQUksQ0FBQyxNQUFNO1lBQ1AsT0FBTyxTQUFTLENBQUM7UUFDckIsT0FBTztZQUNILFVBQVUsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLFVBQVU7WUFDdEMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLE1BQU07WUFDdkMsZ0JBQWdCLEVBQUUsTUFBTSxDQUFDLGdCQUFnQjtZQUN6QyxhQUFhLEVBQUUsTUFBTSxDQUFDLGFBQWEsSUFBSSxxQkFBcUIsQ0FBQyxLQUFLO1NBQ3JFLENBQUM7SUFDTixDQUFDO0lBQ08saUNBQWlDLENBQUMsS0FBMkI7UUFDakUsSUFBSSxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLENBQUM7WUFDNUIsT0FBTyxTQUFTLENBQUM7UUFDckIsT0FBTyxLQUFLLENBQUMsTUFBTSxDQUEyRCxDQUFDLEdBQUcsRUFBRSxJQUFJLEVBQUUsRUFBRSxDQUFDO1lBQ3pGLEdBQUcsR0FBRyxFQUFFO2dCQUNKLGtCQUFrQixFQUFFLElBQUksQ0FBQyxpQkFBaUI7Z0JBQzFDLGNBQWMsRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQztnQkFDN0QsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQzthQUMvRTtTQUNKLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDWCxDQUFDO0lBQ08sWUFBWSxDQUFDLE1BQXFCO1FBQ3RDLElBQUksT0FBMkIsQ0FBQztRQUNoQyxJQUFJLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxPQUFPLEVBQUU7WUFDakIsT0FBTyxHQUFHLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUM3QyxNQUFNLElBQUksR0FBRyxDQUFDLENBQVMsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLGVBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxFQUFFLENBQUM7WUFDM0UsSUFBSSxPQUFPLEdBQUcsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLE9BQU8sR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQzFDLE1BQU0sS0FBSyxDQUFDLG9EQUFvRCxDQUFDLENBQUM7YUFDckU7WUFDRCxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDLENBQUM7U0FDeEM7UUFDRCxPQUFPLElBQUksNkJBQVMsQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFBLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxJQUFJLEtBQUksU0FBUyxRQUFRLEVBQUU7WUFDN0QsT0FBTztZQUNQLFdBQVcsRUFBRSxNQUFNLGFBQU4sTUFBTSx1QkFBTixNQUFNLENBQUUsV0FBVztZQUNoQyxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7U0FDcEIsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7Ozs7Ozs7T0FTRztJQUNJLFdBQVcsQ0FBQyxRQUFnQixFQUFFLFNBQWtCO1FBQ25ELElBQUksQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLFFBQVEsRUFBRSxTQUFTLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0ksT0FBTyxDQUFDLElBQXVCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUNEOzs7Ozs7Ozs7O09BVUc7SUFDSSxRQUFRLENBQUMsU0FBaUIsRUFBRSxLQUFzQjtRQUNyRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7T0FVRztJQUNJLFdBQVcsQ0FBQyxTQUFpQixFQUFFLEtBQXNCO1FBQ3hELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3JELENBQUM7Q0FDSjtBQTdRRCxnQ0E2UUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJVXNlclBvb2wgfSBmcm9tIFwiLi4vLi4vYXdzLWNvZ25pdG9cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2duaXRvJ1xuaW1wb3J0IHsgTWFuYWdlZFBvbGljeSwgUm9sZSwgU2VydmljZVByaW5jaXBhbCwgR3JhbnQsIElHcmFudGFibGUgfSBmcm9tIFwiLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCB7IENmblJlc291cmNlLCBDb25zdHJ1Y3QsIER1cmF0aW9uLCBJUmVzb2x2YWJsZSwgU3RhY2sgfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IENmbkFwaUtleSwgQ2ZuR3JhcGhRTEFwaSwgQ2ZuR3JhcGhRTFNjaGVtYSB9IGZyb20gJy4vYXBwc3luYy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUdyYXBocWxBcGksIEdyYXBocWxBcGlCYXNlIH0gZnJvbSAnLi9ncmFwaHFsYXBpLWJhc2UnO1xuaW1wb3J0IHsgU2NoZW1hIH0gZnJvbSAnLi9zY2hlbWEnO1xuaW1wb3J0IHsgSUludGVybWVkaWF0ZVR5cGUgfSBmcm9tICcuL3NjaGVtYS1iYXNlJztcbmltcG9ydCB7IFJlc29sdmFibGVGaWVsZCB9IGZyb20gJy4vc2NoZW1hLWZpZWxkJztcbmltcG9ydCB7IE9iamVjdFR5cGUgfSBmcm9tICcuL3NjaGVtYS1pbnRlcm1lZGlhdGUnO1xuLyoqXG4gKiBlbnVtIHdpdGggYWxsIHBvc3NpYmxlIHZhbHVlcyBmb3IgQXBwU3luYyBhdXRob3JpemF0aW9uIHR5cGVcbiAqL1xuZXhwb3J0IGVudW0gQXV0aG9yaXphdGlvblR5cGUge1xuICAgIC8qKlxuICAgICAqIEFQSSBLZXkgYXV0aG9yaXphdGlvbiB0eXBlXG4gICAgICovXG4gICAgQVBJX0tFWSA9ICdBUElfS0VZJyxcbiAgICAvKipcbiAgICAgKiBBV1MgSUFNIGF1dGhvcml6YXRpb24gdHlwZS4gQ2FuIGJlIHVzZWQgd2l0aCBDb2duaXRvIElkZW50aXR5IFBvb2wgZmVkZXJhdGVkIGNyZWRlbnRpYWxzXG4gICAgICovXG4gICAgSUFNID0gJ0FXU19JQU0nLFxuICAgIC8qKlxuICAgICAqIENvZ25pdG8gVXNlciBQb29sIGF1dGhvcml6YXRpb24gdHlwZVxuICAgICAqL1xuICAgIFVTRVJfUE9PTCA9ICdBTUFaT05fQ09HTklUT19VU0VSX1BPT0xTJyxcbiAgICAvKipcbiAgICAgKiBPcGVuSUQgQ29ubmVjdCBhdXRob3JpemF0aW9uIHR5cGVcbiAgICAgKi9cbiAgICBPSURDID0gJ09QRU5JRF9DT05ORUNUJ1xufVxuLyoqXG4gKiBJbnRlcmZhY2UgdG8gc3BlY2lmeSBkZWZhdWx0IG9yIGFkZGl0aW9uYWwgYXV0aG9yaXphdGlvbihzKVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF1dGhvcml6YXRpb25Nb2RlIHtcbiAgICAvKipcbiAgICAgKiBPbmUgb2YgcG9zc2libGUgZm91ciB2YWx1ZXMgQXBwU3luYyBzdXBwb3J0c1xuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvc2VjdXJpdHkuaHRtbFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBgQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWWBcbiAgICAgKi9cbiAgICByZWFkb25seSBhdXRob3JpemF0aW9uVHlwZTogQXV0aG9yaXphdGlvblR5cGU7XG4gICAgLyoqXG4gICAgICogSWYgYXV0aG9yaXphdGlvblR5cGUgaXMgYEF1dGhvcml6YXRpb25UeXBlLlVTRVJfUE9PTGAsIHRoaXMgb3B0aW9uIGlzIHJlcXVpcmVkLlxuICAgICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHVzZXJQb29sQ29uZmlnPzogVXNlclBvb2xDb25maWc7XG4gICAgLyoqXG4gICAgICogSWYgYXV0aG9yaXphdGlvblR5cGUgaXMgYEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVlgLCB0aGlzIG9wdGlvbiBjYW4gYmUgY29uZmlndXJlZC5cbiAgICAgKiBAZGVmYXVsdCAtIG5hbWU6ICdEZWZhdWx0QVBJS2V5JyB8IGRlc2NyaXB0aW9uOiAnRGVmYXVsdCBBUEkgS2V5IGNyZWF0ZWQgYnkgQ0RLJ1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGFwaUtleUNvbmZpZz86IEFwaUtleUNvbmZpZztcbiAgICAvKipcbiAgICAgKiBJZiBhdXRob3JpemF0aW9uVHlwZSBpcyBgQXV0aG9yaXphdGlvblR5cGUuT0lEQ2AsIHRoaXMgb3B0aW9uIGlzIHJlcXVpcmVkLlxuICAgICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IG9wZW5JZENvbm5lY3RDb25maWc/OiBPcGVuSWRDb25uZWN0Q29uZmlnO1xufVxuLyoqXG4gKiBlbnVtIHdpdGggYWxsIHBvc3NpYmxlIHZhbHVlcyBmb3IgQ29nbml0byB1c2VyLXBvb2wgZGVmYXVsdCBhY3Rpb25zXG4gKi9cbmV4cG9ydCBlbnVtIFVzZXJQb29sRGVmYXVsdEFjdGlvbiB7XG4gICAgLyoqXG4gICAgICogQUxMT1cgYWNjZXNzIHRvIEFQSVxuICAgICAqL1xuICAgIEFMTE9XID0gJ0FMTE9XJyxcbiAgICAvKipcbiAgICAgKiBERU5ZIGFjY2VzcyB0byBBUElcbiAgICAgKi9cbiAgICBERU5ZID0gJ0RFTlknXG59XG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIENvZ25pdG8gdXNlci1wb29scyBpbiBBcHBTeW5jXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVXNlclBvb2xDb25maWcge1xuICAgIC8qKlxuICAgICAqIFRoZSBDb2duaXRvIHVzZXIgcG9vbCB0byB1c2UgYXMgaWRlbnRpdHkgc291cmNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgdXNlclBvb2w6IElVc2VyUG9vbDtcbiAgICAvKipcbiAgICAgKiB0aGUgb3B0aW9uYWwgYXBwIGlkIHJlZ2V4XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtICBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXBwSWRDbGllbnRSZWdleD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBEZWZhdWx0IGF1dGggYWN0aW9uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBBTExPV1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlZmF1bHRBY3Rpb24/OiBVc2VyUG9vbERlZmF1bHRBY3Rpb247XG59XG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIEFQSSBLZXkgYXV0aG9yaXphdGlvbiBpbiBBcHBTeW5jXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBpS2V5Q29uZmlnIHtcbiAgICAvKipcbiAgICAgKiBVbmlxdWUgbmFtZSBvZiB0aGUgQVBJIEtleVxuICAgICAqIEBkZWZhdWx0IC0gJ0RlZmF1bHRBUElLZXknXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBEZXNjcmlwdGlvbiBvZiBBUEkga2V5XG4gICAgICogQGRlZmF1bHQgLSAnRGVmYXVsdCBBUEkgS2V5IGNyZWF0ZWQgYnkgQ0RLJ1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSB0aW1lIGZyb20gY3JlYXRpb24gdGltZSBhZnRlciB3aGljaCB0aGUgQVBJIGtleSBleHBpcmVzLCB1c2luZyBSRkMzMzM5IHJlcHJlc2VudGF0aW9uLlxuICAgICAqIEl0IG11c3QgYmUgYSBtaW5pbXVtIG9mIDEgZGF5IGFuZCBhIG1heGltdW0gb2YgMzY1IGRheXMgZnJvbSBkYXRlIG9mIGNyZWF0aW9uLlxuICAgICAqIFJvdW5kZWQgZG93biB0byB0aGUgbmVhcmVzdCBob3VyLlxuICAgICAqIEBkZWZhdWx0IC0gNyBkYXlzIGZyb20gY3JlYXRpb24gdGltZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGV4cGlyZXM/OiBzdHJpbmc7XG59XG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIE9wZW5JRCBDb25uZWN0IGF1dGhvcml6YXRpb24gaW4gQXBwU3luY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIE9wZW5JZENvbm5lY3RDb25maWcge1xuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2YgbWlsbGlzZWNvbmRzIGFuIE9JREMgdG9rZW4gaXMgdmFsaWQgYWZ0ZXIgYmVpbmcgYXV0aGVudGljYXRlZCBieSBPSURDIHByb3ZpZGVyLlxuICAgICAqIGBhdXRoX3RpbWVgIGNsYWltIGluIE9JREMgdG9rZW4gaXMgcmVxdWlyZWQgZm9yIHRoaXMgdmFsaWRhdGlvbiB0byB3b3JrLlxuICAgICAqIEBkZWZhdWx0IC0gbm8gdmFsaWRhdGlvblxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRva2VuRXhwaXJ5RnJvbUF1dGg/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIG51bWJlciBvZiBtaWxsaXNlY29uZHMgYW4gT0lEQyB0b2tlbiBpcyB2YWxpZCBhZnRlciBiZWluZyBpc3N1ZWQgdG8gYSB1c2VyLlxuICAgICAqIFRoaXMgdmFsaWRhdGlvbiB1c2VzIGBpYXRgIGNsYWltIG9mIE9JREMgdG9rZW4uXG4gICAgICogQGRlZmF1bHQgLSBubyB2YWxpZGF0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgdG9rZW5FeHBpcnlGcm9tSXNzdWU/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIGNsaWVudCBpZGVudGlmaWVyIG9mIHRoZSBSZWx5aW5nIHBhcnR5IGF0IHRoZSBPcGVuSUQgaWRlbnRpdHkgcHJvdmlkZXIuXG4gICAgICogQSByZWd1bGFyIGV4cHJlc3Npb24gY2FuIGJlIHNwZWNpZmllZCBzbyBBcHBTeW5jIGNhbiB2YWxpZGF0ZSBhZ2FpbnN0IG11bHRpcGxlIGNsaWVudCBpZGVudGlmaWVycyBhdCBhIHRpbWUuXG4gICAgICogQGV4YW1wbGUgLSAnQUJDRHxDREVGJyB3aGVyZSBBQkNEIGFuZCBDREVGIGFyZSB0d28gZGlmZmVyZW50IGNsaWVudElkXG4gICAgICogQGRlZmF1bHQgLSAqIChBbGwpXG4gICAgICovXG4gICAgcmVhZG9ubHkgY2xpZW50SWQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGlzc3VlciBmb3IgdGhlIE9JREMgY29uZmlndXJhdGlvbi4gVGhlIGlzc3VlciByZXR1cm5lZCBieSBkaXNjb3ZlcnkgbXVzdCBleGFjdGx5IG1hdGNoIHRoZSB2YWx1ZSBvZiBgaXNzYCBpbiB0aGUgT0lEQyB0b2tlbi5cbiAgICAgKi9cbiAgICByZWFkb25seSBvaWRjUHJvdmlkZXI6IHN0cmluZztcbn1cbi8qKlxuICogQ29uZmlndXJhdGlvbiBvZiB0aGUgQVBJIGF1dGhvcml6YXRpb24gbW9kZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXV0aG9yaXphdGlvbkNvbmZpZyB7XG4gICAgLyoqXG4gICAgICogT3B0aW9uYWwgYXV0aG9yaXphdGlvbiBjb25maWd1cmF0aW9uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEFQSSBLZXkgYXV0aG9yaXphdGlvblxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlZmF1bHRBdXRob3JpemF0aW9uPzogQXV0aG9yaXphdGlvbk1vZGU7XG4gICAgLyoqXG4gICAgICogQWRkaXRpb25hbCBhdXRob3JpemF0aW9uIG1vZGVzXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIG90aGVyIG1vZGVzXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2Rlcz86IEF1dGhvcml6YXRpb25Nb2RlW107XG59XG4vKipcbiAqIGxvZy1sZXZlbCBmb3IgZmllbGRzIGluIEFwcFN5bmNcbiAqL1xuZXhwb3J0IGVudW0gRmllbGRMb2dMZXZlbCB7XG4gICAgLyoqXG4gICAgICogTm8gbG9nZ2luZ1xuICAgICAqL1xuICAgIE5PTkUgPSAnTk9ORScsXG4gICAgLyoqXG4gICAgICogRXJyb3IgbG9nZ2luZ1xuICAgICAqL1xuICAgIEVSUk9SID0gJ0VSUk9SJyxcbiAgICAvKipcbiAgICAgKiBBbGwgbG9nZ2luZ1xuICAgICAqL1xuICAgIEFMTCA9ICdBTEwnXG59XG4vKipcbiAqIExvZ2dpbmcgY29uZmlndXJhdGlvbiBmb3IgQXBwU3luY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIExvZ0NvbmZpZyB7XG4gICAgLyoqXG4gICAgICogZXhjbHVkZSB2ZXJib3NlIGNvbnRlbnRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgZXhjbHVkZVZlcmJvc2VDb250ZW50PzogYm9vbGVhbiB8IElSZXNvbHZhYmxlO1xuICAgIC8qKlxuICAgICAqIGxvZyBsZXZlbCBmb3IgZmllbGRzXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFVzZSBBcHBTeW5jIGRlZmF1bHRcbiAgICAgKi9cbiAgICByZWFkb25seSBmaWVsZExvZ0xldmVsPzogRmllbGRMb2dMZXZlbDtcbn1cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYW4gQXBwU3luYyBHcmFwaFFMIEFQSVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEdyYXBocWxBcGlQcm9wcyB7XG4gICAgLyoqXG4gICAgICogdGhlIG5hbWUgb2YgdGhlIEdyYXBoUUwgQVBJXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIE9wdGlvbmFsIGF1dGhvcml6YXRpb24gY29uZmlndXJhdGlvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBUEkgS2V5IGF1dGhvcml6YXRpb25cbiAgICAgKi9cbiAgICByZWFkb25seSBhdXRob3JpemF0aW9uQ29uZmlnPzogQXV0aG9yaXphdGlvbkNvbmZpZztcbiAgICAvKipcbiAgICAgKiBMb2dnaW5nIGNvbmZpZ3VyYXRpb24gZm9yIHRoaXMgYXBpXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBsb2dDb25maWc/OiBMb2dDb25maWc7XG4gICAgLyoqXG4gICAgICogR3JhcGhRTCBzY2hlbWEgZGVmaW5pdGlvbi4gU3BlY2lmeSBob3cgeW91IHdhbnQgdG8gZGVmaW5lIHlvdXIgc2NoZW1hLlxuICAgICAqXG4gICAgICogU2NoZW1hLmZyb21GaWxlKGZpbGVQYXRoOiBzdHJpbmcpIGFsbG93cyBzY2hlbWEgZGVmaW5pdGlvbiB0aHJvdWdoIHNjaGVtYS5ncmFwaHFsIGZpbGVcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gc2NoZW1hIHdpbGwgYmUgZ2VuZXJhdGVkIGNvZGUtZmlyc3QgKGkuZS4gYWRkVHlwZSwgYWRkT2JqZWN0VHlwZSwgZXRjLilcbiAgICAgKlxuICAgICAqIEBleHBlcmltZW50YWxcbiAgICAgKi9cbiAgICByZWFkb25seSBzY2hlbWE/OiBTY2hlbWE7XG4gICAgLyoqXG4gICAgICogQSBmbGFnIGluZGljYXRpbmcgd2hldGhlciBvciBub3QgWC1SYXkgdHJhY2luZyBpcyBlbmFibGVkIGZvciB0aGUgR3JhcGhRTCBBUEkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgeHJheUVuYWJsZWQ/OiBib29sZWFuO1xufVxuLyoqXG4gKiBBIGNsYXNzIHVzZWQgdG8gZ2VuZXJhdGUgcmVzb3VyY2UgYXJucyBmb3IgQXBwU3luY1xuICovXG5leHBvcnQgY2xhc3MgSWFtUmVzb3VyY2Uge1xuICAgIC8qKlxuICAgICAqIEdlbmVyYXRlIHRoZSByZXNvdXJjZSBuYW1lcyBnaXZlbiBjdXN0b20gYXJuc1xuICAgICAqXG4gICAgICogQHBhcmFtIGFybnMgVGhlIGN1c3RvbSBhcm5zIHRoYXQgbmVlZCB0byBiZSBwZXJtaXNzaW9uZWRcbiAgICAgKlxuICAgICAqIEV4YW1wbGU6IGN1c3RvbSgnL3R5cGVzL1F1ZXJ5L2ZpZWxkcy9nZXRFeGFtcGxlJylcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGN1c3RvbSguLi5hcm5zOiBzdHJpbmdbXSk6IElhbVJlc291cmNlIHtcbiAgICAgICAgaWYgKGFybnMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IDEgY3VzdG9tIEFSTiBtdXN0IGJlIHByb3ZpZGVkLicpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgSWFtUmVzb3VyY2UoYXJucyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEdlbmVyYXRlIHRoZSByZXNvdXJjZSBuYW1lcyBnaXZlbiBhIHR5cGUgYW5kIGZpZWxkc1xuICAgICAqXG4gICAgICogQHBhcmFtIHR5cGUgVGhlIHR5cGUgdGhhdCBuZWVkcyB0byBiZSBhbGxvd2VkXG4gICAgICogQHBhcmFtIGZpZWxkcyBUaGUgZmllbGRzIHRoYXQgbmVlZCB0byBiZSBhbGxvd2VkLCBpZiBlbXB0eSBncmFudCBwZXJtaXNzaW9ucyB0byBBTEwgZmllbGRzXG4gICAgICpcbiAgICAgKiBFeGFtcGxlOiBvZlR5cGUoJ1F1ZXJ5JywgJ0dldEV4YW1wbGUnKVxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgb2ZUeXBlKHR5cGU6IHN0cmluZywgLi4uZmllbGRzOiBzdHJpbmdbXSk6IElhbVJlc291cmNlIHtcbiAgICAgICAgY29uc3QgYXJucyA9IGZpZWxkcy5sZW5ndGggPyBmaWVsZHMubWFwKChmaWVsZCkgPT4gYHR5cGVzLyR7dHlwZX0vZmllbGRzLyR7ZmllbGR9YCkgOiBbYHR5cGVzLyR7dHlwZX0vKmBdO1xuICAgICAgICByZXR1cm4gbmV3IElhbVJlc291cmNlKGFybnMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBHZW5lcmF0ZSB0aGUgcmVzb3VyY2UgbmFtZXMgdGhhdCBhY2NlcHRzIGFsbCB0eXBlczogYCpgXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBhbGwoKTogSWFtUmVzb3VyY2Uge1xuICAgICAgICByZXR1cm4gbmV3IElhbVJlc291cmNlKFsnKiddKTtcbiAgICB9XG4gICAgcHJpdmF0ZSBhcm5zOiBzdHJpbmdbXTtcbiAgICBwcml2YXRlIGNvbnN0cnVjdG9yKGFybnM6IHN0cmluZ1tdKSB7XG4gICAgICAgIHRoaXMuYXJucyA9IGFybnM7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybiB0aGUgUmVzb3VyY2UgQVJOXG4gICAgICpcbiAgICAgKiBAcGFyYW0gYXBpIFRoZSBHcmFwaFFMIEFQSSB0byBnaXZlIHBlcm1pc3Npb25zXG4gICAgICovXG4gICAgcHVibGljIHJlc291cmNlQXJucyhhcGk6IEdyYXBocWxBcGkpOiBzdHJpbmdbXSB7XG4gICAgICAgIHJldHVybiB0aGlzLmFybnMubWFwKChhcm4pID0+IFN0YWNrLm9mKGFwaSkuZm9ybWF0QXJuKHtcbiAgICAgICAgICAgIHNlcnZpY2U6ICdhcHBzeW5jJyxcbiAgICAgICAgICAgIHJlc291cmNlOiBgYXBpcy8ke2FwaS5hcGlJZH1gLFxuICAgICAgICAgICAgc2VwOiAnLycsXG4gICAgICAgICAgICByZXNvdXJjZU5hbWU6IGAke2Fybn1gLFxuICAgICAgICB9KSk7XG4gICAgfVxufVxuLyoqXG4gKiBBdHRyaWJ1dGVzIGZvciBHcmFwaFFMIGltcG9ydHNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBHcmFwaHFsQXBpQXR0cmlidXRlcyB7XG4gICAgLyoqXG4gICAgICogYW4gdW5pcXVlIEFXUyBBcHBTeW5jIEdyYXBoUUwgQVBJIGlkZW50aWZpZXJcbiAgICAgKiBpLmUuICdseHo3NzVsd2RyZ2NuZGd6M251cnZhYzdvYSdcbiAgICAgKi9cbiAgICByZWFkb25seSBncmFwaHFsQXBpSWQ6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiB0aGUgYXJuIGZvciB0aGUgR3JhcGhRTCBBcGlcbiAgICAgKiBAZGVmYXVsdCAtIGF1dG9nZW5lcmF0ZWQgYXJuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZ3JhcGhxbEFwaUFybj86IHN0cmluZztcbn1cbi8qKlxuICogQW4gQXBwU3luYyBHcmFwaFFMIEFQSVxuICpcbiAqIEByZXNvdXJjZSBBV1M6OkFwcFN5bmM6OkdyYXBoUUxBcGlcbiAqL1xuZXhwb3J0IGNsYXNzIEdyYXBocWxBcGkgZXh0ZW5kcyBHcmFwaHFsQXBpQmFzZSB7XG4gICAgLyoqXG4gICAgICogSW1wb3J0IGEgR3JhcGhRTCBBUEkgdGhyb3VnaCB0aGlzIGZ1bmN0aW9uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gc2NvcGUgc2NvcGVcbiAgICAgKiBAcGFyYW0gaWQgaWRcbiAgICAgKiBAcGFyYW0gYXR0cnMgR3JhcGhRTCBBUEkgQXR0cmlidXRlcyBvZiBhbiBBUElcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21HcmFwaHFsQXBpQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogR3JhcGhxbEFwaUF0dHJpYnV0ZXMpOiBJR3JhcGhxbEFwaSB7XG4gICAgICAgIGNvbnN0IGFybiA9IGF0dHJzLmdyYXBocWxBcGlBcm4gPz8gU3RhY2sub2Yoc2NvcGUpLmZvcm1hdEFybih7XG4gICAgICAgICAgICBzZXJ2aWNlOiAnYXBwc3luYycsXG4gICAgICAgICAgICByZXNvdXJjZTogYGFwaXMvJHthdHRycy5ncmFwaHFsQXBpSWR9YCxcbiAgICAgICAgfSk7XG4gICAgICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIEdyYXBocWxBcGlCYXNlIHtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBhcGlJZCA9IGF0dHJzLmdyYXBocWxBcGlJZDtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBhcm4gPSBhcm47XG4gICAgICAgICAgICBjb25zdHJ1Y3RvcihzOiBDb25zdHJ1Y3QsIGk6IHN0cmluZykge1xuICAgICAgICAgICAgICAgIHN1cGVyKHMsIGkpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIGFuIHVuaXF1ZSBBV1MgQXBwU3luYyBHcmFwaFFMIEFQSSBpZGVudGlmaWVyXG4gICAgICogaS5lLiAnbHh6Nzc1bHdkcmdjbmRnejNudXJ2YWM3b2EnXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGFwaUlkOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogdGhlIEFSTiBvZiB0aGUgQVBJXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGFybjogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIHRoZSBVUkwgb2YgdGhlIGVuZHBvaW50IGNyZWF0ZWQgYnkgQXBwU3luY1xuICAgICAqXG4gICAgICogQGF0dHJpYnV0ZSBHcmFwaFFsVXJsXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGdyYXBocWxVcmw6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiB0aGUgbmFtZSBvZiB0aGUgQVBJXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiB0aGUgc2NoZW1hIGF0dGFjaGVkIHRvIHRoaXMgYXBpXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IHNjaGVtYTogU2NoZW1hO1xuICAgIC8qKlxuICAgICAqIHRoZSBjb25maWd1cmVkIEFQSSBrZXksIGlmIHByZXNlbnRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm8gYXBpIGtleVxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBhcGlLZXk/OiBzdHJpbmc7XG4gICAgcHJpdmF0ZSBzY2hlbWFSZXNvdXJjZTogQ2ZuR3JhcGhRTFNjaGVtYTtcbiAgICBwcml2YXRlIGFwaTogQ2ZuR3JhcGhRTEFwaTtcbiAgICBwcml2YXRlIGFwaUtleVJlc291cmNlPzogQ2ZuQXBpS2V5O1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBHcmFwaHFsQXBpUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgY29uc3QgZGVmYXVsdE1vZGUgPSBwcm9wcy5hdXRob3JpemF0aW9uQ29uZmlnPy5kZWZhdWx0QXV0aG9yaXphdGlvbiA/P1xuICAgICAgICAgICAgeyBhdXRob3JpemF0aW9uVHlwZTogQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSB9O1xuICAgICAgICBjb25zdCBhZGRpdGlvbmFsTW9kZXMgPSBwcm9wcy5hdXRob3JpemF0aW9uQ29uZmlnPy5hZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzID8/IFtdO1xuICAgICAgICBjb25zdCBtb2RlcyA9IFtkZWZhdWx0TW9kZSwgLi4uYWRkaXRpb25hbE1vZGVzXTtcbiAgICAgICAgdGhpcy52YWxpZGF0ZUF1dGhvcml6YXRpb25Qcm9wcyhtb2Rlcyk7XG4gICAgICAgIHRoaXMuYXBpID0gbmV3IENmbkdyYXBoUUxBcGkodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgICAgICAgbmFtZTogcHJvcHMubmFtZSxcbiAgICAgICAgICAgIGF1dGhlbnRpY2F0aW9uVHlwZTogZGVmYXVsdE1vZGUuYXV0aG9yaXphdGlvblR5cGUsXG4gICAgICAgICAgICBsb2dDb25maWc6IHRoaXMuc2V0dXBMb2dDb25maWcocHJvcHMubG9nQ29uZmlnKSxcbiAgICAgICAgICAgIG9wZW5JZENvbm5lY3RDb25maWc6IHRoaXMuc2V0dXBPcGVuSWRDb25uZWN0Q29uZmlnKGRlZmF1bHRNb2RlLm9wZW5JZENvbm5lY3RDb25maWcpLFxuICAgICAgICAgICAgdXNlclBvb2xDb25maWc6IHRoaXMuc2V0dXBVc2VyUG9vbENvbmZpZyhkZWZhdWx0TW9kZS51c2VyUG9vbENvbmZpZyksXG4gICAgICAgICAgICBhZGRpdGlvbmFsQXV0aGVudGljYXRpb25Qcm92aWRlcnM6IHRoaXMuc2V0dXBBZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzKGFkZGl0aW9uYWxNb2RlcyksXG4gICAgICAgICAgICB4cmF5RW5hYmxlZDogcHJvcHMueHJheUVuYWJsZWQsXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLmFwaUlkID0gdGhpcy5hcGkuYXR0ckFwaUlkO1xuICAgICAgICB0aGlzLmFybiA9IHRoaXMuYXBpLmF0dHJBcm47XG4gICAgICAgIHRoaXMuZ3JhcGhxbFVybCA9IHRoaXMuYXBpLmF0dHJHcmFwaFFsVXJsO1xuICAgICAgICB0aGlzLm5hbWUgPSB0aGlzLmFwaS5uYW1lO1xuICAgICAgICB0aGlzLnNjaGVtYSA9IHByb3BzLnNjaGVtYSA/PyBuZXcgU2NoZW1hKCk7XG4gICAgICAgIHRoaXMuc2NoZW1hUmVzb3VyY2UgPSB0aGlzLnNjaGVtYS5iaW5kKHRoaXMpO1xuICAgICAgICBpZiAobW9kZXMuc29tZSgobW9kZSkgPT4gbW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSkpIHtcbiAgICAgICAgICAgIGNvbnN0IGNvbmZpZyA9IG1vZGVzLmZpbmQoKG1vZGU6IEF1dGhvcml6YXRpb25Nb2RlKSA9PiB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIG1vZGUuYXV0aG9yaXphdGlvblR5cGUgPT09IEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVkgJiYgbW9kZS5hcGlLZXlDb25maWc7XG4gICAgICAgICAgICB9KT8uYXBpS2V5Q29uZmlnO1xuICAgICAgICAgICAgdGhpcy5hcGlLZXlSZXNvdXJjZSA9IHRoaXMuY3JlYXRlQVBJS2V5KGNvbmZpZyk7XG4gICAgICAgICAgICB0aGlzLmFwaUtleVJlc291cmNlLmFkZERlcGVuZHNPbih0aGlzLnNjaGVtYVJlc291cmNlKTtcbiAgICAgICAgICAgIHRoaXMuYXBpS2V5ID0gdGhpcy5hcGlLZXlSZXNvdXJjZS5hdHRyQXBpS2V5O1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZHMgYW4gSUFNIHBvbGljeSBzdGF0ZW1lbnQgYXNzb2NpYXRlZCB3aXRoIHRoaXMgR3JhcGhRTEFwaSB0byBhbiBJQU1cbiAgICAgKiBwcmluY2lwYWwncyBwb2xpY3kuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsXG4gICAgICogQHBhcmFtIHJlc291cmNlcyBUaGUgc2V0IG9mIHJlc291cmNlcyB0byBhbGxvdyAoaS5lLiAuLi46W3JlZ2lvbl06W2FjY291bnRJZF06YXBpcy9HcmFwaFFMSWQvLi4uKVxuICAgICAqIEBwYXJhbSBhY3Rpb25zIFRoZSBhY3Rpb25zIHRoYXQgc2hvdWxkIGJlIGdyYW50ZWQgdG8gdGhlIHByaW5jaXBhbCAoaS5lLiBhcHBzeW5jOmdyYXBocWwgKVxuICAgICAqL1xuICAgIHB1YmxpYyBncmFudChncmFudGVlOiBJR3JhbnRhYmxlLCByZXNvdXJjZXM6IElhbVJlc291cmNlLCAuLi5hY3Rpb25zOiBzdHJpbmdbXSk6IEdyYW50IHtcbiAgICAgICAgcmV0dXJuIEdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgICAgICAgIGdyYW50ZWUsXG4gICAgICAgICAgICBhY3Rpb25zLFxuICAgICAgICAgICAgcmVzb3VyY2VBcm5zOiByZXNvdXJjZXMucmVzb3VyY2VBcm5zKHRoaXMpLFxuICAgICAgICAgICAgc2NvcGU6IHRoaXMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGRzIGFuIElBTSBwb2xpY3kgc3RhdGVtZW50IGZvciBNdXRhdGlvbiBhY2Nlc3MgdG8gdGhpcyBHcmFwaFFMQXBpIHRvIGFuIElBTVxuICAgICAqIHByaW5jaXBhbCdzIHBvbGljeS5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWxcbiAgICAgKiBAcGFyYW0gZmllbGRzIFRoZSBmaWVsZHMgdG8gZ3JhbnQgYWNjZXNzIHRvIHRoYXQgYXJlIE11dGF0aW9ucyAobGVhdmUgYmxhbmsgZm9yIGFsbClcbiAgICAgKi9cbiAgICBwdWJsaWMgZ3JhbnRNdXRhdGlvbihncmFudGVlOiBJR3JhbnRhYmxlLCAuLi5maWVsZHM6IHN0cmluZ1tdKTogR3JhbnQge1xuICAgICAgICByZXR1cm4gdGhpcy5ncmFudChncmFudGVlLCBJYW1SZXNvdXJjZS5vZlR5cGUoJ011dGF0aW9uJywgLi4uZmllbGRzKSwgJ2FwcHN5bmM6R3JhcGhRTCcpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGRzIGFuIElBTSBwb2xpY3kgc3RhdGVtZW50IGZvciBRdWVyeSBhY2Nlc3MgdG8gdGhpcyBHcmFwaFFMQXBpIHRvIGFuIElBTVxuICAgICAqIHByaW5jaXBhbCdzIHBvbGljeS5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWxcbiAgICAgKiBAcGFyYW0gZmllbGRzIFRoZSBmaWVsZHMgdG8gZ3JhbnQgYWNjZXNzIHRvIHRoYXQgYXJlIFF1ZXJpZXMgKGxlYXZlIGJsYW5rIGZvciBhbGwpXG4gICAgICovXG4gICAgcHVibGljIGdyYW50UXVlcnkoZ3JhbnRlZTogSUdyYW50YWJsZSwgLi4uZmllbGRzOiBzdHJpbmdbXSk6IEdyYW50IHtcbiAgICAgICAgcmV0dXJuIHRoaXMuZ3JhbnQoZ3JhbnRlZSwgSWFtUmVzb3VyY2Uub2ZUeXBlKCdRdWVyeScsIC4uLmZpZWxkcyksICdhcHBzeW5jOkdyYXBoUUwnKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkcyBhbiBJQU0gcG9saWN5IHN0YXRlbWVudCBmb3IgU3Vic2NyaXB0aW9uIGFjY2VzcyB0byB0aGlzIEdyYXBoUUxBcGkgdG8gYW4gSUFNXG4gICAgICogcHJpbmNpcGFsJ3MgcG9saWN5LlxuICAgICAqXG4gICAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbFxuICAgICAqIEBwYXJhbSBmaWVsZHMgVGhlIGZpZWxkcyB0byBncmFudCBhY2Nlc3MgdG8gdGhhdCBhcmUgU3Vic2NyaXB0aW9ucyAobGVhdmUgYmxhbmsgZm9yIGFsbClcbiAgICAgKi9cbiAgICBwdWJsaWMgZ3JhbnRTdWJzY3JpcHRpb24oZ3JhbnRlZTogSUdyYW50YWJsZSwgLi4uZmllbGRzOiBzdHJpbmdbXSk6IEdyYW50IHtcbiAgICAgICAgcmV0dXJuIHRoaXMuZ3JhbnQoZ3JhbnRlZSwgSWFtUmVzb3VyY2Uub2ZUeXBlKCdTdWJzY3JpcHRpb24nLCAuLi5maWVsZHMpLCAnYXBwc3luYzpHcmFwaFFMJyk7XG4gICAgfVxuICAgIHByaXZhdGUgdmFsaWRhdGVBdXRob3JpemF0aW9uUHJvcHMobW9kZXM6IEF1dGhvcml6YXRpb25Nb2RlW10pIHtcbiAgICAgICAgbW9kZXMubWFwKChtb2RlKSA9PiB7XG4gICAgICAgICAgICBpZiAobW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuT0lEQyAmJiAhbW9kZS5vcGVuSWRDb25uZWN0Q29uZmlnKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNaXNzaW5nIGRlZmF1bHQgT0lEQyBDb25maWd1cmF0aW9uJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAobW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuVVNFUl9QT09MICYmICFtb2RlLnVzZXJQb29sQ29uZmlnKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNaXNzaW5nIGRlZmF1bHQgT0lEQyBDb25maWd1cmF0aW9uJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgICBpZiAobW9kZXMuZmlsdGVyKChtb2RlKSA9PiBtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZKS5sZW5ndGggPiAxKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5cXCd0IGR1cGxpY2F0ZSBBUElfS0VZIGNvbmZpZ3VyYXRpb24uIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvc2VjdXJpdHkuaHRtbCcpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChtb2Rlcy5maWx0ZXIoKG1vZGUpID0+IG1vZGUuYXV0aG9yaXphdGlvblR5cGUgPT09IEF1dGhvcml6YXRpb25UeXBlLklBTSkubGVuZ3RoID4gMSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgY2FuXFwndCBkdXBsaWNhdGUgSUFNIGNvbmZpZ3VyYXRpb24uIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvc2VjdXJpdHkuaHRtbCcpO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBzY2hlbWEgZGVwZW5kZW5jeSB0byBhIGdpdmVuIGNvbnN0cnVjdFxuICAgICAqXG4gICAgICogQHBhcmFtIGNvbnN0cnVjdCB0aGUgZGVwZW5kZWVcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkU2NoZW1hRGVwZW5kZW5jeShjb25zdHJ1Y3Q6IENmblJlc291cmNlKTogYm9vbGVhbiB7XG4gICAgICAgIGNvbnN0cnVjdC5hZGREZXBlbmRzT24odGhpcy5zY2hlbWFSZXNvdXJjZSk7XG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cbiAgICBwcml2YXRlIHNldHVwTG9nQ29uZmlnKGNvbmZpZz86IExvZ0NvbmZpZykge1xuICAgICAgICBpZiAoIWNvbmZpZylcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIGNvbnN0IHJvbGUgPSBuZXcgUm9sZSh0aGlzLCAnQXBpTG9nc1JvbGUnLCB7XG4gICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdhcHBzeW5jLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICAgICAgICAgIE1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTQXBwU3luY1B1c2hUb0Nsb3VkV2F0Y2hMb2dzJyksXG4gICAgICAgICAgICBdLFxuICAgICAgICB9KTtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIGNsb3VkV2F0Y2hMb2dzUm9sZUFybjogcm9sZS5yb2xlQXJuLFxuICAgICAgICAgICAgZXhjbHVkZVZlcmJvc2VDb250ZW50OiBjb25maWcuZXhjbHVkZVZlcmJvc2VDb250ZW50LFxuICAgICAgICAgICAgZmllbGRMb2dMZXZlbDogY29uZmlnLmZpZWxkTG9nTGV2ZWwsXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgc2V0dXBPcGVuSWRDb25uZWN0Q29uZmlnKGNvbmZpZz86IE9wZW5JZENvbm5lY3RDb25maWcpIHtcbiAgICAgICAgaWYgKCFjb25maWcpXG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgYXV0aFR0bDogY29uZmlnLnRva2VuRXhwaXJ5RnJvbUF1dGgsXG4gICAgICAgICAgICBjbGllbnRJZDogY29uZmlnLmNsaWVudElkLFxuICAgICAgICAgICAgaWF0VHRsOiBjb25maWcudG9rZW5FeHBpcnlGcm9tSXNzdWUsXG4gICAgICAgICAgICBpc3N1ZXI6IGNvbmZpZy5vaWRjUHJvdmlkZXIsXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgc2V0dXBVc2VyUG9vbENvbmZpZyhjb25maWc/OiBVc2VyUG9vbENvbmZpZykge1xuICAgICAgICBpZiAoIWNvbmZpZylcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICB1c2VyUG9vbElkOiBjb25maWcudXNlclBvb2wudXNlclBvb2xJZCxcbiAgICAgICAgICAgIGF3c1JlZ2lvbjogY29uZmlnLnVzZXJQb29sLnN0YWNrLnJlZ2lvbixcbiAgICAgICAgICAgIGFwcElkQ2xpZW50UmVnZXg6IGNvbmZpZy5hcHBJZENsaWVudFJlZ2V4LFxuICAgICAgICAgICAgZGVmYXVsdEFjdGlvbjogY29uZmlnLmRlZmF1bHRBY3Rpb24gfHwgVXNlclBvb2xEZWZhdWx0QWN0aW9uLkFMTE9XLFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIHNldHVwQWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2Rlcyhtb2Rlcz86IEF1dGhvcml6YXRpb25Nb2RlW10pIHtcbiAgICAgICAgaWYgKCFtb2RlcyB8fCBtb2Rlcy5sZW5ndGggPT09IDApXG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICByZXR1cm4gbW9kZXMucmVkdWNlPENmbkdyYXBoUUxBcGkuQWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJQcm9wZXJ0eVtdPigoYWNjLCBtb2RlKSA9PiBbXG4gICAgICAgICAgICAuLi5hY2MsIHtcbiAgICAgICAgICAgICAgICBhdXRoZW50aWNhdGlvblR5cGU6IG1vZGUuYXV0aG9yaXphdGlvblR5cGUsXG4gICAgICAgICAgICAgICAgdXNlclBvb2xDb25maWc6IHRoaXMuc2V0dXBVc2VyUG9vbENvbmZpZyhtb2RlLnVzZXJQb29sQ29uZmlnKSxcbiAgICAgICAgICAgICAgICBvcGVuSWRDb25uZWN0Q29uZmlnOiB0aGlzLnNldHVwT3BlbklkQ29ubmVjdENvbmZpZyhtb2RlLm9wZW5JZENvbm5lY3RDb25maWcpLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgXSwgW10pO1xuICAgIH1cbiAgICBwcml2YXRlIGNyZWF0ZUFQSUtleShjb25maWc/OiBBcGlLZXlDb25maWcpIHtcbiAgICAgICAgbGV0IGV4cGlyZXM6IG51bWJlciB8IHVuZGVmaW5lZDtcbiAgICAgICAgaWYgKGNvbmZpZz8uZXhwaXJlcykge1xuICAgICAgICAgICAgZXhwaXJlcyA9IG5ldyBEYXRlKGNvbmZpZy5leHBpcmVzKS52YWx1ZU9mKCk7XG4gICAgICAgICAgICBjb25zdCBkYXlzID0gKGQ6IG51bWJlcikgPT4gRGF0ZS5ub3coKSArIER1cmF0aW9uLmRheXMoZCkudG9NaWxsaXNlY29uZHMoKTtcbiAgICAgICAgICAgIGlmIChleHBpcmVzIDwgZGF5cygxKSB8fCBleHBpcmVzID4gZGF5cygzNjUpKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgRXJyb3IoJ0FQSSBrZXkgZXhwaXJhdGlvbiBtdXN0IGJlIGJldHdlZW4gMSBhbmQgMzY1IGRheXMuJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBleHBpcmVzID0gTWF0aC5yb3VuZChleHBpcmVzIC8gMTAwMCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBDZm5BcGlLZXkodGhpcywgYCR7Y29uZmlnPy5uYW1lIHx8ICdEZWZhdWx0J31BcGlLZXlgLCB7XG4gICAgICAgICAgICBleHBpcmVzLFxuICAgICAgICAgICAgZGVzY3JpcHRpb246IGNvbmZpZz8uZGVzY3JpcHRpb24sXG4gICAgICAgICAgICBhcGlJZDogdGhpcy5hcGlJZCxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEVzY2FwZSBoYXRjaCB0byBhcHBlbmQgdG8gU2NoZW1hIGFzIGRlc2lyZWQuIFdpbGwgYWx3YXlzIHJlc3VsdFxuICAgICAqIGluIGEgbmV3bGluZS5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBhZGRpdGlvbiB0aGUgYWRkaXRpb24gdG8gYWRkIHRvIHNjaGVtYVxuICAgICAqIEBwYXJhbSBkZWxpbWl0ZXIgdGhlIGRlbGltaXRlciBiZXR3ZWVuIHNjaGVtYSBhbmQgYWRkaXRpb25cbiAgICAgKiBAZGVmYXVsdCAtICcnXG4gICAgICpcbiAgICAgKiBAZXhwZXJpbWVudGFsXG4gICAgICovXG4gICAgcHVibGljIGFkZFRvU2NoZW1hKGFkZGl0aW9uOiBzdHJpbmcsIGRlbGltaXRlcj86IHN0cmluZyk6IHZvaWQge1xuICAgICAgICB0aGlzLnNjaGVtYS5hZGRUb1NjaGVtYShhZGRpdGlvbiwgZGVsaW1pdGVyKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkIHR5cGUgdG8gdGhlIHNjaGVtYVxuICAgICAqXG4gICAgICogQHBhcmFtIHR5cGUgdGhlIGludGVybWVkaWF0ZSB0eXBlIHRvIGFkZCB0byB0aGUgc2NoZW1hXG4gICAgICpcbiAgICAgKiBAZXhwZXJpbWVudGFsXG4gICAgICovXG4gICAgcHVibGljIGFkZFR5cGUodHlwZTogSUludGVybWVkaWF0ZVR5cGUpOiBJSW50ZXJtZWRpYXRlVHlwZSB7XG4gICAgICAgIHJldHVybiB0aGlzLnNjaGVtYS5hZGRUeXBlKHR5cGUpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSBxdWVyeSBmaWVsZCB0byB0aGUgc2NoZW1hJ3MgUXVlcnkuIElmIG9uZSBpc24ndCBzZXQgYnlcbiAgICAgKiB0aGUgdXNlciwgQ0RLIHdpbGwgY3JlYXRlIGFuIE9iamVjdCBUeXBlIGNhbGxlZCAnUXVlcnknLiBGb3IgZXhhbXBsZSxcbiAgICAgKlxuICAgICAqIHR5cGUgUXVlcnkge1xuICAgICAqICAgZmllbGROYW1lOiBGaWVsZC5yZXR1cm5UeXBlXG4gICAgICogfVxuICAgICAqXG4gICAgICogQHBhcmFtIGZpZWxkTmFtZSB0aGUgbmFtZSBvZiB0aGUgcXVlcnlcbiAgICAgKiBAcGFyYW0gZmllbGQgdGhlIHJlc29sdmFibGUgZmllbGQgdG8gZm9yIHRoaXMgcXVlcnlcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkUXVlcnkoZmllbGROYW1lOiBzdHJpbmcsIGZpZWxkOiBSZXNvbHZhYmxlRmllbGQpOiBPYmplY3RUeXBlIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuc2NoZW1hLmFkZFF1ZXJ5KGZpZWxkTmFtZSwgZmllbGQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSBtdXRhdGlvbiBmaWVsZCB0byB0aGUgc2NoZW1hJ3MgTXV0YXRpb24uIElmIG9uZSBpc24ndCBzZXQgYnlcbiAgICAgKiB0aGUgdXNlciwgQ0RLIHdpbGwgY3JlYXRlIGFuIE9iamVjdCBUeXBlIGNhbGxlZCAnTXV0YXRpb24nLiBGb3IgZXhhbXBsZSxcbiAgICAgKlxuICAgICAqIHR5cGUgTXV0YXRpb24ge1xuICAgICAqICAgZmllbGROYW1lOiBGaWVsZC5yZXR1cm5UeXBlXG4gICAgICogfVxuICAgICAqXG4gICAgICogQHBhcmFtIGZpZWxkTmFtZSB0aGUgbmFtZSBvZiB0aGUgTXV0YXRpb25cbiAgICAgKiBAcGFyYW0gZmllbGQgdGhlIHJlc29sdmFibGUgZmllbGQgdG8gZm9yIHRoaXMgTXV0YXRpb25cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkTXV0YXRpb24oZmllbGROYW1lOiBzdHJpbmcsIGZpZWxkOiBSZXNvbHZhYmxlRmllbGQpOiBPYmplY3RUeXBlIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuc2NoZW1hLmFkZE11dGF0aW9uKGZpZWxkTmFtZSwgZmllbGQpO1xuICAgIH1cbn1cbiJdfQ==