"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthorizationType = exports.Method = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const mock_1 = require("./integrations/mock");
const restapi_1 = require("./restapi");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.resource = props.resource;
        this.api = props.resource.api;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.api);
        }
        const integration = (_b = (_a = props.integration) !== null && _a !== void 0 ? _a : this.resource.defaultIntegration) !== null && _b !== void 0 ? _b : new mock_1.MockIntegration();
        const bindResult = integration.bind(this);
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.api.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(bindResult),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_c = options.authorizationScopes) !== null && _c !== void 0 ? _c : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        if (restapi_1.RestApiBase._isRestApiBase(props.resource.api)) {
            props.resource.api._attachMethod(this);
        }
        const deployment = props.resource.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({
                method: {
                    ...methodProps,
                    integrationToken: bindResult === null || bindResult === void 0 ? void 0 : bindResult.deploymentToken,
                },
            });
        }
    }
    /**
     * The RestApi associated with this Method
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        return this.resource.restApi;
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled, or can be explicitly assigned.
     * When not configured, {stage} will be set to '*', as a shorthand for 'all stages'.
     *
     * @attribute
     */
    get methodArn() {
        var _a;
        const stage = (_a = this.api.deploymentStage) === null || _a === void 0 ? void 0 : _a.stageName;
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    renderIntegration(bindResult) {
        var _a;
        const options = (_a = bindResult.options) !== null && _a !== void 0 ? _a : {};
        let credentials;
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // eslint-disable-next-line max-len
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: bindResult.type,
            uri: bindResult.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: bindResult.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _a;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_a = options.requestValidator) === null || _a === void 0 ? void 0 : _a.requestValidatorId;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,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