/**
 * @private For internal Amplify use.
 *
 * Creates a new scope for promises, observables, and other types of work or
 * processes that may be running in the background. This manager provides
 * an singular entrypoint to request termination and await completion.
 *
 * As work completes on its own prior to close, the manager removes them
 * from the registry to avoid holding references to completed jobs.
 */
export declare class BackgroundProcessManager {
    /**
     * A string indicating whether the manager is accepting new work ("Open"),
     * waiting for work to complete ("Closing"), or fully done with all
     * submitted work and *not* accepting new jobs ("Closed").
     */
    private _state;
    private _closingPromise;
    /**
     * The list of outstanding jobs we'll need to wait for upon `close()`
     */
    private jobs;
    /**
     * Creates a new manager for promises, observables, and other types
     * of work that may be running in the background. This manager provides
     * a centralized mechanism to request termination and await completion.
     */
    constructor();
    /**
     * Executes an async `job` function, passing the return value through to
     * the caller, registering it as a running job in the manager. When the
     * manager *closes*, it will `await` the job.
     *
     * @param job The function to execute.
     * @param description Optional description to help identify pending jobs.
     * @returns The return value from the given function.
     */
    add<T>(job: () => Promise<T>, description?: string): Promise<T>;
    /**
     * Executes an async `job` function, passing the return value through to
     * the caller, registering it as a running job in the manager. When the
     * manager *closes*, it will request termination by resolving the
     * provided `onTerminate` promise. It will then `await` the job, so it is
     * important that the job still `resolve()` or `reject()` when responding
     * to a termination request.
     *
     * @param job The function to execute.
     * @param description Optional description to help identify pending jobs.
     * @returns The return value from the given function.
     */
    add<T>(job: (onTerminate: Promise<void>) => Promise<T>, description?: string): Promise<T>;
    /**
     * Create a no-op job, registers it with the manager, and returns hooks
     * to the caller to signal the job's completion and respond to termination
     * requests.
     *
     * When the manager closes, the no-op job will be `await`-ed, so its
     * important to always `resolve()` or `reject()` when done responding to an
     * `onTerminate` signal.
     * @param description Optional description to help identify pending jobs.
     * @returns Job promise hooks + onTerminate signaling promise
     */
    add(description?: string): {
        resolve: (value?: unknown) => void;
        reject: (reason?: any) => void;
        onTerminate: Promise<void>;
    };
    /**
     * Adds another job manager to await on at the time of closing. the inner
     * manager's termination is signaled when this manager's `close()` is
     * called for.
     *
     * @param job The inner job manager to await.
     * @param description Optional description to help identify pending jobs.
     */
    add(job: BackgroundProcessManager, description?: string): any;
    /**
     * Adds a **cleaner** function that doesn't immediately get executed.
     * Instead, the caller gets a **terminate** function back. The *cleaner* is
     * invoked only once the mananger *closes* or the returned **terminate**
     * function is called.
     *
     * @param clean The cleanup function.
     * @param description Optional description to help identify pending jobs.
     * @returns A terminate function.
     */
    addCleaner<T>(clean: () => Promise<T>, description?: string): () => Promise<void>;
    private addFunction;
    private addManager;
    /**
     * Creates and registers a fabricated job for processes that need to operate
     * with callbacks/hooks. The returned `resolve` and `reject`
     * functions can be used to signal the job is done successfully or not.
     * The returned `onTerminate` is a promise that will resolve when the
     * manager is requesting the termination of the job.
     *
     * @param description Optional description to help identify pending jobs.
     * @returns `{ resolve, reject, onTerminate }`
     */
    private addHook;
    /**
     * Adds a Promise based job to the list of jobs for monitoring and listens
     * for either a success or failure, upon which the job is considered "done"
     * and removed from the registry.
     *
     * @param promise A promise that is on its way to being returned to a
     * caller, which needs to be tracked as a background job.
     * @param terminate The termination function to register, which can be
     * invoked to request the job stop.
     * @param description Optional description to help identify pending jobs.
     */
    private registerPromise;
    /**
     * The number of jobs being waited on.
     *
     * We don't use this for anything. It's just informational for the caller,
     * and can be used in logging and testing.
     *
     * @returns the number of jobs.
     */
    get length(): number;
    /**
     * The execution state of the manager. One of:
     *
     * 1. "Open" -> Accepting new jobs
     * 1. "Closing" -> Not accepting new work. Waiting for jobs to complete.
     * 1. "Closed" -> Not accepting new work. All submitted jobs are complete.
     */
    get state(): BackgroundProcessManagerState;
    /**
     * The registered `description` of all still-pending jobs.
     *
     * @returns descriptions as an array.
     */
    get pending(): string[];
    /**
     * Whether the manager is accepting new jobs.
     */
    get isOpen(): boolean;
    /**
     * Whether the manager is rejecting new work, but still waiting for
     * submitted work to complete.
     */
    get isClosing(): boolean;
    /**
     * Whether the manager is rejecting work and done waiting for submitted
     * work to complete.
     */
    get isClosed(): boolean;
    private closedFailure;
    /**
     * Signals jobs to stop (for those that accept interruptions) and waits
     * for confirmation that jobs have stopped.
     *
     * This immediately puts the manager into a closing state and just begins
     * to reject new work. After all work in the manager is complete, the
     * manager goes into a `Completed` state and `close()` returns.
     *
     * This call is idempotent.
     *
     * If the manager is already closing or closed, `finalCleaup` is not executed.
     *
     * @param onClosed
     * @returns The settled results of each still-running job's promise. If the
     * manager is already closed, this will contain the results as of when the
     * manager's `close()` was called in an `Open` state.
     */
    close(): Promise<any>;
    /**
     * Signals the manager to start accepting work (again) and returns once
     * the manager is ready to do so.
     *
     * If the state is already `Open`, this call is a no-op.
     *
     * If the state is `Closed`, this call simply updates state and returns.
     *
     * If the state is `Closing`, this call waits for completion before it
     * updates the state and returns.
     */
    open(): Promise<void>;
}
/**
 *
 */
export declare class BackgroundManagerNotOpenError extends Error {
    constructor(message: string);
}
/**
 * All possible states a `BackgroundProcessManager` instance can be in.
 */
export declare enum BackgroundProcessManagerState {
    /**
     * Accepting new jobs.
     */
    Open = "Open",
    /**
     * Not accepting new jobs. Waiting for submitted jobs to complete.
     */
    Closing = "Closing",
    /**
     * Not accepting new jobs. All submitted jobs are complete.
     */
    Closed = "Closed"
}
