# Copyright 2018-2020 Faculty Science Limited
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Access Faculty job logs.
"""


from collections import namedtuple

from marshmallow import fields, post_load

from faculty.clients.base import BaseSchema, BaseClient


LogPart = namedtuple(
    "LogPart", ["log_part_number", "line_number", "content", "timestamp"]
)
LogPartsResponse = namedtuple("LogPartsResponse", ["log_parts"])


class LogClient(BaseClient):
    """Client for the Faculty log service.

    Either build this client with a session directly, or use the
    :func:`faculty.client` helper function:

    >>> client = faculty.client("log")

    Parameters
    ----------
    session : faculty.session.Session
        The session to use to make requests
    """

    _SERVICE_NAME = "wozniak"

    def get_subrun_command_logs(self, project_id, job_id, run_id, subrun_id):
        """Get the logs generated by the main command of a job subrun.

        Parameters
        ----------
        project_id : uuid.UUID
            The ID of the project containing the job.
        job_id : uuid.UUID
            The ID of the job containing the run.
        run_id : uuid.UUID
            The ID of the run containing the subrun.
        subrun_id : uuid.UUID
            The ID of the subrun to get logs from.

        Returns
        -------
        LogPartsResponse
            The logs of the subrun's main command.
        """
        template = "/project/{}/job/{}/run/{}/subrun/{}/command/log-part"
        endpoint = template.format(project_id, job_id, run_id, subrun_id)
        return self._get_logs(endpoint)

    def get_subrun_environment_step_logs(
        self, project_id, job_id, run_id, subrun_id, environment_step_id
    ):
        """Get the logs generated by an environment step of a job subrun.

        Parameters
        ----------
        project_id : uuid.UUID
            The ID of the project containing the job.
        job_id : uuid.UUID
            The ID of the job containing the run.
        run_id : uuid.UUID
            The ID of the run containing the subrun.
        subrun_id : uuid.UUID
            The ID of the subrun to get logs from.
        environment_step_id : uuid.UUID
            The ID of the environment step to retrieve logs for.

        Returns
        -------
        LogPartsResponse
            The logs of the requested environment step.
        """
        template = (
            "/project/{}/job/{}/run/{}/subrun/{}/environment-step/{}/log-part"
        )
        endpoint = template.format(
            project_id, job_id, run_id, subrun_id, environment_step_id
        )
        return self._get_logs(endpoint)

    def _get_logs(self, endpoint):
        response = self._get(endpoint, _LogPartsResponseSchema())
        return response.log_parts


class _LogPartSchema(BaseSchema):
    log_part_number = fields.Integer(data_key="logPartNumber", required=True)
    line_number = fields.Integer(data_key="lineNumber", required=True)
    content = fields.String(required=True)
    timestamp = fields.DateTime(required=True)

    @post_load
    def make_log_part(self, data, **kwargs):
        return LogPart(**data)


class _LogPartsResponseSchema(BaseSchema):
    log_parts = fields.Nested(
        _LogPartSchema, data_key="logParts", many=True, required=True
    )

    @post_load
    def make_log_parts_response(self, data, **kwargs):
        return LogPartsResponse(**data)
