import argparse
import operator
import sys

from app.app import create_app
from app.default_values import DEFAULT_ARGUMENTS as DEFARGS


def get_args():
    parser = argparse.ArgumentParser(
        description="LibreTranslate - Free and Open Source Translation API"
    )
    parser.add_argument(
        "--host", type=str, help="Hostname (%(default)s)", default=DEFARGS['HOST']
    )
    parser.add_argument("--port", type=int, help="Port (%(default)s)", default=DEFARGS['PORT'])
    parser.add_argument(
        "--char-limit",
        default=DEFARGS['CHAR_LIMIT'],
        type=int,
        metavar="<number of characters>",
        help="Set character limit (%(default)s)",
    )
    parser.add_argument(
        "--req-limit",
        default=DEFARGS['REQ_LIMIT'],
        type=int,
        metavar="<number>",
        help="Set the default maximum number of requests per minute per client (%(default)s)",
    )
    parser.add_argument(
        "--req-limit-storage",
        default=DEFARGS['REQ_LIMIT_STORAGE'],
        type=str,
        metavar="<Storage URI>",
        help="Storage URI to use for request limit data storage. See https://flask-limiter.readthedocs.io/en/stable/configuration.html. (%(default)s)",
    )
    parser.add_argument(
        "--daily-req-limit",
        default=DEFARGS['DAILY_REQ_LIMIT'],
        type=int,
        metavar="<number>",
        help="Set the default maximum number of requests per day per client, in addition to req-limit. (%(default)s)",
    )
    parser.add_argument(
        "--req-flood-threshold",
        default=DEFARGS['REQ_FLOOD_THRESHOLD'],
        type=int,
        metavar="<number>",
        help="Set the maximum number of request limit offences that a client can exceed before being banned. (%(default)s)",
    )
    parser.add_argument(
        "--batch-limit",
        default=DEFARGS['BATCH_LIMIT'],
        type=int,
        metavar="<number of texts>",
        help="Set maximum number of texts to translate in a batch request (%(default)s)",
    )
    parser.add_argument(
        "--ga-id",
        type=str,
        default=DEFARGS['GA_ID'],
        metavar="<GA ID>",
        help="Enable Google Analytics on the API client page by providing an ID (%(default)s)",
    )
    parser.add_argument(
        "--debug", default=DEFARGS['DEBUG'], action="store_true", help="Enable debug environment"
    )
    parser.add_argument(
        "--ssl", default=DEFARGS['SSL'], action="store_true", help="Whether to enable SSL"
    )
    parser.add_argument(
        "--frontend-language-source",
        type=str,
        default=DEFARGS['FRONTEND_LANGUAGE_SOURCE'],
        metavar="<language code>",
        help="Set frontend default language - source (%(default)s)",
    )
    parser.add_argument(
        "--frontend-language-target",
        type=str,
        default=DEFARGS['FRONTEND_LANGUAGE_TARGET'],
        metavar="<language code>",
        help="Set frontend default language - target (%(default)s)",
    )
    parser.add_argument(
        "--frontend-timeout",
        type=int,
        default=DEFARGS['FRONTEND_TIMEOUT'],
        metavar="<milliseconds>",
        help="Set frontend translation timeout (%(default)s)",
    )
    parser.add_argument(
        "--api-keys",
        default=DEFARGS['API_KEYS'],
        action="store_true",
        help="Enable API keys database for per-user rate limits lookup",
    )
    parser.add_argument(
        "--api-keys-db-path",
        default=DEFARGS['API_KEYS_DB_PATH'],
        type=str,
        help="Use a specific path inside the container for the local database. Can be absolute or relative (%(default)s)",
    )
    parser.add_argument(
        "--api-keys-remote",
        default=DEFARGS['API_KEYS_REMOTE'],
        type=str,
        help="Use this remote endpoint to query for valid API keys instead of using the local database",
    )
    parser.add_argument(
        "--get-api-key-link",
        default=DEFARGS['GET_API_KEY_LINK'],
        type=str,
        help="Show a link in the UI where to direct users to get an API key",
    )
    parser.add_argument(
        "--require-api-key-origin",
        type=str,
        default=DEFARGS['REQUIRE_API_KEY_ORIGIN'],
        help="Require use of an API key for programmatic access to the API, unless the request origin matches this domain",
    )
    parser.add_argument(
        "--load-only",
        type=operator.methodcaller("split", ","),
        default=DEFARGS['LOAD_ONLY'],
        metavar="<comma-separated language codes>",
        help="Set available languages (ar,de,en,es,fr,ga,hi,it,ja,ko,pt,ru,zh)",
    )
    parser.add_argument(
        "--threads",
        default=DEFARGS['THREADS'],
        type=int,
        metavar="<number of threads>",
        help="Set number of threads (%(default)s)",
    )
    parser.add_argument(
        "--suggestions", default=DEFARGS['SUGGESTIONS'], action="store_true", help="Allow user suggestions"
    )
    parser.add_argument(
        "--disable-files-translation", default=DEFARGS['DISABLE_FILES_TRANSLATION'], action="store_true",
        help="Disable files translation"
    )
    parser.add_argument(
        "--disable-web-ui", default=DEFARGS['DISABLE_WEB_UI'], action="store_true", help="Disable web ui"
    )
    parser.add_argument(
        "--update-models", default=DEFARGS['UPDATE_MODELS'], action="store_true", help="Update language models at startup"
    )
    return parser.parse_args()


def main():
    args = get_args()
    app = create_app(args)

    if sys.argv[0] == '--wsgi':
        return app
    else:
        if args.debug:
            app.run(host=args.host, port=args.port)
        else:
            from waitress import serve

            url_scheme = "https" if args.ssl else "http"
            print("Running on %s://%s:%s" % (url_scheme, args.host, args.port))

            serve(
                app,
                host=args.host,
                port=args.port,
                url_scheme=url_scheme,
                threads=args.threads
            )


if __name__ == "__main__":
    main()
