import pytest

import numpy as np

from suncal import dataset

def test_dsetsummary():
    ''' Test DataSet summary '''
    # DataSet from array data, all same length
    np.random.seed(100)
    x = np.arange(10)
    ynom = 3*x + 2
    samples = 8
    y = np.random.normal(loc=ynom, scale=.4, size=(samples, len(ynom))).T
    dset = dataset.DataSet(y, colnames=x)
    arr = dset.summarize()

    assert np.allclose(arr._pcolnames, x)
    assert np.allclose(arr._means(), y.mean(axis=1))
    assert np.allclose(arr._stds(), y.std(axis=1, ddof=1))


def test_anova():
    ''' Test analysis of variance in DataSet class '''
    # Data from Dilip Shaw's example Excel spreadsheet at NCSLI ABQ Section Meeting 6/4/2018
    x1 = [0.999156418,0.999474238,0.999339856,1.000160754,1.000744245,1.000737791,1.000495481,0.9995736,1.000723728,1.00060071]
    x2 = [1.000349612,0.999883249,0.999050897,1.00017474,0.999602596,1.000824172,0.999356241,0.999806955,1.000643518,1.000444615]
    x3 = [0.999757491,0.999817772,1.000194482,0.999679647,1.000937242,1.000470251,0.999027869,0.999911651,0.999590441,1.000451023]
    x4 = [0.999719098,0.99964781,1.000822782,0.999409639,0.999109266,1.000785761,0.999307039,1.000469276,1.000654864,1.000987112]
    dat = np.vstack((x1, x2, x3, x4))
    
    g = dataset.DataSet(dat)
    result = g.anova()
    gstats = g.group_stats()
    pstats = g.pooled_stats()
    assert np.isclose(result.F, .0850416)
    assert np.isclose(result.Fcrit, 2.866266)
    assert np.isclose(result.P, .96777478)
    assert np.isclose(pstats.reproducibility, 57.6E-6, atol=.01E-5)
    assert np.isclose(pstats.poolstd, 624.4E-6, atol=.1E-6)
    assert np.isclose(pstats.allstd, 602.1E-6, atol=.1E-6)
    assert np.allclose(gstats.df, [9,9,9,9])
    assert np.allclose(gstats.mean, [1.000100682,1.000013659,0.999983787,1.000091265])
    assert np.allclose(gstats.stdev, [0.000646229,0.00057407,0.000542976,0.000719539])
    assert pstats.reproducibilitydf == 3
    assert pstats.pooldf == 36
    assert pstats.alldf == 39

    # Also try groups of uneven lengths, use excel anova function to verify
    dat[1, 8:] = np.nan
    dat[2, 9:] = np.nan
    dat[3, 6:] = np.nan
    g = dataset.DataSet(dat)
    gstats = g.group_stats()
    pstats = g.pooled_stats()
    assert np.allclose(gstats.df, [9,7,8,5])
    assert pstats.reproducibilitydf == 3
    assert pstats.pooldf == 29
    result = g.anova()
    assert np.isclose(result.F, .23239)
    assert np.isclose(result.Fcrit, 2.93403)
    assert np.isclose(result.P, .873061)


def test_anovameans():
    ''' Test ANOVA when only means and standard deviations of each group are given. Use GUM H.5 data. '''
    x = np.arange(1, 11)
    y = np.array([10.000172, 10.000116, 10.000013, 10.000144, 10.000106, 10.000031, 10.000060, 10.000125, 10.000163, 10.000041])
    ystd = np.array([60, 77, 111, 101, 67, 93, 80, 73, 88, 86]) * 1E-6
    nmeas = np.full(len(y), 5)
    a = dataset.DataSetSummary(x, y, ystd, nmeas)
    gstats = a.group_stats()
    pstats = a.pooled_stats()
    assert np.isclose(pstats.mean, 10.000097, atol=.0000005)  # From GUM Table H.9
    assert np.isclose(pstats.reproducibility, 57E-6, atol=.5E-6)
    assert np.isclose(pstats.poolstd, 85E-6, atol=.5E-6)
    result = a.anova()
    assert np.isclose(result.F, 2.25, atol=.02)  # Equation H.27
    assert a.ncolumns() == 10

    # Data from Glantz Table 7-4, results in table 7-5
    x = ['Healthy', 'Nonmelancholic', 'Melancholoc']  # Check with x as strings too
    y = np.array([9.2, 10.7, 13.5])
    ystd = np.array([2.9, 2.8, 4.7])
    nmeas = np.array([16, 22, 18])
    a = dataset.DataSetSummary(x, y, ystd, nmeas)
    result = a.anova()
    assert np.isclose(result.SSbet, 164.67, atol=.01)
    assert np.isclose(result.SSwit, 666.02, atol=1)  # Rounding error in table, large atol
    assert np.isclose(result.MSbet, 82.3, atol=.1)
    assert np.isclose(result.MSwit, 12.5, atol=.1)
    assert np.isclose(result.F, 6.612, atol=.1)
    assert np.isclose(result.P, .003, atol=.001)


def test_autocorrelation():
    ''' Test autocorrelation calculation using data from NIST handbook example at
        https://www.itl.nist.gov/div898/handbook/eda/section3/eda35c.htm
    '''
    # Data from "LEW.DAT" https://www.itl.nist.gov/div898/handbook/eda/section4/eda4251.htm
    lewdata = np.array([-213, -564,   -35,   -15,   141,   115,  -420,  -360,   203,  -338,  -431,   194,  -220,  -513,
   154,  -125,  -559,    92,   -21,  -579,   -52,    99,  -543,  -175,   162,  -457,  -346,   204,  -300,  -474,   164,
  -107,  -572,    -8,    83,  -541,  -224,   180,  -420,  -374,   201,  -236,  -531,    83,    27,  -564,  -112,   131,
  -507,  -254,   199,  -311,  -495,   143,   -46,  -579,   -90,   136,  -472,  -338,   202,  -287,  -477,   169,  -124,
  -568,    17,    48,  -568,  -135,   162,  -430,  -422,   172,   -74,  -577,   -13,    92,  -534,  -243,   194,  -355,
  -465,   156,   -81,  -578,   -64,   139,  -449,  -384,   193,  -198,  -538,   110,   -44,  -577,    -6,    66,  -552,
  -164,   161,  -460,  -344,   205,  -281,  -504,   134,   -28,  -576,  -118,   156,  -437,  -381,   200,  -220,  -540,
    83,    11,  -568,  -160,   172,  -414,  -408,   188,  -125,  -572,   -32,   139,  -492,  -321,   205,  -262,  -504,
   142,   -83,  -574,     0,    48,  -571,  -106,   137,  -501,  -266,   190,  -391,  -406,   194,  -186,  -553,    83,
   -13,  -577,   -49,   103,  -515,  -280,   201,   300,  -506,   131,   -45,  -578,   -80,   138,  -462,  -361,   201,
  -211,  -554,    32,    74,  -533,  -235,   187,  -372,  -442,   182,  -147,  -566,    25,    68,  -535,  -244,   194,
  -351,  -463,   174,  -125,  -570,    15,    72,  -550,  -190,   172,  -424,  -385,   198,  -218,  -536,    96])

    # Suncal Autocorrelation calculation
    d = dataset.DataSet(lewdata)
    acorr = d.autocorrelation()[:50]  # Compare first 50 lag values

    # NIST's autocorrelation values
    nistacorr = np.array([1.00,-0.31,-0.74, 0.77, 0.21,-0.90, 0.38, 0.63,-0.77,-0.12, 0.82,-0.40,-0.55, 0.73, 0.07,-0.76,
                          0.40, 0.48,-0.70,-0.03, 0.70,-0.41,-0.43,0.67, 0.00,-0.66, 0.42, 0.39,-0.65, 0.03, 0.63,-0.42,
                          -0.36, 0.64,-0.05,-0.60, 0.43, 0.32,-0.64, 0.08, 0.58,-0.45,-0.28, 0.62,-0.10,-0.55, 0.45, 0.25,-0.61, 0.14])

    assert np.allclose(acorr, nistacorr, atol=.006)


def test_autocorrelation2():
    ''' Test autocorrelation against data/results in Figure 1 and Figure 3 of Zhang, Metrologia 43 (2006) '''
    # Data extracted from plot using PlotDigitizer...
    zhang1 = np.array([451.23105,451.30222,450.40115,452.597,450.4439,451.4304,449.6235,448.7272,451.74823,451.1033,449.70428,450.88522,447.9259,450.25928,
    449.52893,450.82843,448.6896,452.02362,448.02097,448.77032,449.8896,448.3388,449.202,450.22168,448.87958,450.48734,450.63913,451.30786,447.92648,
    449.01254,449.73343,449.7335,449.99435,449.08856,450.82437,449.44434,451.9105,448.77097,449.69104,448.58606,449.52988,451.58817,451.89648,450.1987,
    448.87555,449.87625,448.59583,450.7774,449.2029,449.3879,449.00854,451.138,450.20374,450.71597,448.67197,451.02432,448.62463,450.74457,452.29068,
    450.9628,450.92966,449.93848,449.49747,449.26038,450.44604,450.9725,450.4366,450.5742,449.73953,450.45096,449.41238,450.74982,448.94766,449.69705,
    451.29056,448.88614,449.71613,450.08606,450.74057,450.4466,451.3809,450.79285,448.56863,449.89658,449.8255,450.84042,449.30386,449.5932,449.2992,
    448.8724,450.21933,452.11163,450.56088,450.16254,449.25198,450.66058,449.0671,449.70264,449.6268,448.64038,449.6648,449.8925,451.1825,451.48605,
    451.40546,450.16296,451.22534,450.71317,449.69357,452.37314,452.82373,452.08868,450.91257,451.3821,452.15045,452.39233,452.71484,451.59094,453.20816,
    450.58557,452.9095,452.34512,452.61075,451.05997,452.45908,451.90897,451.5865,452.8196,451.40634,451.06018,449.90305,450.82312,448.81705,450.56235,
    451.3639,450.22095,448.73184,450.55777,450.45346,450.96094,450.00772,448.63242,450.15482,451.46378,451.1935,451.32632,451.14142,450.33997,451.4213,
    451.80072,451.41663,451.59686,452.69244,451.45465,449.36325,451.616,451.26508,451.37894,449.77597,453.38983,453.11008,450.76254,449.14062,450.71994,
    450.6441,451.6543,450.17938,450.02765,449.7289,450.815,450.37872,449.98987,450.38354,450.57327,450.99063,449.56317,450.6872,450.6303,450.82007,
    449.73407,450.88654,450.18466,452.17184,450.25586,451.32297,450.23697,452.4233,453.15845,451.41797,451.32315,451.04813,450.517,452.71283,
    453.81314,454.77118,455.48257,455.0084,456.36478,455.64395,457.10944,455.93805,456.99567,456.51196,457.55063,456.9863,454.91385,457.32782,
    456.09955,457.17615,455.94788,458.15793,454.8287,456.2752,456.61197,455.0849,453.21637,453.59106,453.42038,452.14465,450.57016,450.37576,449.70236,
    450.0723,449.2661,449.14285,449.63135,449.78314,449.53183,449.84015,450.35712,450.00146,452.3348,449.4846,450.1628,452.53885,452.5484,450.26248,
    450.36212,452.17856,451.46722,453.7247,451.15903,450.2959,450.40503,452.17877,450.5142,449.7981,448.63147,449.59897,449.51364,450.35785,450.2915,
    449.39044,449.28616,449.14865,449.2388,449.9265,450.1115,450.29175,449.846,451.3731,450.60013,450.23022,448.70792,449.9742,450.26355,452.45465,
    452.29816,450.8944,450.31586,452.1892,451.82407,452.73465,454.05313,452.79163,452.49765,454.50375,451.27414,452.42184,453.61227,449.59064,450.43484,
    451.61102,449.28247,449.70935,448.6328,451.4784,450.22165,450.11734,449.57196,448.47177,448.80853,449.463,449.1216,448.21106,451.87234,449.94217,
    448.012,449.16446,452.31354,452.41318,453.6178,450.89087,453.15787,452.1003,450.73923,451.57394,450.58752,449.99475,451.53137,451.38913,451.67374,
    450.8296,449.37363,450.83914,451.32764,452.03906,451.49374,451.37045,450.81082,450.75397,450.67816,451.65515,452.92145,450.7162,451.48926,450.9866,
    451.9446,449.3979,449.50702,448.13644,447.335,449.19412,449.4455,448.56818,448.1746,447.75253,447.47275,449.89148,449.97687,448.38342,448.70593,
    447.68158,448.5495,448.01364,449.01437,449.16617,449.02866,450.4752,448.23672,449.32755,449.09045,448.35538,448.0045,448.57364,447.81958,448.8962,
    449.84,448.54056,447.31705,449.32315,448.60233,447.57797,448.08548,447.63495,449.64108,448.52188,448.18045,446.99484,448.2422,447.92447,448.72125,
    447.94824,446.92862,446.98083,449.99713,447.9816,449.4803,448.05753,448.56027,448.1667,448.6789,448.5699,449.54688,448.56046,448.38977,448.51312,
    450.89392,450.82755,450.74698,452.18872,449.33374,450.80872,449.66107,451.1218,450.09744,451.57242,452.336,449.46207,])
    d = dataset.DataSet(zhang1)
    acorr = d.autocorrelation_uncert()
    # Compare with Zhang's results (Section 4)
    assert acorr.nc == 13
    assert np.isclose(acorr.uncert, 0.378, atol=.005)  # Relaxed tolerance since data points are estimated from image.
    assert np.isclose(acorr.r_unc, 3.9, atol=.05)

    zhang3 = np.array([-19.518744,-19.498219,-19.478113,-19.492563,-19.527327,-19.501986,-19.492142,-19.469103,-19.497375,-19.467844,-19.521456,-19.526062,
    -19.438942,-19.499044,-19.501556,-19.507002,-19.439775,-19.468466,-19.46281,-19.438934,-19.45841,-19.47244,-19.440607,-19.512857,-19.520187,
    -19.505316,-19.432436,-19.43055,-19.439344,-19.442696,-19.5095,-19.424263,-19.501541,-19.439968,-19.512638,-19.448973,-19.461536,-19.458603,
    -19.453577,-19.44122,-19.492317,-19.392002,-19.419437,-19.440588,-19.451895,-19.443518,-19.46069,-19.454405,-19.447702,-19.454823,-19.496706,
    -19.475134,-19.493563,-19.504034,-19.481205,-19.503403,-19.454397,-19.483925,-19.5034,-19.50633,-19.45921,-19.491669,-19.479523,-19.527689,
    -19.468212,-19.458576,-19.4502,-19.505068,-19.497736,-19.453337,-19.463388,-19.497734,-19.496058,-19.482025,-19.528097,-19.513645,-19.513645,
    -19.520346,-19.484533,-19.505894,-19.500029,-19.450394,-19.511335,-19.46233,-19.504423,-19.451647,-19.52222,-19.459602,-19.513634,-19.462114,
    -19.523474,-19.516771,-19.450802,-19.450592,-19.488077,-19.464203,-19.496452,-19.429646,-19.517393,-19.461685,-19.509644,-19.527023,-19.467337,
    -19.45896,-19.479063,-19.496862,-19.450579,-19.452253,-19.449112,-19.47152,-19.488062,-19.44136,-19.432981,-19.434446,-19.510468,-19.481146,
    -19.469418,-19.491825,-19.469206,-19.464808,-19.47046,-19.43444,-19.527632,-19.453495,-19.430876,-19.417473,-19.448675,-19.502914,-19.535585,
    -19.467312,-19.43087,-19.450346,-19.441969,-19.467726,-19.473589,-19.4294,-19.4294,-19.480078,-19.542067,-19.452852,-19.44552,-19.470442,
    -19.450754,-19.450544,-19.532637,-19.540384,-19.511484,-19.45766,-19.45766,-19.530119,-19.401114,-19.474411,-19.436714,-19.378702,-19.428335,
    -19.53828,-19.408018,-19.433777,-19.426865,-19.427074,-19.414927,-19.423931,-19.450945,-19.413248,-19.440891,-19.43712,-19.44403,-19.444239,
    -19.444447,-19.551252,-19.474812,-19.447376,-19.44612,-19.446745,-19.406326,-19.392294,-19.401089,-19.42224,-19.37826,-19.445902,-19.473127,
    -19.444435,-19.464539,-19.465166,-19.420767,-19.43417,-19.428722,-19.483591,-19.4438,-19.467464,-19.447567,-19.414268,-19.391022,-19.44191,
    -19.426413,-19.405258,-19.441908,-19.410912,-19.412376,-19.431433,-19.457819,-19.47227,-19.462006,-19.458027,-19.452162,-19.462631,-19.500956,
    -19.431425,-19.44755,-19.438963,-19.415297,-19.394564,-19.434563,-19.413828,-19.447964,-19.43016,-19.438955])
    d = dataset.DataSet(zhang3)
    acorr = d.autocorrelation_uncert()
    # Compare with Zhang's results (Section 4)
    assert acorr.nc == 17
    assert np.isclose(acorr.uncert, 0.0067, atol=.0005)
    assert np.isclose(acorr.r_unc, 2.8, atol=.05)

