"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Table = exports.TableEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const glue_generated_1 = require("./glue.generated");
/**
 * (experimental) Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 * @experimental
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    TableEncryption["KMS"] = "SSE-KMS";
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * (experimental) A Glue table.
 *
 * @experimental
 */
class Table extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.tableName,
        });
        /**
         * Partition indexes must be created one at a time. To avoid
         * race conditions, we store the resource and add dependencies
         * each time a new partition index is created.
         */
        this.partitionIndexCustomResources = [];
        jsiiDeprecationWarnings._aws_cdk_aws_glue_TableProps(props);
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = (_b = props.s3Prefix) !== null && _b !== void 0 ? _b : '';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = (_c = props.compressed) !== null && _c !== void 0 ? _c : false;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new glue_generated_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    classification: (_d = props.dataFormat.classificationString) === null || _d === void 0 ? void 0 : _d.value,
                    has_encrypted_data: this.encryption !== TableEncryption.UNENCRYPTED,
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: (_e = props.storedAsSubDirectories) !== null && _e !== void 0 ? _e : false,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className,
                    },
                },
                tableType: 'EXTERNAL_TABLE',
            },
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`,
        });
        this.node.defaultChild = tableResource;
        // Partition index creation relies on created table.
        if (props.partitionIndexes) {
            this.partitionIndexes = props.partitionIndexes;
            this.partitionIndexes.forEach((index) => this.addPartitionIndex(index));
        }
    }
    /**
     * @experimental
     */
    static fromTableArn(scope, id, tableArn) {
        const tableName = core_1.Fn.select(1, core_1.Fn.split('/', core_1.Stack.of(scope).splitArn(tableArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName,
        });
    }
    /**
     * (experimental) Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes.
     * @experimental
     */
    static fromTableAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_glue_TableAttributes(attrs);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Add a partition index to the table.
     *
     * You can have a maximum of 3 partition
     * indexes to a table. Partition index keys must be a subset of the table's
     * partition keys.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/partition-indexes.html
     * @experimental
     */
    addPartitionIndex(index) {
        var _b;
        jsiiDeprecationWarnings._aws_cdk_aws_glue_PartitionIndex(index);
        const numPartitions = this.partitionIndexCustomResources.length;
        if (numPartitions >= 3) {
            throw new Error('Maximum number of partition indexes allowed is 3');
        }
        this.validatePartitionIndex(index);
        const indexName = (_b = index.indexName) !== null && _b !== void 0 ? _b : this.generateIndexName(index.keyNames);
        const partitionIndexCustomResource = new cr.AwsCustomResource(this, `partition-index-${indexName}`, {
            onCreate: {
                service: 'Glue',
                action: 'createPartitionIndex',
                parameters: {
                    DatabaseName: this.database.databaseName,
                    TableName: this.tableName,
                    PartitionIndex: {
                        IndexName: indexName,
                        Keys: index.keyNames,
                    },
                },
                physicalResourceId: cr.PhysicalResourceId.of(indexName),
            },
            policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                resources: cr.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        this.grantToUnderlyingResources(partitionIndexCustomResource, ['glue:UpdateTable']);
        // Depend on previous partition index if possible, to avoid race condition
        if (numPartitions > 0) {
            this.partitionIndexCustomResources[numPartitions - 1].node.addDependency(partitionIndexCustomResource);
        }
        this.partitionIndexCustomResources.push(partitionIndexCustomResource);
    }
    generateIndexName(keys) {
        const prefix = keys.join('-') + '-';
        const uniqueId = core_1.Names.uniqueId(this);
        const maxIndexLength = 80; // arbitrarily specified
        const startIndex = Math.max(0, uniqueId.length - (maxIndexLength - prefix.length));
        return prefix + uniqueId.substring(startIndex);
    }
    validatePartitionIndex(index) {
        if (index.indexName !== undefined && (index.indexName.length < 1 || index.indexName.length > 255)) {
            throw new Error(`Index name must be between 1 and 255 characters, but got ${index.indexName.length}`);
        }
        if (!this.partitionKeys || this.partitionKeys.length === 0) {
            throw new Error('The table must have partition keys to create a partition index');
        }
        const keyNames = this.partitionKeys.map(pk => pk.name);
        if (!index.keyNames.every(k => keyNames.includes(k))) {
            throw new Error(`All index keys must also be partition keys. Got ${index.keyNames} but partition key names are ${keyNames}`);
        }
    }
    /**
     * (experimental) Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal.
     * @experimental
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * (experimental) Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal.
     * @experimental
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * (experimental) Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal.
     * @experimental
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * (experimental) Grant the given identity custom permissions.
     *
     * @experimental
     */
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
    /**
     * (experimental) Grant the given identity custom permissions to ALL underlying resources of the table.
     *
     * Permissions will be granted to the catalog, the database, and the table.
     *
     * @experimental
     */
    grantToUnderlyingResources(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [
                this.tableArn,
                this.database.catalogArn,
                this.database.databaseArn,
            ],
            actions,
        });
    }
    getS3PrefixForGrant() {
        return this.s3Prefix + '*';
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-glue.Table", version: "1.138.1" };
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but \'${column.name}\' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey,
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey,
    };
}
const readPermissions = [
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersion',
    'glue:GetTableVersions',
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition',
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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