"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route = exports.RouteType = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * Type of route
 */
var RouteType;
(function (RouteType) {
    /**
     * HTTP route
     */
    RouteType["HTTP"] = "http";
    /**
     * TCP route
     */
    RouteType["TCP"] = "tcp";
})(RouteType = exports.RouteType || (exports.RouteType = {}));
/**
 * Route represents a new or existing route attached to a VirtualRouter and Mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
 */
class Route extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.routeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.weightedTargets = new Array();
        this.virtualRouter = props.virtualRouter;
        const routeType = props.routeType !== undefined ? props.routeType :
            props.prefix !== undefined ? RouteType.HTTP :
                RouteType.TCP;
        if (routeType === RouteType.HTTP) {
            this.httpRoute = this.renderHttpRoute(props);
        }
        else {
            this.tcpRoute = this.renderTcpRoute(props);
        }
        const route = new appmesh_generated_1.CfnRoute(this, 'Resource', {
            routeName: this.physicalName,
            meshName: this.virtualRouter.mesh.meshName,
            virtualRouterName: this.virtualRouter.virtualRouterName,
            spec: {
                tcpRoute: this.tcpRoute,
                httpRoute: this.httpRoute,
            },
        });
        this.routeName = this.getResourceNameAttribute(route.attrRouteName);
        this.routeArn = this.getResourceArnAttribute(route.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter/${props.virtualRouter.virtualRouterName}/route`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing route given an ARN
     */
    static fromRouteArn(scope, id, routeArn) {
        return new ImportedRoute(scope, id, { routeArn });
    }
    /**
     * Import an existing route given its name
     */
    static fromRouteName(scope, id, meshName, virtualRouterName, routeName) {
        return new ImportedRoute(scope, id, { meshName, virtualRouterName, routeName });
    }
    /**
     * Utility method to add weighted route targets to an existing route
     */
    renderWeightedTargets(props) {
        for (const t of props) {
            this.weightedTargets.push({
                virtualNode: t.virtualNode.virtualNodeName,
                weight: t.weight || 1,
            });
        }
        return this.weightedTargets;
    }
    renderHttpRoute(props) {
        return {
            match: {
                prefix: props.prefix || '/',
            },
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
    renderTcpRoute(props) {
        return {
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
}
exports.Route = Route;
/**
 * Represents and imported IRoute
 */
class ImportedRoute extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.routeArn) {
            this.routeArn = props.routeArn;
            this.routeName = cdk.Fn.select(4, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.routeArn).resourceName));
        }
        else if (props.routeName && props.meshName && props.virtualRouterName) {
            this.routeName = props.routeName;
            this.routeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter/${props.virtualRouterName}/route`,
                resourceName: this.routeName,
            });
        }
        else {
            throw new Error('Need either arn or three names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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