"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Represents the Wait state which delays a state machine from continuing for a specified time
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-wait-state.html
 */
class WaitTime {
    constructor(json) {
        this.json = json;
    }
    /**
     * Wait a fixed amount of time.
     */
    static duration(duration) { return new WaitTime({ Seconds: duration.toSeconds() }); }
    /**
     * Wait until the given ISO8601 timestamp
     *
     * @example 2016-03-14T01:59:00Z
     */
    static timestamp(timestamp) { return new WaitTime({ Timestamp: timestamp }); }
    /**
     * Wait for a number of seconds stored in the state object.
     *
     * @example $.waitSeconds
     */
    static secondsPath(path) { return new WaitTime({ SecondsPath: path }); }
    /**
     * Wait until a timestamp found in the state object.
     *
     * @example $.waitTimestamp
     */
    static timestampPath(path) { return new WaitTime({ TimestampPath: path }); }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.WaitTime = WaitTime;
/**
 * Define a Wait state in the state machine
 *
 * A Wait state can be used to delay execution of the state machine for a while.
 */
class Wait extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.time = props.time;
        this.endStates = [this];
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.WAIT,
            Comment: this.comment,
            ...this.time._json,
            ...this.renderNextEnd(),
        };
    }
}
exports.Wait = Wait;
//# sourceMappingURL=data:application/json;base64,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