# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['manim_physics']

package_data = \
{'': ['*']}

install_requires = \
['manim>=0.6.0', 'pymunk>=6.0.0,<7.0.0']

entry_points = \
{'manim.plugins': ['manim_physics = manim_physics']}

setup_kwargs = {
    'name': 'manim-physics',
    'version': '0.2.2',
    'description': 'Support physics simulation',
    'long_description': "# manim-physics (Under Active Development)\n## Introduction\nThis is a 2D physics simulation plugin that allows you to generate complicated scenes in various branches of Physics such as rigid mechanics, electromagnetism, wave etc.\n\nContributors: \n- [**pdcxs**](https://github.com/pdcxs)\n- [**Matheart**](https://github.com/Matheart)\n- [**Iced-Tea3**](https://github.com/Iced-Tea3)\n## Contents\n- [manim-physics (Under Active Development)](#manim-physics-under-active-development)\n  - [Introduction](#introduction)\n  - [Contents](#contents)\n- [Installation](#installation)\n- [Usage](#usage)\n  - [Rigid Mechanics](#rigid-mechanics)\n  - [Electromagnetism](#electromagnetism)\n  - [Waves](#waves)\n  - [Pendulums](#pendulums)\n- [Contribution Guidelines](#contribution-guidelines)\n- [Other beautiful animations based on manim-physics](#other-beautiful-animations-based-on-manim-physics)\n- [Changelog](#changelog)\n  - [**v0.2.2 2021.07.06**](#v022-20210706)\n    - [New objects](#new-objects)\n    - [Bugfixes](#bugfixes)\n    - [Improvements](#improvements)\n  - [**v0.2.1 2021.07.03**](#v021-20210703)\n    - [New objects](#new-objects-1)\n    - [Bugfixes](#bugfixes-1)\n    - [Improvements](#improvements-1)\n  - [**v0.2.0 2021.07.01**](#v020-20210701)\n    - [Breaking Changes](#breaking-changes)\n# Installation\n`manim-physics` is a package on pypi, and can be directly installed using pip:\n```\npip install manim-physics\n```\n\n**Warnings: Please do not directly clone the github repo! The repo is still under development and it is not a stable version, download manim-physics through pypi.**\n\n# Usage\nMake sure include these two imports at the top of the .py file\n```py\nfrom manim import *\nfrom manim_physics import *\n```\n## Rigid Mechanics\nMost objects can be made into a rigid body (moves according to gravity and collision) or a static body (stays still within the scene).\n\nTo use this feature, the `SpaceScene` must be used, to access the specific functions of the space.\n\n\n>**NOTE**\n>- This feature utilizes the pymunk package. Although unnecessary, it might make it easier if you knew a few things on how to use it.\n>\n>    [Official Documentation](http://www.pymunk.org/en/latest/pymunk.html)\n>\n>    [Youtube Tutorial](https://youtu.be/pRk---rdrbo)\n>\n>- A low frame rate might cause some objects to pass static objects as they don't register collisions finely enough. Trying to increase the config frame rate might solve the problem.\n\n**Example**\n```py\n# use a SpaceScene to utilize all specific rigid-mechanics methods\nclass TwoObjectsFalling(SpaceScene):\n    def construct(self):\n        circle = Circle().shift(UP)\n        circle.set_fill(RED, 1)\n        circle.shift(DOWN + RIGHT)\n\n        rect = Square().shift(UP)\n        rect.rotate(PI / 4)\n        rect.set_fill(YELLOW_A, 1)\n        rect.shift(UP * 2)\n        rect.scale(0.5)\n\n        ground = Line([-4, -3.5, 0], [4, -3.5, 0])\n        wall1 = Line([-4, -3.5, 0], [-4, 3.5, 0])\n        wall2 = Line([4, -3.5, 0], [4, 3.5, 0])\n        walls = VGroup(ground, wall1, wall2)\n        self.add(walls)\n\n        self.play(\n            DrawBorderThenFill(circle),\n            DrawBorderThenFill(rect),\n        )\n        self.make_rigid_body(rect, circle)  # Mobjects will move with gravity\n        self.make_static_body(walls)  # Mobjects will stay in place\n        self.wait(5)\n        # during wait time, the circle and rect would move according to the simulate updater\n```\n![TwoObjectsFalling](/media/TwoObjectsFalling_ManimCE_v0.8.0.gif)\n## Electromagnetism\nThis section introduces new mobjects:\n- Charge\n- ElectricField\n- Current\n- CurrentMagneticField\n- BarMagnet\n- BarmagneticField\n```py\nclass ElectricFieldExampleScene(Scene):\n    def construct(self):\n        charge1 = Charge(-1, LEFT + DOWN)\n        charge2 = Charge(2, RIGHT + DOWN)\n        charge3 = Charge(-1, UP)\n        field = ElectricField(charge1, charge2, charge3)\n        self.add(charge1, charge2, charge3)\n        self.add(field)\n```\n![ElectricFieldExampleScene](/media/ElectricFieldExampleScene_ManimCE_v0.8.0.png)\n```py\nclass MagnetismExample(Scene):\n    def construct(self):\n        current1 = Current(LEFT * 2.5)\n        current2 = Current(RIGHT * 2.5, direction=IN)\n        field = CurrentMagneticField(current1, current2)\n        self.add(field, current1, current2)\n```\n![MagnetismExample](/media/MagnetismExample_ManimCE_v0.8.0.png)\n```py\nclass BarMagnetExample(Scene):\n    def construct(self):\n        bar1 = BarMagnet().rotate(PI / 2).shift(LEFT * 3.5)\n        bar2 = BarMagnet().rotate(PI / 2).shift(RIGHT * 3.5)\n        self.add(BarMagneticField(bar1, bar2))\n        self.add(bar1, bar2)\n```\n![BarMagnetExample](/media/BarMagnetExample_ManimCE_v0.8.0.png)\n## Waves\nThis section introduces new wave mobjects into manim:\n- LinearWave (3D) \n- RadialWave (3D) \n- StandingWave (2D) \n\n```py\nclass LinearWaveExampleScene(ThreeDScene):\n    def construct(self):\n        self.set_camera_orientation(60 * DEGREES, -45 * DEGREES)\n        wave = LinearWave()\n        self.add(wave)\n        wave.start_wave()\n        self.wait()\n        wave.stop_wave()\n```\n![LinearWaveExampleScene](/media/LinearWaveExampleScene_ManimCE_v0.7.0.gif)\n```py\nclass RadialWaveExampleScene(ThreeDScene):\n    def construct(self):\n        self.set_camera_orientation(60 * DEGREES, -45 * DEGREES)\n        wave = RadialWave(\n            LEFT * 2 + DOWN * 5, # Two source of waves\n            RIGHT * 2 + DOWN * 5,\n            checkerboard_colors=[BLUE_D],\n            stroke_width=0,\n        )\n        self.add(wave)\n        wave.start_wave()\n        self.wait()\n        wave.stop_wave()\n```\n![RadialWaveExampleScene](/media/RadialWaveExampleScene_ManimCE_v0.7.0.gif)\n```py\nclass StandingWaveExample(Scene):\n    def construct(self):\n        wave1 = StandingWave(1)\n        wave2 = StandingWave(2)\n        wave3 = StandingWave(3)\n        wave4 = StandingWave(4)\n        waves = VGroup(wave1, wave2, wave3, wave4)\n        waves.arrange(DOWN).move_to(ORIGIN)\n        self.add(waves)\n        for wave in waves:\n            wave.start_wave()\n        self.wait()\n```\n![StandingWaveExample](/media/StandingWaveExample_ManimCE_v0.7.0.gif)\n## Pendulums\n`MultiPendulum` and `Pendulum` both stem from the [Rigid Mechanics](#rigid-mechanics) feature.\n> **Note**:\n>\n> Sometimes the updaters lag behind the intended target. This may be solved by increasing config frame rate.\n```py\nclass PendulumExample(SpaceScene):\n    def construct(self):\n        pends = VGroup(*[Pendulum(i) for i in np.linspace(1,5,7)])\n        self.add(pends)\n        for p in pends:\n            self.make_rigid_body(p.bobs)\n            p.start_swinging()\n        self.wait(10)\n```\n![MultiPendulumExample](./media/PendulumExample_ManimCE_v0.8.0.gif)\n```py\nclass MultiPendulumExample(SpaceScene):\n    def construct(self):\n        p = MultiPendulum(\n            RIGHT, LEFT # positions of the bobs.\n        )\n        self.add(p)\n        self.make_rigid_body(p.bobs) # make the bobs fall free.\n        p.start_swinging() # attach them to their pivots.\n        self.add(TracedPath(p.bobs[-1].get_center, stroke_color=BLUE))\n        self.wait(10)\n```\n![MultiPendulumExample](./media/MultiPendulumExample_ManimCE_v0.8.0.gif)\n# Contribution Guidelines\nThe manim-physics plugin contains objects that are classified into **several main branches**, now including rigid mechanics simulation, electromagnetism and wave. \n\nIf you want to add more objects to the plugin, The classes of the objects should be placed in the python file of corresponding branch, for example, `wave.py`, and place it under the folder src\\manim_physics. The tests of objects should be named as `test_thefilename.py` such as `test_wave.py`, with some documentation, so the maintainer of this repo could ensure that it runs as expected.\n\n\n# Other beautiful animations based on manim-physics\n\n![Falling formulas](/media/TexFalling_ManimCE_v0.7.0.gif)\n\n# Changelog\n## **v0.2.2 2021.07.06**\n### New objects\n- **Rigid Mechanics**: Pendulum\n### Bugfixes\n- Fix the `__all__` bug, now `rigid_mechanics.py` can run normally.\n### Improvements\n- Rewrite README.md to improve its readability\n\n## **v0.2.1 2021.07.03**\n### New objects\n- **Electromagnetism**: Charge, ElectricField, Current, CurrentMagneticField, BarMagnet, and BarMagnetField\n- **Wave**: LinearWave, RadialWave, StandingWave\n\n### Bugfixes\n- Fix typo\n\n### Improvements\n- Simplify rigid-mechanics\n\n## **v0.2.0 2021.07.01**\n### Breaking Changes\n- Objects in the manim-physics plugin are classified into several **main branches** including rigid mechanics simulation, electromagnetism and wave.\n",
    'author': 'Matheart',
    'author_email': 'waautomationwong@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Matheart/manim-physics',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
