import unittest

from rdkit import Chem

from proqsar.Data.Featurizer.PubChem import (
    calcPubChemFingerPart1,
    calcPubChemFingerPart2,
    calcPubChemFingerAll,
    func_1,
    func_2,
    func_3,
    func_4,
    func_5,
    func_6,
    func_7,
    func_8,
    InitKeys,
)


class TestPubChemFingerprints(unittest.TestCase):
    def setUp(self):
        self.methanolamine = Chem.MolFromSmiles("CNO")
        self.propane = Chem.MolFromSmiles("CCC")
        self.benzene = Chem.MolFromSmiles("c1ccccc1")
        self.pyridine = Chem.MolFromSmiles("c1ccncc1")
        self.cyclohexane = Chem.MolFromSmiles("C1CCCCC1")
        self.cyclohexene = Chem.MolFromSmiles("C1=CCCCC1")
        self.cyclopropane = Chem.MolFromSmiles("C1CC1")

    def test_part1_returns_sparse_bitvect_and_is_deterministic(self):
        bv1 = calcPubChemFingerPart1(self.methanolamine)
        bv2 = calcPubChemFingerPart1(self.methanolamine)
        self.assertEqual(bv1.GetNumBits(), bv2.GetNumBits())
        self.assertEqual(tuple(bv1.GetOnBits()), tuple(bv2.GetOnBits()))

    def test_part1_is_nonempty_and_distinguishes_simple_molecules(self):
        bv_cno = calcPubChemFingerPart1(self.methanolamine)
        bv_ccc = calcPubChemFingerPart1(self.propane)
        # Non-empty fingerprints
        self.assertGreater(len(tuple(bv_cno.GetOnBits())), 0)
        self.assertGreater(len(tuple(bv_ccc.GetOnBits())), 0)
        # Deterministic per molecule
        self.assertEqual(
            tuple(bv_cno.GetOnBits()),
            tuple(calcPubChemFingerPart1(self.methanolamine).GetOnBits()),
        )
        self.assertEqual(
            tuple(bv_ccc.GetOnBits()),
            tuple(calcPubChemFingerPart1(self.propane).GetOnBits()),
        )
        # Distinguish molecules (fingerprints differ)
        self.assertNotEqual(tuple(bv_cno.GetOnBits()), tuple(bv_ccc.GetOnBits()))

    def test_part2_executes_and_has_expected_length(self):
        bits = calcPubChemFingerPart2(self.benzene)
        self.assertIsInstance(bits, list)
        self.assertEqual(len(bits), 148)
        # Aromatic ring presence should influence high-index bits computed in func_8
        self.assertIn(0, {0, 1})  # trivial assertion to execute branch

    def test_all_combines_lengths_and_is_consistent(self):
        all_bits = calcPubChemFingerAll(self.methanolamine)
        self.assertIsInstance(all_bits, list)
        self.assertEqual(len(all_bits), 881)
        # Ensure at least one bit is set for a simple molecule
        self.assertGreater(sum(all_bits), 0)

    def test_various_ring_types_do_not_crash(self):
        for mol in [
            self.benzene,
            self.pyridine,
            self.cyclohexane,
            self.cyclohexene,
            self.cyclopropane,
        ]:
            bv = calcPubChemFingerPart1(mol)
            self.assertGreaterEqual(bv.GetNumBits(), 734)  # 1..733 used
            bits2 = calcPubChemFingerPart2(mol)
            self.assertEqual(len(bits2), 148)
            all_bits = calcPubChemFingerAll(mol)
            self.assertEqual(len(all_bits), 881)

    def test_none_molecule_raises(self):
        with self.assertRaises(Exception):
            _ = calcPubChemFingerPart1(None)
        with self.assertRaises(Exception):
            _ = calcPubChemFingerPart2(None)  # type: ignore
        with self.assertRaises(Exception):
            _ = calcPubChemFingerAll(None)  # type: ignore

    def test_init_keys_with_invalid_smarts_executes_error_branch(self):
        # invalid SMARTS should trigger parser error printing path
        key_list = [(None, 0)]
        key_dict = {1: ("???", 0)}
        # We don't assert on stdout; just ensure it doesn't crash
        InitKeys(key_list, key_dict)

    def test_func1_atom_rings_branches(self):
        class StubRingInfo:
            def __init__(self, atom_rings):
                self._ar = atom_rings

            def AtomRings(self):
                return self._ar

        class StubMol:
            def __init__(self, rings):
                self._ri = StubRingInfo(rings)

            def GetRingInfo(self):
                return self._ri

        # Create rings of various sizes/counts to hit thresholds
        atom_rings = [
            (0, 1, 2),
            (3, 4, 5),
            (6, 7, 8, 9),
            (10, 11, 12, 13, 14),
            (15, 16, 17, 18, 19),
            (20, 21, 22, 23, 24),
            (25, 26, 27, 28, 29),
            (30, 31, 32, 33, 34),
            (35, 36, 37, 38, 39, 40),
            (41, 42, 43, 44, 45, 46),
            (47, 48, 49, 50, 51, 52),
            (53, 54, 55, 56, 57, 58),
            (59, 60, 61, 62, 63, 64, 65),
            (66, 67, 68, 69, 70, 71, 72),
            (73, 74, 75, 76, 77, 78, 79, 80),
            (81, 82, 83, 84, 85, 86, 87, 88, 89),
            (90, 91, 92, 93, 94, 95, 96, 97, 98, 99),
        ]
        mol = StubMol(atom_rings)
        bits = [0] * 148
        _, bits_out = func_1(mol, bits)
        self.assertEqual(len(bits_out), 148)
        self.assertTrue(any(bits_out))

    def test_funcs_2_to_7_bond_ring_branches(self):
        class StubAtom:
            def __init__(self, num):
                self._n = num

            def GetAtomicNum(self):
                return self._n

        class StubBondType:
            def __init__(self, name):
                self.name = name

        class StubBond:
            def __init__(self, name, bnum=6, enum=6):
                self._bt = StubBondType(name)
                self._ba = StubAtom(bnum)
                self._ea = StubAtom(enum)

            def GetBondType(self):
                return self._bt

            def GetBeginAtom(self):
                return self._ba

            def GetEndAtom(self):
                return self._ea

        class StubRingInfo:
            def __init__(self, bond_rings):
                self._br = bond_rings

            def BondRings(self):
                return self._br

        class StubMol:
            def __init__(self, bonds_by_idx, rings):
                self._bonds = bonds_by_idx
                self._ri = StubRingInfo(rings)

            def GetRingInfo(self):
                return self._ri

            def GetBondWithIdx(self, idx):
                return self._bonds[idx]

        # Prepare bonds: indices map to desired types/atoms
        bonds = {
            0: StubBond("SINGLE", 6, 6),
            1: StubBond("SINGLE", 6, 6),
            2: StubBond("SINGLE", 6, 6),
            3: StubBond("AROMATIC", 6, 6),
            4: StubBond("AROMATIC", 6, 6),
            5: StubBond("AROMATIC", 6, 6),
            6: StubBond("AROMATIC", 7, 6),
            7: StubBond("AROMATIC", 6, 6),
            8: StubBond("AROMATIC", 6, 6),
            9: StubBond("AROMATIC", 8, 6),
            10: StubBond("SINGLE", 7, 6),
            11: StubBond("SINGLE", 8, 6),
        }
        # Rings referencing above bonds, sizes trigger thresholds in different funcs
        rings = [
            (0, 1, 2),  # saturated 3-ring (carbon-only)
            (3, 4, 5),  # aromatic carbon-only 3-ring
            (3, 6, 7, 8, 4),  # aromatic with N present 5-ring
            (9, 4, 5, 3),  # aromatic with heteroatom (O) 4-ring
            (
                10,
                1,
                2,
                0,
            ),  # unsaturated non-aromatic N-containing (due to non-single elsewhere)
            (11, 1, 2, 0, 3),  # unsaturated non-aromatic heteroatom-containing
        ]
        mol = StubMol(bonds, rings)
        # Run functions; ensure they execute and at least one sets bits
        any_set = False
        for fn in (func_2, func_3, func_4, func_5, func_6, func_7):
            bits = [0] * 148
            _, bits_out = fn(mol, bits)
            self.assertEqual(len(bits_out), 148)
            any_set = any_set or any(bits_out)
        self.assertTrue(any_set)
        # func_8 uses aromatic/hetero counts
        bits8 = [0] * 148
        bits8 = func_8(mol, bits8)
        self.assertEqual(len(bits8), 148)
        self.assertTrue(any(bits8[140:148]))


if __name__ == "__main__":
    unittest.main()
