import pandas as pd
import numpy as np
from .pbt_logger import PBT_Logger


def one_sided_welch_t_test(x1_mean, x2_mean, x1_std, x2_std, N, alpha=0.05):
    """Perform a simple one-sided t-test: H0: x1 = x2 vs H1: x1 > x2."""
    std_est = np.sqrt(x1_std ** 2 + x2_std ** 2)
    t_stat = (x1_mean - x2_mean) / std_est
    dof = 2 * N - 2

    # Get percentile of t distribution and compare with statistic
    # TODO: check if alpha has to be divided by 2? Only one tail
    from scipy.stats import t

    critical_value = t.ppf(1.0 - alpha / 2, dof)
    return t_stat > critical_value


class SelectionStrategy(object):
    def __init__(self, strategy: str):
        """Exploitation Strategies for PBT (Jaderberg et al. 17)."""
        assert strategy in ["t_test", "truncation", "binary_tournament"]
        self.strategy = strategy
        self.truncation_percent = 0.2

    def t_test_selection(self, worker_id: int, pbt_log: PBT_Logger):
        """
        Uniformly sample other network in population. Compare performance using
        Welch’s t-test. If sampled network is better/satisfies t-test, copy it.
        """
        copy_info = False
        hyperparams = {}
        ckpt_path = "temp.pt"
        return copy_info, hyperparams, ckpt_path

    def truncation_selection(self, worker_id: int, pbt_log: PBT_Logger):
        """
        Rank networks in population by performance. If network is in bottom 20%
        sample another uniformly from top 20% of population, and copy it.
        """
        most_recent_df = pbt_log.get_most_recent_data()
        top_df, bottom_df = pbt_log.get_truncation_population(
            most_recent_df, self.truncation_percent
        )
        # Check if worker is in top_df - if not sample from top
        bottom_performer = worker_id in bottom_df["worker_id"].tolist()
        if not bottom_performer:
            # Keep data from worker to continue training
            copy_info, hyperparams, ckpt_path = get_no_copy_data(
                most_recent_df, worker_id
            )
        else:
            # Sample worker to copy from top dataframe (20%)
            copy_from_id = top_df["worker_id"].sample(n=1, random_state=1).values[0]
            copy_info, hyperparams, ckpt_path = get_copy_data(
                most_recent_df, copy_from_id, pbt_log.eval_metric
            )
        return copy_info, hyperparams, ckpt_path

    def binary_tournament(self, worker_id: int, pbt_log: PBT_Logger):
        """
        Each member of population randomly selects another member and copies
        all its parameters (hyper & weights) if other member’s score is better.
        """
        df = pbt_log.get_most_recent_data()
        # Only use workers that are already recorded in dataframe
        potential_ids = df["worker_id"].unique().tolist()
        copy_from_id = np.random.choice(potential_ids)
        perf1 = df[df["worker_id"] == worker_id][pbt_log.eval_metric].values[0]
        perf2 = df[df["worker_id"] == copy_from_id][pbt_log.eval_metric].values[0]
        # Check if worker is worse than other randomly sampled member
        bottom_performer = perf1 < perf2 if pbt_log.max_objective else perf1 > perf2
        # Keep data from worker to continue training otw. copy better worker
        if not bottom_performer:
            copy_info, hyperparams, ckpt_path = get_no_copy_data(df, worker_id)
        else:
            copy_info, hyperparams, ckpt_path = get_copy_data(
                df, copy_from_id, pbt_log.eval_metric
            )
        return copy_info, hyperparams, ckpt_path

    def select(self, worker_id: int, pbt_log: PBT_Logger):
        if self.strategy == "t_test":
            raise NotImplementedError
            # Problem: We need to somehow record the std of the performance
            # In original paper: Only RL problem w. 10 episode returns
            # Basically enhanced binary tournament: 1) greater mean 2) test satisfied
            copy_info, hyperparams, ckpt = self.t_test_selection(worker_id, pbt_log)
        elif self.strategy == "truncation":
            copy_info, hyperparams, ckpt = self.truncation_selection(worker_id, pbt_log)
        elif self.strategy == "binary_tournament":
            copy_info, hyperparams, ckpt = self.binary_tournament(worker_id, pbt_log)
        return copy_info, hyperparams, ckpt


def get_copy_data(df: pd.DataFrame, copy_from_id: int, eval_metric: str):
    """Helper to extract data to copy from other worker."""
    hyperparams = df[df["worker_id"] == copy_from_id]["hyperparams"].values[0]
    ckpt_path = df[df["worker_id"] == copy_from_id]["model_ckpt"].values[0]
    pbt_step_id = df[df["worker_id"] == copy_from_id]["pbt_step_id"].values[0]
    num_updates = df[df["worker_id"] == copy_from_id]["num_updates"].values[0]
    performance = df[df["worker_id"] == copy_from_id][eval_metric].values[0]
    copy_info = {}
    copy_info["copy_bool"] = True
    copy_info["copy_from_worker_id"] = copy_from_id
    copy_info["copy_from_pbt_steps"] = pbt_step_id
    copy_info["copy_from_num_updates"] = num_updates
    copy_info["copy_from_ckpt"] = ckpt_path
    copy_info["copy_from_hyperparams"] = hyperparams
    copy_info["copy_from_performance"] = performance
    return copy_info, hyperparams, ckpt_path


def get_no_copy_data(df: pd.DataFrame, worker_id: int):
    """Helper to summarize data from same worker to continue training."""
    copy_info = {}
    copy_info["copy_bool"] = False
    copy_info["copy_from_worker_id"] = None
    copy_info["copy_from_pbt_steps"] = None
    copy_info["copy_from_num_updates"] = None
    copy_info["copy_from_performance"] = None
    # Keep same parameters and continue training last checkpoint
    hyperparams = df[df["worker_id"] == worker_id]["hyperparams"].values[0]
    ckpt_path = df[df["worker_id"] == worker_id]["model_ckpt"].values[0]
    copy_info["copy_from_ckpt"] = ckpt_path
    copy_info["copy_from_hyperparams"] = hyperparams
    return copy_info, hyperparams, ckpt_path
