# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['harborapi', 'harborapi.models']

package_data = \
{'': ['*']}

install_requires = \
['backoff>=2.1.2,<3.0.0',
 'httpx>=0.23.0,<0.24.0',
 'loguru>=0.6.0,<0.7.0',
 'pydantic>=1.9.1,<2.0.0']

setup_kwargs = {
    'name': 'harborapi',
    'version': '0.1.1',
    'description': 'Async Harbor API v2.0 wrapper',
    'long_description': '# Harbor API\n\nPython async API wrapper for the Harbor v2.0 REST API.\n\n## Features\n\n- Async API\n- Fully typed\n- Data validation with [Pydantic](https://pydantic-docs.helpmanual.io/)\n- HTTP handled by [HTTPX](https://www.python-httpx.org/)\n- Extensive test coverage powered by [Hypothesis](https://hypothesis.works/)\n\n## Implemented endpoints\n\n- [x] user\n- [ ] gc\n- [x] scanAll\n- [ ] configure\n- [ ] usergroup\n- [ ] preheat\n- [ ] replication\n- [ ] label\n- [ ] robot\n- [ ] webhookjob\n- [ ] icon\n- [ ] project\n- [ ] webhook\n- [x] scan\n- [ ] member\n- [ ] ldap\n- [x] registry\n- [x] search\n- [x] artifact\n- [ ] immutable\n- [ ] retention\n- [x] scanner\n- [x] systeminfo**\n- [x] statistic\n- [x] quota\n- [x] repository\n- [x] ping\n- [x] oidc\n- [x] SystemCVEAllowlist\n- [x] Health\n- [ ] robotv1\n- [ ] projectMetadata\n- [x] auditlog\n\n\\*\\* `/systeminfo/getcert` NYI\n\n\n## Usage\n\nThe client can be instatiated with either a username and password, or a base64-encoded [HTTP Basic Access Authentication](https://en.wikipedia.org/wiki/Basic_access_authentication) credential string.\n\n### Username and Password\n\n```py\nfrom harborapi import HarborAsyncClient\n\nclient = HarborAsyncClient(\n    url="https://your-harbor-instance.com/api/v2.0",\n    username="username",\n    secret="secret",\n)\n```\n\n### Basic Access Authentication Credentials\n\n```py\nfrom harborapi import HarborAsyncClient\n\nclient = HarborAsyncClient(\n    url="https://your-harbor-instance.com/api/v2.0",\n    credentials="base64_string_here",\n)\n```\n\n\n\n## Examples\n\n### Get Current User\n\n```py\nimport asyncio\n\nfrom harborapi import HarborAsyncClient\n\nclient = HarborAsyncClient(\n    url="https://your-harbor-instance.com/api/v2.0",\n    username="username",\n    secret="secret",\n)\n\n\nasync def main():\n    res = await client.get_current_user()\n    print(repr(res))\n\n\nasyncio.run(main())\n```\n\nProduces:\n\n```py\nUserResp(\n    email=None,\n    realname=\'Firstname Lastname\',\n    comment=\'from LDAP.\',\n    user_id=123,\n    username=\'firstname-lastname\',\n    sysadmin_flag=False,\n    admin_role_in_auth=True,\n    oidc_user_meta=None,\n    creation_time=datetime.datetime(2022, 7, 1, 13, 19, 36, 26000, tzinfo=datetime.timezone.utc),\n    update_time=datetime.datetime(2022, 7, 1, 13, 19, 36, 26000, tzinfo=datetime.timezone.utc)\n)\n```\n\n### Get Artifacts\n\n```py\nawait client.get_artifacts("project", "repository")\n```\n\nProduces:\n\n```py\n[\n    Artifact(\n        id=1,\n        type=\'IMAGE\',\n        media_type=\'application/vnd.docker.container.image.v1+json\',\n        manifest_media_type=\'application/vnd.docker.distribution.manifest.v2+json\',\n        project_id=1,\n        repository_id=1,\n        digest=\'sha256:f8410cc846de810e23ada839511f04efd998e0e4728d63ea997001f4ead0acac\',\n        size=106474226,\n        icon=\'sha256:0048162a053eef4d4ce3fe7518615bef084403614f8bca43b40ae2e762e11e06\',\n        push_time=datetime.datetime(2022, 7, 4, 8, 18, 46, 891000, tzinfo=datetime.timezone.utc),\n        pull_time=datetime.datetime(2022, 7, 4, 8, 19, 7, 131000, tzinfo=datetime.timezone.utc),\n        extra_attrs=ExtraAttrs(...),\n        annotations=None,\n        references=None,\n        tags=[Tag[...], Tag[...]],\n        labels=None,\n        scan_overview=None,\n        accessories=None,\n    ),\n    Artifact(\n        ...\n    ),\n    ...\n]\n```\n\nPassing `with_scan_overview=True` will also include a `NativeReportSummary` if possible (otherwise `ScanOverview`) along with the artifact if the artifact has a scan report associated with it.\n\n```py\nawait client.get_artifacts("project", "repository", with_scan_overview=True)\n```\n\n```py\nArtifact(\n    ...,\n    scan_overview=NativeReportSummary(\n        report_id=\'a0c40f3b-0403-441b-72e6-38cc725e3bfb\',\n        scan_status=\'Success\',\n        severity=\'Critical\',\n        duration=20,\n        summary=VulnerabilitySummary(\n            total=1179,\n            fixable=394,\n            critical=3,\n            high=50,\n            medium=615,\n            low=511,\n            summary={\'Critical\': 3, \'High\': 50, \'Low\': 511, \'Medium\': 615},\n        ),\n        start_time=datetime.datetime(2022, 7, 4, 8, 18, 58, tzinfo=datetime.timezone.utc),\n        end_time=datetime.datetime(2022, 7, 4, 8, 19, 18, tzinfo=datetime.timezone.utc),\n        complete_percent=100,\n        version=\'v0.29.2\'\n    ),\n)\n```\n\n## Exception Handling (WIP)\n\nAll methods raise exceptions derived from `harborapi.exceptions.StatusError` for responses with non-2xx status codes unless otherwise specified.\n\n### Status Code\n\n```py\ntry:\n    await client.delete_artifact("project", "repository", "latest")\nexcept StatusError as e:\n    print(e.status_code)\n```\n\n### Granular Exception Handling\n\nIf more granular exception handling is required, all documented HTTP exceptions in the API spec are implemented as discrete classes derived from `StatusError`\n\n```py\nfrom harborapi.exceptions import (\n    BadRequest,\n    Forbidden,\n    NotFound,\n    Unauthorized,\n    PreconditionFailed,\n    InternalServerError,\n    StatusError\n)\n\nproject, repo, tag = "testproj", "testrepo", "latest"\n\ntry:\n    await client.delete_artifact(project, repo, tag)\nexcept NotFound as e:\n    print(f"\'{repo}:{tag}\' not found for project \'{project}\'")\nexcept StatusError as e:\n    # catch all other HTTP exceptions\n```\n\n### Inspecting Errors\n\nThe `StatusError.errors` attribute contains a list of `Error` objects that contain\nmore detailed information about the error(s) that have occured.\n\n```py\ntry:\n    await client.delete_artifact("project", "repository", "latest")\nexcept StatusError as e:\n    for error in e.errors:\n        print(error.code, error.message)\n```\n\nAn `Error` object has the following structure\n\n```py\nclass Error(BaseModel):\n    code: Optional[str] = Field(None, description="The error code")\n    message: Optional[str] = Field(None, description="The error message")\n```\n\n(It is likely that `code` and `message` are both not `None` on runtime, but the API specification states that both these fields are optional.)\n\n## Non-Async Client (Blocking)\n\nIn order to support use cases where users do not want to use the async client, a non-async client exists in the form of `HarborClient`.\n\nAll methods should be invoked identically to the async client, with `await` omitted.\n\n**NOTE:** The implementation of `HarborClient` is extremely hacky, and it is _highly_ recommended to use the async client whenever possible.\n\n### Example\n\n```py\nimport asyncio\nfrom harborapi import HarborClient\n\nclient = HarborClient(\n    loop=asyncio.new_event_loop(), # pass a new event loop from main thread\n    url="https://your-harbor-instance.com/api/v2.0",\n    username="username",\n    secret="secret",\n)\n\nres = client.get_current_user()\nprint(res)\n```\n',
    'author': 'Peder Hovdan Andresen',
    'author_email': 'pederhan@usit.uio.no',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
