# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sentinelpy', 'sentinelpy.request']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2.25.0,<3.0.0']

setup_kwargs = {
    'name': 'sentinelpy',
    'version': '0.1.0',
    'description': 'Queries ESA Sentinel APIs for products',
    'long_description': '# sentinelpy\n\n![Python Package](https://github.com/UKHO/sentinelpy/workflows/Python%20Package/badge.svg?branch=main)\n[![Dependabot Status](https://api.dependabot.com/badges/status?host=github&repo=UKHO/sentinelpy&identifier=304287716)](https://dependabot.com)\n\nQueries ESA Sentinel APIs for products\n\n\n* Free software: MIT license\n\n# Usage\n## `query_sentinel_hub`\n\n```python\nimport logging\n\nfrom sentinelpy import (\n    query_sentinel_hub,\n    SentinelProductRequestBuilder,\n    RequestQueryBuilder,\n    PlatformName,\n    QuerySentinelProductsResponse\n)\n\nrequest = (\n    SentinelProductRequestBuilder()\n    .with_username("username")\n    .with_password("password")\n    .with_query(\n        RequestQueryBuilder()\n        .platform_name(PlatformName.SENTINEL_1)\n    )\n    .build()\n)\n\nresult = query_sentinel_hub(request, log_level=logging.DEBUG)\n\nassert isinstance(result, QuerySentinelProductsResponse)\n\nprint(result.status_code)\nprint(result.body)\nprint(result.error)\n```\n\n**Positional arguments:**\n\n* `sentinel_product_request` (_[SentinelProductRequest](#SentinelProductRequest)_)\n\n    Request object containing the details of the query. You should use the _[SentinelProductRequestBuilder](#SentinelProductRequestBuilder)_ to\n    construct\n\n**Keyword arguments:**\n\n* `log_level` (_int_)\n\n    Level for which to log at use `logging` to define correct level, defaults to \'INFO\'.\n\n* `logger` (_logging.Logger_)\n\n    Logger to use to log messages with, defaults to `logging.getLogger(__name__)` if no value supplied\n\n**Returns:** _[QuerySentinelProductsResponse](#QuerySentinelProductsResponse)_ object\n\n## API Documentation\n<details>\n<summary><strong>range_value</strong></summary>\n\n<p>\n\n### range_value (`function`)\n\nA helper function for defining range values for queries using the `RequestQueryBuilder`\n\n**Parameters**:\n\n* `start_val` (_str_)\n\n    Start of the range\n* `end_val` (_str_)\n\n    End of the range\n\n**Returns**: _str_ range string in format of "[MIN TO MAX]" where `MIN` is `start_val` and `MAX` is `end_val`\n\n#### Example\n\n```python\nfrom sentinelpy import (\n    RequestQueryBuilder,\n    range_value\n)\n\nquery = (\n    RequestQueryBuilder()\n    .orbit_number(range_value("1", "2"))\n    .build()\n)\n\nassert query == "orbitnumber:[1 TO 2]"\n```\n</p>\n</details>\n\n---\n\n<details id="SentinelProductRequestBuilder">\n<summary><strong>SentinelProductRequestBuilder</strong></summary>\n\n<p>\n\n### SentinelProductRequestBuilder (`class`)\nBuilder resposible for creating _[SentinelProductRequest](#SentinelProductRequest)_ objects.\n\nAs a minimum, the username and password for the Sentinel Hub should be supplied.\n\n<details>\n<summary><strong>Constructor details</strong></summary>\n\n<p>\n\n* **`default_query`** (_str_): Default value for query, _defaults to `*`_\n* **`default_rows`** (_int_): Default value for rows, _defaults to `30`_\n* **`default_order_by`** (_Optional[str]_): Default value for order by, _defaults to `None`_\n* **`default_start`** (_int_): Default value for start, _defaults to `0`_\n</p>\n</details>\n\n<details>\n<summary><strong>Method details</strong></summary>\n\n<p>\n\n##### `build`\n\nMethod that constructs the _[SentinelProductRequest](#SentinelProductRequest)_ using the values supplied to the builder\n\n**Returns**: _`SentinelProductRequest`_ - Built request from input data\n\n**Raises**: _`ValueError`_ - if username or password missing\n\n---\n\n##### `with_username`\n\nSets the Sentinel Hub username\n\n**Parameter**:\n\n* `username` (_str_)\n\n    The username for Sentinel Hub API\n\n**Returns**: _SentinelProductRequestBuilder_ Builder object with `username` supplied\n\n---\n\n##### `with_password`\n\nSets the Sentinel Hub password\n\n**Parameter**:\n\n* `password` (_str_)\n\n    The associated password for the user for the Sentinel Hub API\n\n**Returns**: _SentinelProductRequestBuilder_ Builder object with `password` supplied\n\n---\n\n##### `with_query`\n\nSets the query (q) value\n\n**Parameter**:\n\n* `query` (_str_ or _RequestQueryBuilder_)\n\n    The query to use to filter results. If it is a _RequestQueryBuilder_, then `build` will\n    call `build` on the _RequestQueryBuilder_ before constructing the _[SentinelProductRequest](#SentinelProductRequest)_.\n\n**Returns**: _SentinelProductRequestBuilder_ Builder object with `query` supplied\n\n---\n\n##### `with_rows`\n\nSets the rows value\n\n**Parameter**:\n\n* `rows` (_int_)\n\n    The value for rows to return in each request\n\n**Returns**: _SentinelProductRequestBuilder_ Builder object with `rows` supplied\n\n---\n\n##### `with_order_by`\n\nSets the order_by value\n\n**Parameter**:\n\n* `order_by` (_str_)\n\n    The value for order_by to return in each request\n\n**Returns**: _SentinelProductRequestBuilder_ Builder object with `order_by` supplied\n\n---\n\n##### `with_start`\n\nSets the start value\n\n**Parameter**:\n\n* `start` (_int_)\n\n    The value for start to return in each request\n\n**Returns**: _SentinelProductRequestBuilder_ Builder object with `start` supplied\n\n\n</p>\n</details>\n\n<details>\n<summary><strong>Example usage</strong></summary>\n\n<p>\n\n```python\nfrom sentinelpy import (\n    SentinelProductRequestBuilder,\n    SentinelProductRequest,\n    RequestQueryBuilder,\n    PlatformName,\n)\n\nminimal = (\n    SentinelProductRequestBuilder()\n    .with_username("username")\n    .with_password("password")\n    .build()\n)\n\nassert minimal == SentinelProductRequest(\n    query="*",\n    rows=None,\n    order_by=None,\n    start=0,\n    username="username",\n    password="password"\n)\n\nfull = (\n    SentinelProductRequestBuilder()\n    .with_username("username")\n    .with_password("password")\n    .with_query(\n        RequestQueryBuilder()\n        .platform_name(PlatformName.SENTINEL_1)\n    )\n    .with_start(15)\n    .with_rows(15)\n    .with_order_by(\'ingestiondate desc\')\n    .build()\n)\n\nassert full == SentinelProductRequest(\n    query="platformname:Sentinel-1",\n    rows=15,\n    order_by=\'ingestiondate desc\',\n    start=15,\n    username="username",\n    password="password"\n)\n```\n</p>\n</details>\n\n</p>\n</details>\n\n---\n\n<details id="RequestQueryBuilder">\n<summary><strong>RequestQueryBuilder</strong></summary>\n\n<p>\n\n### RequestQueryBuilder (`class`)\nA builder utility to build values for queries. [Refer to the Sentinel Hub API documentation for more information about the values](https://scihub.copernicus.eu/twiki/do/view/SciHubUserGuide/FullTextSearch?redirectedfrom=SciHubUserGuide.3FullTextSearch). The methods of this class\nmap to search keywords/operators described in the former documentation. The keywords use snake case rather than\nall lowercase to adhere to Python conventions.\n\n<details>\n<summary><strong>Method details</strong></summary>\n\n<p>\n\n##### `build`\n\nCreates the value for query/`q` using the supplied values. If two non-operators supplied in order without an operator (i.e. `and_`, `or_`, or `not_`)\ndefaults to `and_` operator.\n\nIf the query has `and_` or `or_` operators at the start or an operator at the end then these are removed from the query.\n\n**Returns**: _str_  - the query constructed using the builder, if no methods called returns `*` by default\n\n**Example**\n\n```python\nfrom sentinelpy import RequestQueryBuilder, PlatformName\n\ndefault_build_behaviour = RequestQueryBuilder().build()\n\nassert default_build_behaviour == "*"\n\nhanging_operator_start = RequestQueryBuilder().and_().platform_name(PlatformName.SENTINEL_1).build()\n\nassert hanging_operator_start == "platformname:Sentinel-1"\n\nhanging_operator_end = RequestQueryBuilder().not_().platform_name(PlatformName.SENTINEL_1).and_().build()\n\nassert hanging_operator_end == "NOT platformname:Sentinel-1"\n```\n\n---\n\n##### `and_`\n\nLogical `and` - combines the previous and next clauses i.e. `platform_name(X).and_().platform_name(Y)` results in\nthe query being `platformname:X AND platform_name:Y`\n\n**Returns**: _RequestQueryBuilder_ self\n\n---\n\n##### `or_`\n\nLogical `or` - either the previous or next clauses i.e. `platform_name(X).or_().platform_name(Y)` results in\nthe query being `platformname:X OR platform_name:Y`\n\n**Returns**: _RequestQueryBuilder_ self\n\n---\n\n##### `not_`\n\nNegates the following clause.\n\n**Returns**: _RequestQueryBuilder_ self\n\n---\n\n##### `group_`\n\nCreates a grouped clause in the query. If the parameter is a _RequestQueryBuilder_ then it calls the `build` method\nbefore.\n\n**Parameter**:\n\n* `inner_query` (_str_ or _RequestQueryBuilder_)\n\n    The query that is part of the group.\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Example**\n\n```python\nfrom sentinelpy import (\n    RequestQueryBuilder,\n    PlatformName,\n    PolarisationMode\n)\n\nq = (\n    RequestQueryBuilder()\n    .group_(\n        RequestQueryBuilder()\n        .platform_name(PlatformName.SENTINEL_1)\n        .and_()\n        .polarisation_mode(PolarisationMode.HH)\n    )\n    .or_()\n    .group_(\n        RequestQueryBuilder()\n        .platform_name(PlatformName.SENTINEL_1)\n        .and_()\n        .not_()\n        .polarisation_mode(PolarisationMode.VH)\n    )\n    .build()\n)\nassert q == (\n    "(platformname:Sentinel-1 AND polarisationmode:HH) OR "\n    "(platformname:Sentinel-1 AND NOT polarisationmode:VH)"\n)\n```\n\n---\n\n##### `begin_position`\n\nSets the beginposition filter between the parameters start/end\n\n**Parameters**:\n\n* `begin_position_start` (_str_)\n\n    Start of the range that the query is interested in, in ISO date/time stamp with millis, or relative\n    to NOW (e.g. NOW/NOW-1DAY etc.)\n\n* `begin_position_end` (_str_)\n\n    End of the range that the query is interested in, in ISO date/time stamp with millis, or relative\n    to NOW (e.g. NOW/NOW-1DAY etc.)\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ - if `begin_position_start` or `begin_position_end` are not valid, i.e not ISO or relative\ndate\n\n---\n\n##### `cloud_cover_percentage`\n\nThe range of acceptable values for cloud cover percentage as per Sentinel Dataset\n\n**Parameter**:\n\n* `percentage` (_int_ or _string_)\n\n    The percentage value or range (the format of `MIN TO MAX` where MIN is lowest acceptable and MAX is upper limit)\n    of limit for Cloud Cover Percentage\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ if the value is not an integer or valid percentage or range\n\n---\n\n##### `collection`\n\nSet the value for collection. Used to specify the name of a predefined collection of products\n\n**Parameter**:\n\n* `collection` (_str_)\n\n    Value for collection\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ if supplied string is empty i.e. \'\' or just whitespace\n\n---\n\n##### `end_position`\n\nSet a filter on the range for endposition (that is Sensing Stop Time) that the query is interested in.\n\n**Parameter**:\n\n* `end_position_start` (_str_)\n\n    Start of the period, in ISO date/time stamp with millis, or relative to NOW (e.g. NOW/NOW-1DAY etc.)\n\n* `end_position_end` (_str_)\n\n    End of the period, in ISO date/time stamp with millis, or relative to NOW (e.g. NOW/NOW-1DAY etc.)\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ - if `end_position_start` or `end_position_end` are not valid, i.e not ISO or relative\ndate\n\n---\n\n##### `file_name`\n\nSets a filter on product filename.\n\n**Parameter**:\n\n* `filename` (_str_)\n\n    Name of the product file to filter results by\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ if supplied string is empty i.e. \'\' or just whitespace\n\n---\n\n##### `footprint`\n\nSets a filter on geographic area that the query is interested in. Can use either a simple bounding box described\nas a WKT Polygon or a point described by a `Latitude` `Longitude` pair. Refer to the Sentinel Hub documentation for\nin depth information about footprint.\n\n**Parameter**:\n\n* `geographic_type` (_str_)\n\n    The Area of Interest for the query. Can either be a point (lat/lon\n    pair e.g. "0.000, 1.000") or a Polygon (WKT polygon without cut outs,\n    e.g. POLYGON ((30 10, 40 40, 20 40, 10 20, 30 10)))\n\n    Can have the Intersects() or can be just the coordinate pair or Polygon\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ if not valid WKT Polygon or point\n\n---\n\n##### `ingestion_date`\n\nSets a filter on the date the Sentinel product was ingested using the supplied range.\n\n**Parameter**:\n\n* `ingestion_date_start` (_str_)\n\n    Start of the period,  in ISO date/time stamp with millis, or relative to NOW (e.g. NOW/NOW-1DAY etc.)\n\n* `ingestion_date_end` (_str_)\n\n    End of the period, in ISO date/time stamp with millis, or relative to NOW (e.g. NOW/NOW-1DAY etc.)\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ - if `ingestion_date_start` or `ingestion_date_end` are not valid, i.e not ISO or relative\n\n---\n\n##### `last_orbit_number`\n\nSets on a filter on the last orbit number or range range of last orbit numbers (i.e `[MIN TO MAX]` whereby MIN\nis the lowest last orbit number and MAX is highest).\n\n**Parameter**:\n\n* `orbit_number` (_str_ or _int_)\n\n    The orbit number or range that should be used. Can be a single value i.e. 1234 or a range such as [1234 TO 4321]\n    must be between 0 and 999999\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ if value is not a valid number or range\n\n---\n\n##### `last_relative_orbit_number`\n\nSets a filter on the last orbit number or range range of last orbit numbers (i.e `[MIN TO MAX]` whereby MIN\nis the lowest last orbit number and MAX is highest). Relative orbit number of the oldest line within the image\ndata (the start of the product) and relative orbit number of the most recent line within the image data\n(the end of the product), respectively.\n\n**Parameter**:\n\n* `orbit_number` (_str_ or _int_)\n\n    The orbit number or range that should be used. Can be a single value i.e. 1234 or a range such as [1234 TO 4321]\n    must be between 0 and 175\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ if value is not a valid number or range\n\n---\n\n##### `orbit_direction`\n\nSets a filter on the orbit direction for the oldest data in the product\n\n**Parameter**:\n\n* `orbit_direction` (_[OrbitDirection](#Enumerations)_)\n\n    Direction that the query is interested in\n\n**Returns**: _RequestQueryBuilder_ self\n\n---\n\n##### `orbit_number`\n\nSets a filter on the orbit number or range range of orbit numbers (i.e `[MIN TO MAX]` whereby MIN\nis the lowest orbit number and MAX is highest).\n\n**Parameter**:\n\n* `orbit_number` (_str_ or _int_)\n\n    The orbit number or range that should be used. Can be a single value i.e. 1234 or a range such as [1234 TO 4321].\n    must be between 0 and 999999\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ if value is not a valid number or range\n\n---\n\n##### `platform_name`\n\nSets a filter on the platform name\n\n**Parameter**:\n\n* `platform_name` (_[PlatformName](#Enumerations)_)\n\n    The platform name to filter the results by\n\n**Returns**: _RequestQueryBuilder_ self\n\n---\n\n##### `polarisation_mode`\n\nSets a filter on polarisation mode.\n\n**Parameter**:\n\n* `polarisation_mode` (_[PolarisationMode](#Enumerations)_)\n\n\n    Specified value for polarisation_mode\n\n**Returns**: _RequestQueryBuilder_ self\n\n---\n\n##### `product_type`\n\nSets a filter on product type. Note the valid combinations with `platform_name`\n\n**Parameter**:\n\n* `product_type` [(_`Sentinel1ProductType`_ or _`Sentinel2ProductType`_ or _`Sentinel3ProductType`_ or _`Sentinel5PProductType`_)](#Enumerations)\n\n    Specified value for product type to filter the results on\n\n**Returns**: _RequestQueryBuilder_ self\n\n---\n\n##### `relative_orbit_number`\n\nSet filter on relative orbit number of the oldest line within the image data (the start of the product).\n\n**Parameter**:\n\n* `orbit_number` (_str_ or _int_)\n\n    The orbit number or range that should be used. Can be a single value i.e. 123 or a range such as [123 TO 124].\n    must be between 0 and 175\n\n**Returns**: _RequestQueryBuilder_ self\n\n**Raises**: _ValueError_ if value is not a valid number or range\n\n---\n\n##### `sensor_operational_mode`\n\nSet filter on sensor operational mode\n\n**Parameter**:\n\n* `sensor_operational_mode` (_[SensorOperationalMode](#Enumerations)_)\n\n    The value to filter products on\n\n**Returns**: _RequestQueryBuilder_ self\n\n---\n\n##### `swath_identifier`\n\nSearch all valid swath identifiers for the Sentinel-1 SAR instrument. The S1-S6 swaths apply to SM products, the IW and IW1-3 swaths apply to IW products (IW is used for detected IW products where the 3 swaths are merged into one image), the EW and EW1-5 swaths apply to EW products (EW is used for detected EW products where the 5 swaths are merged into one image).\n\n**Parameter**:\n\n* `swath_identifier` (_[SwathIdentifier](#Enumerations)_)\n\n    Swath Identifier to filter products with\n\n**Returns**: _RequestQueryBuilder_ self\n\n---\n\n##### `timeliness`\n\nFilter sentinel products on timeliness\n\n**Parameter**:\n\n* `timeliness` (_[Timeliness](#Enumerations)_)\n\n    Value of timeliness that the query is interested in\n\n**Returns**: _RequestQueryBuilder_ self\n</p>\n</details>\n\n<details>\n<summary><strong>Example</strong></summary>\n\n<p>\n\n```python\nfrom sentinelpy import RequestQueryBuilder, PlatformName\n\nsimple_q = (\n    RequestQueryBuilder()\n    .platform_name(PlatformName.SENTINEL_1)\n    .build()\n)\n\nassert simple_q == "platformname:Sentinel-1"\n\nmultiple_clauses_q = (\n    RequestQueryBuilder()\n        .platform_name(PlatformName.SENTINEL_3)\n        .and_()\n        .cloud_cover_percentage("[0 TO 5]")\n        .and_()\n        .footprint(\n            "POLYGON((-4.53 29.85, 26.75 29.85, 26.75 46.80,-4.53 46.80,-4.53 29.85))"\n        )\n        .build()\n)\n\nassert multiple_clauses_q == (\n    \'platformname:Sentinel-3 AND cloudcoverpercentage:[0 TO 5] AND \'\n    \'footprint:"Intersects(POLYGON((-4.53 29.85, 26.75 29.85, 26.75 46.\'\n    \'80,-4.53 46.80,-4.53 29.85)))"\'\n)\n```\n</p>\n</details>\n</p>\n</details>\n\n---\n\n<details id="SentinelProductRequest">\n<summary><strong>SentinelProductRequest</strong></summary>\n\n<p>\n\n### SentinelProductRequest (`NamedTuple`/`class`)\n\nNamed Tuple representing a request. Best practice would be to use the builders (_[SentinelProductRequestBuilder](#SentinelProductRequestBuilder)_ and _[RequestQueryBuilder](#RequestQueryBuilder)_) to derive.\n\n#### Properties:\n\n* `query` (_str_)\n\n    Query string value for \'q\' in the request. [Sentinel hub documentation](https://scihub.copernicus.eu/twiki/do/view/SciHubUserGuide/FullTextSearch?redirectedfrom=SciHubUserGuide.3FullTextSearch). This can be constructed using the _[RequestQueryBuilder](#RequestQueryBuilder)_.\n\n* `rows` (_int_)\n\n    Number of rows to return from the API [Reference to documentation](https://scihub.copernicus.eu/userguide/OpenSearchAPI#Paging_results)\n\n* `order_by` (_Optional[str]_)\n\n    Field to order the results on [Reference to documentation](https://scihub.copernicus.eu/userguide/OpenSearchAPI#Sorting_results)\n\n* `start` (_int_)\n\n    Start position of the records to return [Reference to documentation](https://scihub.copernicus.eu/userguide/OpenSearchAPI#Paging_results)\n\n* `username` (_str_)\n\n    Valid username to use to authenticate with the Sentinel Hub API\n\n* `password` (_str_)\n\n    Valid password to use to authenticate with the Sentinel Hub API\n</p>\n</details>\n\n---\n\n<details id="QuerySentinelProductsResponse">\n<summary><strong>QuerySentinelProductsResponse</strong></summary>\n\n<p>\n\n### QuerySentinelProductsResponse (`NamedTuple`/`class`)\n\nRepresents the result from the Sentinel Hub API\n\nCan be interacted with as an object or in a more functional style using\nthe \'on\' methods (`on_success`/`on_failure`).\n\n<details>\n<summary><strong>Property Details</strong></summary>\n\n<p>\n\n* `status_code` (_Optional[int]_)\n\n    The HTTP Status code representing the outcome of the query\n\n* `body` (_Optional[Dict[str, Any]]_)\n\n    The resulting data from the Sentinel Hub\n\n* `error` (_Optional[BaseException]_)\n\n    An error object if there was an error or exception raised\n\n* `success` (_bool_)\n\n    Whether or not the query was successful\n</p>\n</details>\n\n\n<details>\n<summary><strong>Method details</strong></summary>\n\n<p>\n\n##### `raise_error`\n\nIf encountered an error raise error encountered. Otherwise do nothing\n\nSituations this would be useful:\n\n- Prevent exceptions being swallowed\n- You prefer to handle exception rather than checking if value is None\n\n**Returns**: _None_\n\n**Raises**: Error that was encountered when querying the API\n\n---\n\n##### `on_success`\n\nA functional style method for handling successful results. When the action was successful, calls\nthe supplied function with data from Sentinel Hub API and returns the `QuerySentinelProductsResponse`\nso that other methods can be chained\n\n**Parameters**:\n\n* `callback`: (_`Callable[[Dict[str, Any]], None]`_)\n\n    Function which defines the successful behaviour\n\n**Returns**: `QuerySentinelProductsResponse` _self_\n\n---\n\n##### `on_failure`\n\nA functional style method for handling cases where the API was not reachable or\nthere was an error either in the response or parsing the response, i.e. the request\nwas not successful.\n\n**Parameters**:\n\n* `callback`: (_`Callable[[QuerySentinelProductsResponse], None]`_)\n\n    Function which defines the failure behaviour, which will be called with named tuple representing the\n    result\n\n**Returns**: `QuerySentinelProductsResponse` _self_\n\n<details>\n<summary><strong>Example</strong></summary>\n\n<p>\n\n```python\nfrom sentinelpy import QuerySentinelProductsResponse\n\n# You can use the attributes of the response in a more object fashion\n# Successful response:\nsuccessful_response = QuerySentinelProductsResponse(200, {})\n\nassert successful_response.success\nassert successful_response.status_code == 200\nassert successful_response.body == {}\nassert successful_response.error is None\n\n# Failed response:\nfailed_response = QuerySentinelProductsResponse(400, {})\nassert not failed_response.success\nassert failed_response.status_code == 400\nassert failed_response.body == {}\nassert failed_response.error is None\n\n# Erroneous response\nerroneous_response = QuerySentinelProductsResponse(None, None, IOError())\nassert not erroneous_response.success\nassert erroneous_response.status_code is None\nassert erroneous_response.body is None\nassert isinstance(erroneous_response.error, IOError)\n\n# Using the functional style methods\nsuccessful_response.on_success(\n    lambda data: print(f\'success:{data}\')\n).on_failure(\n    lambda failure_response: print(f\'failure:{failure_response.status_code}\')\n) # success:{}\n\nfailed_response.on_success(\n    lambda data: print(f\'success:{data}\')\n).on_failure(\n    lambda failure_response: print(f\'failure:{failure_response.status_code}\')\n) # failure:400\n```\n</p>\n</details>\n\n</p>\n</details>\n</p>\n</details>\n\n---\n\n<details id="Enumerations">\n<summary><strong>Enumerations</strong></summary>\n\n<p>\n\nIn order to simplify validation there are some Enumerations representing some of the types in the `RequestQueryBuilder`, each\nvalid option maps to a value defined by the API. [Refer to the Sentinel Hub API documentation for more information about the values](https://scihub.copernicus.eu/twiki/do/view/SciHubUserGuide/FullTextSearch?redirectedfrom=SciHubUserGuide.3FullTextSearch)\n\nThe enumerations are as follows:\n\n* `OrbitDirection`\n* `PlatformName`\n* `PolarisationMode`\n* `ProductType`\n* `SensorOperationalMode`\n* `Sentinel1ProductType`\n* `Sentinel2ProductType`\n* `Sentinel3ProductType`\n* `Sentinel5PProductType`\n* `SwathIdentifier`\n* `Timeliness`\n\n</p>\n</details>\n\n# Features\n\n* Queries the Sentinel Hub for products\n* Define your requests using the `RequestQueryBuilder` and `SentinelProductRequestBuilder` objects\n\n# Development Documentation\n\nBuilt using Poetry.\n\n`poetry install`\n\nTo install the library locally from source\n\n# Credits\n\nThis package was created with Cookiecutter and the `UKHO/cookiecutter-pypackage` project template.\n\n* Cookiecutter: https://github.com/cookiecutter/cookiecutter\n* UKHO/cookiecutter-pypackage: https://github.com/UKHO/cookiecutter-pypackage\n',
    'author': 'UK Hydrographic Office',
    'author_email': 'datascienceandengineering@ukho.gov.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/UKHO/sentinelpy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
