import logging
import os
import pandas as pd
from ekorpkit import eKonf


log = logging.getLogger(__name__)


class DummyCorpus:
    def __init__(self, **args):
        self.args = eKonf.to_config(args)
        self.autoload = self.args.get("autoload", True)
        self.name = self.args.get("name", None)
        self.verbose = self.args.get("verbose", True)

        self.output_dir = self.args.output_dir
        os.makedirs(self.output_dir, exist_ok=True)
        self.output_file = os.path.join(self.output_dir, self.args.output_file)
        self.force_download = self.args.force_download

        if self.autoload:
            self.build()

    def build(self):
        if not os.path.exists(self.output_file) or self.force_download:
            self.build_corpus()
        else:
            log.info(f"{self.output_file} already exists. skipping..")

    def build_corpus(self):

        docs = _dummy_data[self.name]

        df = pd.DataFrame(docs)
        df = df.dropna()

        if not self.force_download:
            if os.path.isfile(self.output_file):
                log.info("file already exists. combining with the existing file..")
                existing_df = pd.read_csv(self.output_file, index_col=None)
                df = existing_df.combine_first(df)
                df = df.drop_duplicates(subset=["id"])

        df.to_csv(self.output_file, encoding="utf-8", index=False)
        if self.verbose:
            print(df.tail())
        log.info(f"Saved {len(docs)} documents to {self.output_file}")


_dummy_data = {
    "bok_minutes": [
        {
            "id": "BOK_20181130_20181218_S1",
            "filename": "BOK_20181130_20181218",
            "mdate": "2018-11-30 10:00:00",
            "rdate": "2018-12-18 16:00:00",
            "section": "Economic Situation",
            "text": "일부 위원은 관련부서에서 지난 3/4분기 중 유로지역 경제성장 부진을 자동차 관련 규제 등 일시적 요인에 의한 것으로 평가한 것과 관련하여, 동 요인만으로는 성장률 둔화 설명에 충분하지 않아 보인다고 언급하였음.\n또한, 브렉시트(Brexit), 이탈리아 재정 이슈 등은 중기적인 문제로 판단되며, 최근 유로지역의 교역 둔화 등에 비추어 단기간 내 경기 반등을 기대하기는 어려워 보인다고 첨언하면서, 이에 대한 관련부서의 견해를 물었음.\n이에 대해 관련부서에서는 유로지역의 성장률이 낮아지고 있기는 하나, 4/4분기에는 일시적 요인이 해소되면서 소폭 반등할 것으로 보인다고 언급하였음.\n이어서 동 위원은 최근 중국과 유로지역의 경기 상황을 고려할 때, 이들 국가의 영향을 많이 받는 아세안(ASEAN) 주요국들이 향후 안정적 성장세를 유지할 수 있을지에 대한 불확실성이 커 보인다고 언급하면서, 관련부서의 견해를 물었음.\n이에 대해 관련부서에서는 금년 들어서도 아세안 주요 5개국의 성장률은 2010년 이후 평균인 5% 초중반 수준을 나타내고 있으며, 앞으로 중국경제의 향방 등에 영향을 받겠지만 현재로서는 동 수준의 성장세를 유지할 수 있을 것으로 예상된다고 답변하였음.\n다음으로 동 위원은 최근 취업자수 증가규모의 소폭 확대에도 불구하고 고용상황은 여전히 부진하다는 관련부서의 평가에 동의하면서, 이러한 고용 부진이 고령화 등 인구구조적인 변화에 상당 부분 기인한 것으로 보인다고 언급하였음.\n일례로 15∼59세 생산가능인구가 감소하는 반면 60세 이상 인구가 증가하는 구조적 변화 속에서 노동수요의 변화가 없을 경우 정규직 대신 비정규직 또는 임시일용직 취업자가 늘어나는 현상을 자연스러운 결과로 볼 수도 있다면서, 앞으로 고용상황을 평가하거나 분석할 때 이러한 고령화 현상을 염두에 둘 필요가 있다는 견해를 제시하였음.\n아울러 고령화 문제가 소비를 통해 잠재성장률에 상당한 영향을 미칠 수 있는 만큼 향후 잠재성장률에 관한 분석 시 이러한 측면을 감안하여 살펴봐 줄 것을 당부하였음.\n다른 일부 위원은 세계 산업생산과 교역량, 제조업심리지수, OECD 경기선행지수 등의 추이를 볼 때 세계경제가 정점을 지난 것으로 보인다고 언급하였음.\n또한 주요국의 GDP갭률에 대한 IMF의 추정치를 살펴보면, 독일은 2014년 이후, 미국의 경우 2016년 이후 상당 기간 플러스(+)를 지속하고 있어 이를 감안하면 내년 중 세계경제 성장세가 점차 둔화되는 것을 정상화 과정으로 평가할 수도 있다는 의견을 나타내었음.\n한편 IMF가 추정한 우리나라의 GDP갭률은 금년에도 소폭의 마이너스(-)를 지속하고 있는데, 잠재성장률 추정의 불확실성을 감안하더라도 최근의 고용상황, 제조업가동률, 물가상승률 등에 비추어 볼 때 동 추정치가 어느 정도 타당성이 있어 보인다고 언급하면서 관련부서의 견해를 물었음.\n이에 대해 관련부서에서는 큰 흐름에서는 IMF와 당행의 GDP갭률 추정치가 서로 비슷한 모습을 보이고 있으며, 추정의 불확실성이 크다는 한계를 감안할 필요가 있다고 답변하였음.\n이에 동 위원은 특정 잠재성장률 및 GDP갭률 추정치가 반드시 절대적인 기준을 제공하는 것은 아니므로, 대외 커뮤니케이션 시 잠재성장률 추정의 불확실성을 염두에 둔 유연한 접근이 필요하다는 의견을 제시하였음.\n이어서 동 위원은 향후 1년 간 세계경제가 지난 1년보다 나을 것으로 보이지 않는다고 언급하였음.\n이와 관련하여 최근 IMF에서 내년도 세계경제 성장률 전망치를 하향 조정하고 하방 리스크를 강조한 점을 상기하면서, 과거 사례를 보면 IMF가 전망치를 특정한 방향으로 수정하기 시작하면 한동안 같은 방향으로 수정하는 경향이 있었던 점에 비추어 향후 동 전망치가 추가로 하향 조정될 가능성이 있다는 의견을 덧붙였음.\n또한 동 위원은 민간소비 증가세가 완만하게 둔화되고 있다는 일부의 평가에 대해 관련부서의 견해를 물었음.\n이에 대해 관련부서에서는 최근 소매판매 실적이 예상을 하회하였으나 이것이 소비흐름의 변화 조짐인지 시간을 두고 확인할 필요가 있으며, 4/4분기에는 정부정책 효과 등으로 민간소비가 개선될 여지가 있으므로 현 시점에서 민간소비가 둔화되고 있다고 평가하기는 이르다고 답변하였음.\n한편 동 위원은 최근 발표된 가계동향조사를 살펴보면 비소비지출이 전체 소득에 비해 작지 않은 규모인 데다 그 증가세도 무척 빠른 것으로 나타나, 민간소비의 증가 여력이 제한적일 수 있다는 견해를 제시하였음.\n한편 최근 소비자심리지수가 상당 폭 하락하였는데, 동 지수가 민간소비에 대해 선행성을 갖는 것으로 볼 수 있는지 질의하였음.\n이에 관련부서에서는 소비자심리지수가 경제 뉴스, 주가 등에 따라 높은 변동성을 보이는 경향이 있어, 최근과 같은 상황에서는 민간소비에 대한 선행성 여부를 명확히 판단하기 쉽지 않다고 답변하였음.\n다음으로 동 위원은 현재의 물가상황에 대해 물가목표에 도달한 것으로 평가할 수 있는지 물었으며, 이에 대해 관련부서에서는 10월 중 소비자물가가 2%의 오름세를 보였으나 목표수준에 안착하였다고 평가하기는 조심스럽다고 답변하였음.\n이에 동 위원은 식료품·에너지 가격의 경우 변동성이 매우 크고 수요 측면과 무관한 요인에 영향을 받기 때문에 기조적 물가를 파악함에 있어 이를 제외하는 것이 바람직하다는 데 별다른 이견이 없으나, 최근 주목받고 있는 관리물가의 경우 이를 제외할지 여부에 대한 일반적인 기준이 없다고 언급하였음.\n다만 식료품·에너지 가격에서와 같이 변동성과 변동요인의 두 측면에서 판단해 볼 때, 관리물가는 비록 변동성이 낮으나 정부정책 등 비시장적 요인에 영향을 받는 점을 제외의 근거로 볼 수 있다는 견해를 제시하였음.\n이를 종합하면 기조적 물가를 파악하기 위해서 일부 품목을 제외해야 한다면 우선 식료품·에너지 가격을, 그리고 다음으로 관리물가를 제외하는 것이 합리적으로 보인다고 언급하였음.\n이어서 동 위원은 전체 취업자수 증가규모에서 공공행정, 국방, 사회보장행정, 보건·사회서비스 등 정부정책의 영향을 받는 부문과 농림어업을 제외할 경우 취업자수가 감소한 것으로 나타난다고 언급하면서, 이는 민간부문의 고용창출력이 저하되고 있음을 의미하는 것으로 보인다는 견해를 나타내었음.\n아울러 동 위원은 향후 성장경로 상에는 앞서 일부 위원이 언급한 바와 같이 고령화 등에 따른 잠재성장률 저하의 하방 리스크와 함께 경기적 측면에서도 하방 리스크가 조금 더 커 보인다고 언급하면서, 보다 면밀한 점검을 당부하였음.\n또 다른 일부 위원은 최근의 경기상황과 관련하여 금리인상이 바람직하지 않다는 일부의 시각이 있는 만큼 우리 경제가 지난 10월 경제전망의 경로를 유지하고 있는지, 그리고 향후에도 동 경로를 이탈하지 않을지 여부를 확인하는 것이 중요하다고 언급하였음.\n민간소비의 경우 앞서 일부 위원이 언급한 바와 같이 하방 리스크가 작지 않으나, 최근 발표되었거나 연내 발표할 것으로 예상되는 정부정책의 효과를 감안하면 상방 리스크도 있어 보인다고 언급하면서, 지난 전망에 이러한 정부정책의 효과를 반영하였는지 질의하였음.\n이에 대해 관련부서에서는 향후 발표될 정부정책은 그 구체적인 내용을 알 수 없으므로 전망에는 반영하지 않고 상방 리스크로 인식하고 있다고 답변하였음.\n이에 동 위원은 일부에서 내년도 수출증가율이 금년보다 크게 하락할 것으로 전망하고 있고, 반도체가격 하락 가능성을 우려하는 시각도 적지 않다고 언급하면서, 투자의 조정국면이 지속되는 상황에서 민간소비와 수출이 전망경로와 부합하는지 면밀히 살펴볼 필요가 있다고 당부하였음.\n이어서 동 위원은 최근 OECD가 내년도 세계경제 성장률을 종전보다 하향 조정한 반면 우리나라에 대해서는 2019∼20년 성장률을 금년 성장률보다 소폭 높게 전망하였다고 언급하면서, 이에 대한 관련부서의 견해를 물었음.\n이에 대해 관련부서에서는 글로벌 경기사이클이나 대내외 여건을 감안할 때 OECD 전망이 다소 낙관적으로 보인다고 답변하였음.\n또한 동 위원은 최근 유가가 급락하는 과정에서 수급요인 외에 투기적 요인도 작용한 것인지 관련부서에 질의하였으며, 관련부서에서는 유가가 단기간에 큰 폭 하락한 점에 비추어 투기적 요인도 일부 영향을 미친 것으로 보인다고 답변하였음.\n이에 동 위원은 내년도 유가 향방에 불확실성이 커진 만큼 유가 동향과 함께 우리 물가에 미치는 영향에 대해 면밀히 점검해 줄 것을 당부하였음.\n한편 동 위원은 물가의 움직임을 가늠하기 위해 다양한 지표를 참고할 필요성은 있으나, 기조적 물가지표의 경우 그 추정방법에 따라 편차가 심하여 물가 움직임에 대한 판단을 오히려 어렵게 하는 측면이 있다는 의견을 나타내었음.\n결국 물가의 추세적 움직임을 파악하는 지표로서 소비자물가지수와 근원인플레이션에 주목해야 할 것으로 보인다는 견해를 밝혔음.\n이에 대해 관련부서에서는 소비자물가지수와 근원인플레이션에 중점을 두어야 한다는 위원의 견해에 동의하면서, 다만 동 지표들도 일시적, 불규칙 요인에 영향을 받는 만큼 다양한 보조지표를 참고하고 있다고 설명하였음.\n다음으로 동 위원은 최근 부동산 거래가 크게 줄고 있다고 언급하면서, 일부에서 제기되고 있는 부동산가격의 급락 가능성에 대한 관련부서의 견해를 물었음.\n이에 대해 관련부서에서는 과거 주택가격 움직임이나 최근의 주택 실수요 등을 감안할 때 거래 둔화가 부동산가격 급락으로 이어진다고 보기 어렵다고 답변하였음.\n이에 동 위원은 최근 건설업 고용이 입주물량 증가 등으로 개선되는 모습이나, 내년에도 건설투자의 조정국면이 지속될 것으로 전망되는 데다 각종 부동산대책들이 건설업에 부정적 영향을 미칠 수 있으므로 주의 깊게 살펴볼 필요가 있다고 당부하였음.\n아울러 동 위원은 주택자산 보유의 세대별 격차가 소비에 미치는 영향에 대한 관련부서의 분석이 유익했다고 평가하면서, 자산효과에 국한하면 주택가격 하락이 소비에 미치는 영향은 제한적이겠지만 경우에 따라서는 대출 부실화 등을 통해 소비에 부정적 영향을 초래할 가능성도 배제할 수 없다는 견해를 표명하였음.\n한편 일부 위원은 금년 들어 대외 여건의 불확실성이 높아졌으며, 특히 지난 1∼2개월 사이에 불확실성 정도가 크게 심화된 것으로 보인다는 의견을 제시하였음.\n단적으로 미·중 무역분쟁을 둘러싼 우려감 고조 속에 독일의 3/4분기 GDP성장률 마이너스 전환 등 유로지역 경제지표 부진이 가시화되면서 세계금융시장 지표에서도 민감한 반응이 나타나는 모습이라고 평가하였음.\n일부 선진국과 신흥국을 중심으로 세계경제가 둔화 조짐을 보이고 있음에도 불구하고 우리 수출은 전반적으로 양호한 흐름을 지속하고 있다고 언급하였음.\n다만 지역별로 살펴보면 신흥국에 대한 수출의 경우 이들 국가의 실물동향에 부합하여 둔화되는 모습을 보이고 있는 반면, 선진국에 대한 수출은 상당히 높은 증가율을 나타내는 등 다소 설명하기 어려운 흐름을 보이고 있다고 평가하였음.\n이에 대해 일부에서는 미·중 무역분쟁 심화에 대비한 선수출, 공급망별 재고 확보 노력 등에 기인하는 것으로 분석하고 있는 만큼 내년에는 우리나라의 수출증가율이 반락할 가능성을 배제할 수 없어 보인다는 견해를 표명하였음.\n이어서 동 위원은 민간소비가 지난해 4/4분기 이후 회복되는 모습을 보였으나, 가계의 소비성향이 더욱 높아지지 않는 한 금년 4/4분기 이후에는 소비증가세가 약화될 가능성이 있어 보인다는 의견을 밝혔음.\n지난 2012년 이후 상당 폭 하락하였던 소비성향이 하락세를 멈추고 올라오고는 있으나 하락 이전 수준을 회복하지 못하는 모습인데, 그 이면에 일시적 요인이 아니라 고령화 등 구조적 요인이 자리하고 있을 가능성이 있다는 점에서 민간소비 전망에 다소 하방위험이 있어 보인다는 견해를 덧붙였음.\n또한 동 위원은 통화정책을 통해 화폐가치의 안정을 추구하는 중앙은행의 입장에서는 특정한 물가를 통해 중장기적인 화폐가치를 간접적으로 측정할 수밖에 없는데, 궁극적으로는 일반인들이 생활하면서 평균적으로 사용하는 상품과 서비스의 조합(basket)인 소비자물가를 통해 화폐가치를 측정하는 것이 이론적으로나 실제적으로 바람직하다는 의견을 표명하였음.\n다만 소비자물가지수에 중장기적인 예측이 곤란한 교란요인들이 포함되어 있어, 단기 시계에서는 근원물가를 보거나 관리물가를 제외한 물가지수를 기조적 물가흐름의 참고지표로 사용하고 있는 것으로 보인다는 의견을 나타내었음.\n이처럼 단기 시계에서 기저물가의 흐름을 포착하는 데 있어서는 여러 물가지표 가운데 경기민감물가지수가 가장 유용해 보인다고 첨언하였음.\n한편 물가목표 달성 여부에 대한 평가와 관련하여, 목표수준으로 명시되어 있는 2%가 중기 평균의 개념임에도 불구하고 일부에서는 이를 상한(ceiling) 개념으로 해석하고 있다고 지적하면서, 대외 커뮤니케이션 시 물가목표를 중기 평균으로 관리한다는 점을 잘 설명할 필요가 있다는 견해를 밝혔음.\n다음으로 동 위원은 최근 우리 경제의 성장경로에 불확실성이 더 높아진 것으로 보이는지 관련부서에 질의하였음.\n이에 대해 관련부서에서는 가장 큰 하방 리스크로 평가되는 미·중 무역분쟁의 경우 연초 예상보다 그 정도가 심화된 데다 경제 외적인 요인도 작용하고 있어 향후 방향성을 예단하기도 어렵다는 점을 감안할 때, 우리 경제의 불확실성은 보다 확대된 것으로 보인다고 첨언하였음.\n이에 동 위원은 주요국 금융시장의 움직임을 통해 볼 때, 미·중 무역분쟁 등에 대한 민감도가 점점 높아지고 있는 점에 유의할 필요가 있다고 당부하였음.\n다른 일부 위원은 관련부서에서 민간소비는 완만하나마 증가세를 이어가는 모습이고 동 흐름이 4/4분기에도 이어질 것으로 예상하고 있다고 언급하면서, 이는 지난 10월 금년과 내년 민간소비 증가율을 2.7%로 전망한 것과 부합하는 것으로 보이며, GDP성장률이 3.1%로 높아졌던 지난해 민간소비 증가율이 2.6%였다는 점을 감안하면 금년과 내년 민간소비 전망치는 상당히 양호한 수준으로 보인다는 견해를 나타내었음.\n앞서 일부 위원이 언급한 소비자심리지수의 경우 지난해 하반기 이후 계속 낮아져 최근에는 기준치 100을 하회하고 있어 민간소비와 다소 괴리된 모습을 보이고 있다고 언급하였음.\n이는 전반적인 소비자심리가 최근의 주가 하락, 고용 부진, 영세자영업자 불황, 양극화 심화, 미·중 무역분쟁 등과 같은 대내외 불확실성에 크게 좌우되고 있고, 실제 가계의 소비지출은 아동수당 지급, 기초연금 인상, 개별소비세와 유류세 인하 등 여러 정부대책들에 의해 뒷받침되고 있는 점 등에 기인한 것으로 보인다는 의견을 나타내었음.\n다만 소비자심리 위축이 더욱 장기화되거나 심화될 경우 자기실현적 과정 등을 통해서 실제 민간소비에 부정적 영향을 미칠 가능성을 배제하기 어렵다고 첨언하였음.\n민간소비는 전체 GDP의 절반 정도를 차지하고 있으며 여타 수요항목에 비해서 상당히 안정적인 흐름을 보이는 경향이 있어 경기변동을 완충하는 역할을 하고 있고, 최근에는 설비·건설 투자의 부진이 지속됨에도 불구하고 수출과 함께 잠재성장률 수준의 성장세를 지탱하고 있다고 평가하였음.\n고령화 진전, 고용 부진, 가계부채 누증 등이 향후 민간소비에 하방 리스크로 작용할 수 있으나, 최근 명목임금 증가세가 확대되고 정부로부터 가계로의 이전소득이 증가하는 등 상방 리스크도 있으므로, 이러한 여러 요인을 고려하여 관련부서에서는 앞으로 민간소비 흐름에 대해 보다 면밀하게 점검하는 한편 소비자심리지수와 실제 민간소비와의 관계에 대해서도 관심을 가지고 살펴볼 필요가 있다고 당부하였음.\n이어서 동 위원은 최근 경기 동행지수, 선행지수 등이 상당 폭 하락하고 있지만, GDP성장률 흐름을 살펴보면 과거의 경기하강기에 비해 안정적인 모습을 보이고 있다고 언급하였음.\n비록 금년 들어 동 흐름이 낮아지는 모습을 보이고 있으나, 경기국면의 전환을 시사할 정도는 아닌 것으로 판단된다고 평가하였음.\n물론 선행지수가 가지는 선행성을 감안하면 향후 GDP성장률 흐름이 나빠질 가능성을 배제할 수는 없으나, 현 시점에서 경기국면이 전환되었다고 예단하기는 이르다는 견해를 제시하였음.\n한편 동 위원은 최근 기조적 물가흐름이 대체로 낮아지는 모습을 보이고 있으나, 전체적으로는 1.5%에서 2.0% 사이에 있는 것으로 보인다고 언급하였음.\n이와 같이 기조적 물가흐름이 1% 중후반 수준이라면 디플레이션을 크게 우려할 필요가 없고 수요압력을 걱정해야 하는 것도 아닌, 즉 다소 시간을 두고 지켜볼 수 있는 상황이라고 평가할 수 있어 보인다는 의견을 나타내었음.\n또 다른 일부 위원은 세계경제가 지난 2년 이상 잠재성장률을 상당 폭 웃도는 성장률을 지속하여 왔는데, 이를 주도해 왔던 미국의 경기가 성숙국면에 진입하면서 경기상황에 대한 논쟁이 최근 많이 제기되고 있다고 언급하였음.\n과거 30∼40년의 경기순환 사례를 돌이켜 볼 때, 경제가 잠재성장률을 상회하는 성장세를 보이다가 성숙국면에 진입하면서 성장률이 잠재성장률 수준으로 완만히 낮아지는 것은 매우 자연스러운 현상으로 평가할 수 있겠으나, 최근에는 미·중 무역분쟁, 브렉시트, 이탈리아 재정문제 등 여러 리스크 요인들이 중첩적으로 나타남에 따라 성장률이 잠재성장률 수준으로 완만해지는 것이 아니라 본격적인 경기 둔화가 시작되는 것이 아닌가 하는 우려가 일부에서 제기되는 것으로 보인다고 언급하였음.\n그러나 앞서 언급한 불확실성 요인들을 제외하고 경제적 요인에 국한하여 볼 경우에는 과거 세계 또는 우리나라 경기가 본격적인 둔화국면에 진입할 때 일반적으로 나타났던 현상, 즉 글로벌 유동성의 급격한 축소, 생산성을 웃도는 임금상승률에 따른 기업부담의 가중, 제조업 부문의 재고누증 등이 아직 가시화되지 않고 있는 것으로 보인다고 언급하면서, 이에 대한 관련부서의 견해를 물었음.\n관련부서에서는 최근 세계경제는 경제 외적인 요인들로 인해 불확실성이 매우 높아졌으며, 그 결과 연초에 전망하였던 것보다는 성장세가 약화된 것으로 보인다고 답변하였음.\n다만 현재의 글로벌 경기상황에 대해 둔화 또는 침체로 진입하는 국면으로 평가하기보다는 그간 높았던 성장세가 조정되는 과정으로 이해하는 것이 타당해 보인다고 첨언하였음.\n이에 동 위원은 미·중 무역분쟁과 브렉시트 문제가 예상 외의 방향으로 전개될 경우 글로벌 공급 구조에 영향을 미침으로써 궁극적으로 우리나라의 잠재성장률에도 부정적인 영향이 초래될 수 있는 점에 유의할 필요가 있다고 당부하였음.\n이어서 동 위원은 그동안 우리나라의 잠재성장률이나 경기 진폭이 꾸준히 축소된 만큼 경기지표를 볼 때 전년동기대비보다는 전기대비 수치를 보다 많이 활용할 필요가 있다고 언급하였음.\n경기 진폭이 축소된 상황에서 전년동기대비 지표를 사용할 경우 경기 판단이 어려워질 수 있는 데다 어떤 충격이 왔을 때 기저효과가 크게 나타나 지표 해석에 왜곡이 초래될 소지도 있다는 의견을 나타내었음.\n또한 경기상황에 대한 커뮤니케이션 시 시장참가자들의 과도한 우려나 낙관이 나타나지 않도록 하기 위해서는 GDP성장률 등 경기지표의 수준과 흐름에 대한 판단과 평가가 보다 정확히 전달될 수 있는 방안을 지속적으로 모색해 나가야 한다는 견해를 밝혔음.\n또한 동 위원은 물가동향 판단을 위해 다양한 기조적 물가지표를 활용하고 있는데 동 지표의 상·하한 범위가 확대될 경우 물가흐름에 대한 판단이 어려워질 수 있다는 점을 언급하면서, 동 지표들을 해석하고 활용하는 방안에 대해 고민해 본다면 통화정책 수행에 도움이 될 것이라는 의견을 나타내었음.",
        },
        {
            "id": "BOK_20181130_20181218_S2",
            "filename": "BOK_20181130_20181218",
            "mdate": "2018-11-30 10:00:00",
            "rdate": "2018-12-18 16:00:00",
            "section": "Foreign Currency",
            "text": "일부 위원은 그동안 글로벌펀드와 패시브펀드의 규모가 크게 확대되어 우리나라 자본유출입의 중요한 변수로 자리잡은 것으로 보인다고 언급하면서, 동 펀드 자금의 유출입을 촉발하는 요인들을 사전에 면밀히 파악해 두는 것이 중요하다고 당부하였음.\n이어서 동 위원은 글로벌 금융위기 이후 은행권에 대한 규제가 강화되면서 비은행금융기관을 중심으로 금융불균형에 대한 우려가 커지고 있다고 지적하였음.\n이러한 가운데 미국의 경우 레버리지론이 빠르게 증가하면서 이를 유동화한 CLO(collateralized loan obligation) 잔액도 가파르게 늘어나고 있다고 언급하면서, 우리나라 금융기관들이 이러한 CLO 상품에 어느 정도 투자하고 있는 것으로 보는지 질의하였음.\n이에 관련부서에서는 정확히 파악하기가 쉽지 않으나 아직까지는 그 규모가 그리 크지 않은 것으로 보인다고 답변하였음.\n이에 동 위원은 해외 대체투자펀드의 경우 대출채권이나 메자닌 트랜치(mezzanine tranche) 등 상대적으로 신용위험이 높은 상품에 투자하는 것으로 알려져 있는데, 이에 대해 어떻게 평가하는지 관련부서의 견해를 물었음.\n이에 관련부서에서는 아직 우려되는 상황은 아닌 것으로 보이나, 최근 발표된 미국 금융안정보고서에서 드러난 바와 같이 미국 주택가격이 장기추세보다 높은 수준에 있는 만큼 경계심을 가지고 살펴보겠다고 답변하였음.\n또한 동 위원은 최근 글로벌 펀드자금 흐름의 특징 중 하나로 MMF 증가를 들 수 있는데, 과거 금융위기 당시 MMF가 위기 파급의 주요 채널이었던 점을 상기하면 향후 동 펀드의 흐름에 대해 관심을 가지고 살펴볼 필요가 있다는 견해를 제시하였음.\n이와 관련하여 금년 1∼9월 중 국내 금융권의 자금흐름을 살펴보면 은행에서 비은행금융기관으로 자금이 많이 흘러갔는데, 전반적으로 단기 조달이 확대됨으로써 리스크가 누적되지 않도록 총체적인 자금흐름을 면밀히 모니터링하는 것이 중요하다고 당부하였음.\n이에 관련부서에서는 최근 MMF가 늘어난 것은 연말 효과와 함께 단기 조달 확대에도 일부 기인한 것으로 보인다고 답변하였음.\n다만 MMF의 규모 확대에도 불구하고 과거 금융위기 시와 달리 CLO 등 구조화상품에 대한 익스포저는 제한적이어서 우려할 상황은 아닌 것으로 보인다고 첨언하였음.\n다음으로 동 위원은 3/4분기 중 건설업의 매출액이 전년동기대비 감소한 반면 영업이익은 증가하였다고 언급하면서, 이러한 현상이 나타난 원인에 대해 질의하였음.\n이에 대해 관련부서에서는 아파트 분양을 중심으로 주택사업이 호조를 보임에 따라 수익성은 좋아졌으나 주택사업 이외의 부문은 부진하여 매출액이 감소한 것으로 보인다고 답변하였음.\n다른 일부 위원은 그동안 둔화추세를 보여 온 가계대출이 10월 들어 증가세가 일시적으로 확대되었으나 앞으로 이러한 증가세가 지속될 것으로는 보이지 않는다고 언급하고, 내년에는 5%대로 증가세가 더욱 둔화될 것으로 보인다면서 관련부서의 견해를 물었음.\n이에 대해 관련부서에서는 연말로 갈수록 가계대출 증가세가 둔화되겠지만 그 속도는 완만하고 여전히 가계대출 증가율이 소득증가율을 웃돌 것으로 보인다고 답변하였음.\n이어서 동 위원은 최근 언론보도에 따르면 감독당국에서 주택담보대출금리가 낮아지는 효과가 나타날 수 있는 방향으로 COFIX 산정체계를 개편하려는 움직임을 보이고 있으므로 관련 동향을 주의 깊게 살펴볼 필요가 있다고 언급하였음.\n또 다른 일부 위원은 중국의 부채 리스크에 대한 관련부서의 분석이 유익했다고 평가하면서, 중국의 실물경제가 나빠지게 되면 그동안 많이 회자되었던 부채 리스크가 크게 부각될 가능성이 높아 보이는데, 중국 정부당국과 중앙은행은 동 리스크에 대해 어떻게 인식하고 있는지 질의하였음.\n이에 대해 관련부서에서는 2012년경부터 중국 정부당국에서 부채 리스크에 대한 논의와 함께 대응을 계속하여 왔고, 특히 지난해 하반기 이후에는 디레버리징 정책을 강화하였다고 답변하였음.\n이에 동 위원은 중국경기가 둔화되면서 중국 당국이 기존의 디레버리징 정책을 어떤 식으로 조정할지 숙고하고 있는 것으로 보인다고 언급하였음.\n또한 이러한 중국의 사례에 비추어 보면, 우리 경제가 둔화되고 기초경제여건에 대한 신뢰가 약화될 경우 그동안 관리가능하다고 평가되어 왔던 가계부채 문제가 크게 부각될 가능성이 있으므로, 이러한 상황이 발생하기 전에 미리 대응해 나가는 것이 중요하다는 견해를 표명하였음.\n이어서 동 위원은 미·중 양국 정상회담이 곧 열리는데 앞으로 미·중 무역분쟁이 어떻게 진행될 것으로 보는지 관련부서의 견해를 물었음.\n이에 관련부서에서는 이번 무역분쟁에 무역 이외에도 여러 복잡한 문제들이 결부되어 있어 쉽게 해결되기 어려우며 장기화될 가능성이 높다고 보는 시각이 중론이라고 답변하였음.\n다음으로 동 위원은 최근 Powell 미연준 의장의 발언이 금리인상 기대와 연결되면서 일부 시장이 민감하게 반응한 바 있는데, 향후 연준의 금리인상 행보에 대해 관련부서의 견해를 물었음.\n관련부서에서는 연준 의장의 발언은 지표의존적(data dependent)으로 리스크를 관리하면서 정책을 운영하겠다는 기존 입장을 견지한 것으로 보이며, 시장에서는 이에 다소 민감하게 반응한 것으로 보인다고 답변하였음.\n한편 동 위원은 미 통화정책 정상화, 미·중 무역분쟁 등의 리스크 요인과 비교할 때 이탈리아 재정문제 등 유로지역의 정치적 불확실성은 국제금융시장에 미치는 영향 측면에서 얼마나 큰 리스크로 보이는지 관련부서에 질의하였음.\n관련부서에서는 유로지역 경제에서 이탈리아가 차지하는 비중이 상당히 높고, 이탈리아의 국채발행 규모가 역내 세 번째로 클 뿐만 아니라 동 국채에 대한 유로지역 금융기관의 익스포저도 상당하기 때문에, 가능성은 낮지만 만일 이탈리아 재정문제가 과거 그리스 사태와 같은 식으로 비화될 경우 그 충격은 상당할 것으로 보인다고 답변하였음.\n또한 동 위원은 최근 가계부채 증가규모가 크게 줄지 않고 있지만 전반적인 증가속도는 둔화되고 있는 것으로 보인다고 언급하였음.\n가계부채 문제가 중요한 만큼 정부당국이 취약차주와 고위험가구 등을 위한 여러 가지 대책을 시행하고 추가적으로 검토하는 것은 당연하고 불가피한 측면이 있지만, 기본적으로 대출이 용이하게 이루어질 수 있음에 따라 레버리지가 증가하고, 그 결과 정부대책이 다시 마련되는 일련의 과정이 반복되는 것은 바람직하지 않다는 의견을 나타내었음.\n따라서 가계의 레버리지가 쉽게 확대되지 않도록 하는 환경을 조성하는 것이 우리 경제 전체적으로 중요하다고 덧붙였음.\n이어서 동 위원은 국내 금융권의 상업용 부동산 익스포저가 크게 증가한 상황에서, 만일 상업용 부동산가격이 하락할 경우 관련 대출의 부실화와 투자상품의 유동성 리스크 증가 가능성을 배제할 수 없으므로 시장 동향을 면밀히 살펴볼 필요가 있다고 당부하였음.\n이러한 맥락에서 내년중 상업용 부동산의 수익률 전망이 어떠한지 관련부서에 질의하였음.\n이에 대해 관련부서에서는 상업용 부동산 공급이 증가할 것으로 보여 투자수익률이 다소 낮아질 가능성이 있지만, 은행 예금금리와의 격차는 상당한 수준으로 유지될 것으로 보인다고 답변하였음.\n다음으로 동 위원은 금년 들어 은행과 비은행의 수신이 동시에 증가한 가운데 특히 기업을 중심으로 은행 정기예금이 큰 폭 늘었는데, 그 이유가 무엇인지 질의하였음.\n관련부서에서는 기업의 실적이 양호한 반면 투자수요는 크지 않아 여유자금이 증가한 데다 수시입출식예금에서 정기예금으로의 자금 이동 움직임도 있었기 때문으로 보인다고 답변하였음.\n한편 일부 위원은 금년 8월 이후 홍콩 은행 간 시장금리(Hibor)와 상하이 은행 간 시장금리(Shibor) 간 격차가 확대되었는데, 그 이면에 어떤 요인이 작용하고 있는지 면밀히 살펴볼 필요가 있다고 언급하였음.\n이어서 동 위원은 금년 중 은행의 주택담보대출 증가율을 대출종류별로 살펴보면, 집단대출이 지난해보다 증가세가 둔화되었고, 전세자금대출을 제외한 개별 주택담보대출은 오히려 마이너스(-) 증가율을 보인 반면 전세자금대출 증가율은 지난해보다 큰 폭 확대되었다고 언급하였음.\n이처럼 전세자금대출이 은행의 주택담보대출 증가세를 주도하고 있는 최근 상황은, 지난 2014∼16년 중 가계대출이 급증할 당시 신규분양시장 활황에 기인한 집단대출과 전세자금대출을 제외한 개별 주택담보대출이 대출증가세를 주도했던 것과는 확연히 다른, 새로운 국면으로 보인다는 의견을 나타내었음.\n은행의 집단대출은 최근의 신규분양 흐름을 감안하면 향후 그 증가율이 보다 축소될 가능성이 있고, 전세자금대출을 제외한 개별 주택담보대출의 경우에도 감소세가 확대될 수 있을 것으로 보인다고 언급하였음.\n이에 반해 전세자금대출의 경우 DSR 규제가 완화 적용되는 가운데 은행들이 신용대출과 더불어 적극적으로 영업에 나섬에 따라 소득증가율을 크게 웃도는 증가세를 보이고 있으므로, 현재의 규제 수준이 적절한지 점검해 볼 필요가 있다는 의견을 덧붙였음.\n다른 일부 위원은 10월 이후 글로벌 금융시장에서 위험회피 성향이 증대되면서 신용위험에 대한 경계감이 높아지고 있는 것으로 보인다고 언급하였음.\nLibor-OIS 스프레드와 미국 및 유럽계 은행 CDS 프리미엄이 상승하고, 미국 회사채시장에서는 비우량물과 우량물의 신용스프레드가 높아지고 레버리지론에 대한 우려도 커지고 있으며, 중국에서도 이와 비슷한 현상이 나타나고 있는 가운데 주요 외신이나 글로벌 투자은행 보고서에서도 글로벌 신용시장에 대한 우려의 목소리가 커지고 있다고 첨언하였음.\n이러한 현상은 저금리 기조가 장기간 지속되고 경기확장 과정에서 기업부문의 신용위험이 낮게 평가되어 오다가 미 연준의 통화정책 정상화, 미·중 무역분쟁에 따른 기업실적 둔화 우려 등을 계기로 기업 신용위험에 대한 인식에 변화가 나타나고 있음을 시사하는 것으로 보인다는 견해를 표명하였음.\n특히 글로벌 금융위기 이후 비우량 회사채 비중이 크게 확대된 점, 일부 신흥국 기업의 외화표시 회사채 발행이 늘어나면서 통화불일치 현상이 심화된 점 등을 근거로 향후 글로벌 회사채시장에서의 조정 폭이 작지 않을 수 있다는 우려가 제기되고 있으며, 최근 미 연준이 처음으로 발표한 금융안정보고서에서도 비우량 기업부채에 대한 위험을 적시하고 있다고 소개하였음.\n우리나라의 상황을 살펴보면 회사채시장이나 CP, 전자단기사채 시장이 대체로 양호한 모습을 지속하고 있고, 금융기관의 기업대출도 꾸준히 증가하고 있어 글로벌 신용시장과는 다소 다른 모습을 나타내고 있다고 언급하면서, 국내 금융시장이 글로벌 금융시장과 차별화된 모습을 보이는 이유에 대해 관련부서에 질의하였음.\n이에 대해 관련부서에서는 국내 금융시장의 차별화 현상이 우리나라의 대외건전성이나 재정건전성이 견실하다는 점, 기업의 재무건전성이 꾸준히 개선되어 온 가운데 기업실적도 우려했던 것보다 양호하다는 점, 그리고 시중 유동성이 풍부하다는 점 등에 주로 기인한 것으로 보인다고 답변하였음.\n특히 최근 채권시장에서는 국고채 발행물량 축소 등으로 수요 우위의 수급상황이 이어지고 있고, 국고채 금리가 하락하면서 상대적으로 금리 메리트가 높은 회사채에 대한 투자수요가 견조한 상황이라고 첨언하였음.\n아울러 그동안 국내 일반기업의 외화채권 발행규모가 크지 않았고 장기시계에서 국내 회사채시장이 과열되지 않았던 점 등도 긍정적 영향을 미친 것으로 보인다고 덧붙였음.\n이에 동 위원은 향후 국내 금융시장에서 신용위험에 대한 경계감이 글로벌 추세를 따라서 높아질 가능성을 배제할 수 없으므로, 이 점에 유의하여 세심히 모니터링할 필요가 있다고 당부하였음.\n이에 관련부서에서는 미 연준의 금리인상 속도, 주요국 보호무역주의의 전개양상 등에 따라 국내외 유동성사정이나 기업실적 등이 크게 영향받을 수 있는 점에 유념하겠다고 답변하였음.\n또 다른 일부 위원은 금년 1∼4월에 이어 10월 이후 글로벌 금융시장의 변동성이 재차 큰 폭 확대되었는데, 달러인덱스(DXY)와 신용스프레드의 움직임이 지난번 변동성 확대 시기와 다소 차이가 있고, 글로벌 경기 둔화 우려도 이번에 더 크게 작용하는 것으로 보인다는 의견을 제시하면서 관련부서의 견해를 물었음.\n아울러 금년 초에는 주가가 급락하였다가 단기간에 회복한 바 있는데 이번에는 어떻게 진행될 것으로 보는지 질의하였음.\n관련부서에서는 연초와 최근 상황 모두 미 연준의 금리인상이라고 하는 큰 흐름이 주요 동인으로 작용하였으나 주가변동성지수(VIX)의 움직임이 대변하듯이 조정의 폭과 기간에서 차이가 있다고 답변하였음.\n구체적으로 2017년 중 매우 낮은 수준을 보이던 VIX가 2월 초 미 국채금리의 3% 상회와 함께 일시 급등하였다가 급락하였던 데 비해, 10월 이후에는 VIX 상승 폭은 상대적으로 크지 않지만 높아진 수준이 지속되고 있다고 설명하였음.\n이는 동 위원이 언급한 글로벌 경기 둔화 우려, 중국경제의 불안 등이 반영되면서 리스크 회피가 이어지고 있었기 때문이며, 시장에서는 현재와 같은 상황이 장기화될 가능성이 제기되고 있다고 첨언하였음.\n이어서 동 위원은 국제금융시장에서 글로벌 유동성의 축소 가능성에 대한 경계감이 지속되고 있다는 관련부서의 평가와 관련하여, 글로벌 유동성 축소의 정확한 의미가 무엇인지 물었음.\n예를 들면 주요국 통화정책 정상화에 따라 중앙은행이 공급하는 유동성이 단순히 줄어드는 것을 뜻하는지, 아니면 글로벌 금융시장 전반의 신용여건이 경색되는 측면까지 포함하는 것인지 구분이 필요해 보인다고 언급하였음.\n이에 대해 관련부서에서는 유동성을 중앙은행 및 금융기관의 통화성 부채를 포함한 넓은 개념으로 사용하였으며, 미 연준의 금리인상이라는 큰 흐름 속에서 금융기관들이 디레버리징을 위한 위험자산 축소를 저울질하는 최근 상황을 의미한 것이라고 답변하였음.\n한편 동 위원은 DSR 규제가 현재 은행의 가계대출에만 적용되고 있고 비은행 금융기관의 경우에는 내년 상반기 중 업권별로 순차적으로 시행될 예정인 가운데 기업대출로 분류되는 개인사업자대출의 경우 DSR 규제 적용대상이 아니어서 일부 풍선효과가 나타날 수 있어 보인다고 언급하면서, 이에 대한 관련부서의 견해를 물었음.\n이에 관련부서에서는 최근 개인사업자대출에 대해서도 LTV 규제가 도입된 데다 임대업이자상환비율(RTI) 규제가 추가로 강화된 부분도 있어 대출수요 이전효과가 종전보다는 크지 않을 것으로 보인다고 답변하였음.\n또한 동 위원은 최근 회사채 신용스프레드가 확대되고 대기업 대출금리도 여타 금리보다 더 상승하였는데, 이 과정에서 신용위험에 대한 인식 변화 등이 작용한 것은 아닌지 질의하였음.\n관련부서에서는 금리가 전반적으로 하락하는 과정에서 국고채 금리가 상대적으로 큰 폭 하락함에 따라 우량물 회사채를 중심으로 스프레드가 소폭 확대되었다고 답변하였음.\n또한 대기업 대출금리의 상승은 기준금리 인상 기대로 지표금리인 단기시장금리가 올라간 데 영향을 받은 측면이 강하며, 신용위험과는 관련이 없는 것으로 보인다고 첨언하였음.\n이에 동 위원은 하반기 중 대기업의 회사채 순발행규모가 전년동기보다 소폭 늘어났는데, 이러한 모습도 금리인상 기대를 반영한 것인지 물었음.\n이에 관련부서에서는 금리인상에 대비한 선차환 및 운전자금 조달 목적의 발행이 많은 것으로 보인다고 답변하였음.",
        },
        {
            "id": "BOK_20181130_20181218_S3",
            "filename": "BOK_20181130_20181218",
            "mdate": "2018-11-30 10:00:00",
            "rdate": "2018-12-18 16:00:00",
            "section": "Financial Markets",
            "text": "일부 위원은 현재 대기업들이 전반적으로는 문제가 없지만, 건설 조선업 등에 속하는 중견대기업 중 일부는 자금사정에 상당한 어려움을 겪고 있어 향후 경기부진이 지속될 경우 이들이 위기를 극복할 수 있을지 우려된다는 의견을 표명하고, 이와 관련하여 취약업종에 속한 중견대기업의 경영상황을 면밀히 점검한 후, 경제의 성장동력과 관계없는 한계기업에 대해서는 과감하게 구조조정 절차를 개시함으로써 경제의 불확실성을 줄이는 것이 바람직하겠지만, 경제에 미치는 파급효과가 크고 또 성장동력으로 작동할 수 있는 기업에 대해서는 선제적인 지원대책을 마련할 필요가 있다는 의견을 나타내었음.\n또한, 동 위원은 최근 들어 가계대출이 다시 증가한 이유가 적격대출 및 신용대출의 증가, 정부의 세제감면 등에 있는 것으로 보이나, 이들 요인 이외에 금리인하의 영향도 있는 것으로 생각되므로 그 효과를 분석해 볼 필요가 있다는 의견을 피력하였음.\n이에 대해 관련부서에서는 기준금리 인하가 가계대출에 영향을 미치지만 그 외에 최근 은행의 가산금리 및 CD금리가 큰 폭 하락한 점, 은행들이 고정금리대출 목표를 맞출 필요가 있었던 점, 정부의 세제 혜택 등의 영향이 가세하면서 대출 규모가 확대된 측면이 있다고 답변하였음.\n이와 관련하여 동 위원은 일반적인 주택담보대출의 경우 금리보다는 부동산 가격 전망 등이 대출수요를 결정하지만, 최근의 주택담보대출의 구성요인을 보면 생계자금 창업자금 수요 등이 주된 요인인 것으로 파악되었으므로 금리의 영향을 주목하여 지켜봐 줄 것을 요청하였음.\n다른 일부 위원은 금년 7월 이후 장단기금리의 역전현상이 지속됨에 따라, 장기투자자들의 투자태도가 신용위험을 선호하도록 유도하여 왔는데, 최근 역전현상이 해소된 상황에서도 장기투자자들의 투자태도에 변화가 없는지 물었으며, 이에 대해 관련부서에서는 장단기금리 역전 현상 해소에도 불구하고 여전히 장기금리 자체가 매우 낮기 때문에 장기투자기관들은 큰 변화 없이 상대적으로 금리가 높은 신용채권에 대한 투자를 확대하고 있다고 답변하였음.\n또 다른 일부 위원은 저금리 기조가 대내적으로 상당히 오랜 기간 동안 지속됨으로 인해 기업들이 은행 차입 대신 직접금융을 통한 자금조달을 선호하는 한편, 은행들은 늘어난 예금을 운용할 데가 없음으로 인해 수지가 악화되는 현상을 피하기 위해 가계대출을 확대하려는 움직임을 보이고 있고, 가계 정부 기업의 총 차입이 증가하는 등 우리 금융시장에 미치는 영향이 다각적으로 나타나고 있으므로 전체적인 시각에서 시장상황을 점검하고 대응방안 마련을 고려할 필요가 있다는 의견을 피력하였음.\n이에 대해 관련부서에서는 장기금리 하락이 경기적 요인에 의해 나타나는 단기적 현상이 아니라 국고채 수요 우위, 인구구조 등에 영향을 받는 구조적이고 장기적인 현상이라고 인식하고 있으며, 이런 현상이 지속될 경우 금융산업구조에도 상당한 영향을 미칠 수 있으므로 예의 주시하겠다고 답변하였음.\n또한, 동 위원은 가계대출의 증가가 과다하여 시스템 리스크를 유발하는 등 공적인 부담을 증대시키는 기관에 대해서는 분담금을 부과하거나 대손충당금 적립비중을 높이는 등의 억제 방안을 고려해 볼 필요가 있다는 의견을 개진하였으며,  이와 관련하여 일부 집행간부는 영국 인도 등 일부 국가에서 주택담보대출 증가분에 대해 위험가중치(risk weight)를 차등하여 부과하고, 그 금액을 자본금으로 적립하는 방식으로 주택담보대출을 억제하고자 하였으나, 은행들이 주택담보대출을 줄이는 대신 중소기업대출을 줄이는 행태를 보임에 따라 어려움을 겪고 있다는 사례를 소개하였음.\n일부 위원은 경기의 하방위험(downside risk)이 증대되고 있음에도 불구하고 최근 선진국들의 각종 조치로 인해 금융시장의 변동성이 완화됨에 따라 시장참가자들이 테일리스크를 간과하는 경향이 있다고 지적한 후, 따라서 우리나라 금융시장의 안정성에 대해 여러 분석지표로 평가해 볼 때, 연체율과 외화자금 유출 가능성은 안정적인 범위에 있으나 GDP대비 가계부채 비중, 가계의 순저축률, 평균노동인구 연령 등의 측면에서는 취약한 것으로 보인다면서, 향후 국내 금융시장이 외부충격에 대해 강한 면역시스템을 가질 수 있도록 고민해 보아야 한다는 의견을 제시하였음.\n또한, 동 위원은 지난 2차례의 기준금리 인하 이후 은행의 예대금리가 하락하고 예대금리차도 줄어들어서 대출금리를 통한 통화정책 파급경로가 원활히 작동하고 있다고 볼 수도 있지만, 통화지표 M1과 M2의 증가율이 상반된 모습을 보이는 가운데, 중소기업에 대한 은행들의 보수적인 대출태도도 그대로 지속되고 있으며, 대기업의 은행 차입금도 줄어들고 있어서 통화정책 파급경로가 제대로 작동하는지 점검해 볼 필요가 있다는 의견을 나타내었음.\n이어서 동 위원은 IMF 자료에 따르면, 아시아 신흥국의 경우 장기금리는 글로벌 요인에 의해 좌우되고, 통화정책은 단기금리를 통해 작동되는데, 이마저도 지역경제의 글로벌 경제에의 편입 정도와 선진국 양적완화정책의 강도 등 외부요인에 의해 더 크게 영향을 받는다는 분석이 있다고 소개한 후, 이를 참고하여 국내 통화정책의 전달경로를 면밀히 살펴볼 필요가 있다는 견해를 제시하였음.\n다른 일부 위원은 MMF를 포함하여 자산운용사가 운용하는 단기상품의 규모는 은행 수시입출식 예금 규모에 비해 작은 수준이지만, 변동성이 크고 금리에 매우 민감하다는 특징을 가지고 있어 시스템적 위험요인이 될 수 있으므로 그 움직임을 예의주시할 필요가 있다는 의견을 밝혔음.\n또한, 동 위원은 최근 은행의 신규취급액기준 예대금리차가 많이 낮아지기는 했지만 과거 금융위기 이전의 최소 수준을 여전히 상회하고 있는데, 향후 추가 하락할 여지가 있는지 물었으며, 이에 대해 관련부서에서는 은행의 수익성과 관련 있는 잔액기준 예대금리차가 현재 글로벌 위기 이전 수준보다 낮은 수준에 있음을 감안할 때 추가로 하락할 여지는 크지 않은 것으로 보인다고 답변하였음.\n아울러 동 위원은 은행 가산금리와 관련하여 은행들이 경기가 부진할 때는 신용리스크 상승이나 수익성 저하에 대처해 가산금리를 높이고, 경기가 좋을 때는 가산금리를 낮추는 경기순응적(procyclical)인 운용행태를 보이고 있지만, 이러한 부분이 불투명하게 되면 문제가 발생할 수 있으므로 투명하게 운용해야 할 필요가 있으며, 정책대응과 관련하여서도 경기부진시 금리를 올리려고 하는 미시적 금융안정과 경기부진시 금리를 내리려고 하는 거시적 금융안정 사이에 상충성이 있을 수 있으므로 이를 해소할 수 있는 수단을 모색해볼 필요가 있다는 의견을 밝혔음.\n한편, 동 위원은 향후 경기부진과 더불어 저금리 기조가 지속될 경우 일부 금융기관의 수익성이 나빠지면서 저신용등급의 회사채 내지 파생상품과 같은 고위험 고수익 상품 매입을 늘릴 가능성이 있으므로 이러한 부분에 대해서도 미리 대비해둘 필요가 있다는 의견을 밝혔음.",
        },
        {
            "id": "BOK_20181130_20181218_S4",
            "filename": "BOK_20181130_20181218",
            "mdate": "2018-11-30 10:00:00",
            "rdate": "2018-12-18 16:00:00",
            "section": "Monetary Policy",
            "text": "일부 위원은 최근 경기상황과 금융불균형 등을 고려할 때 확장적 재정정책의 필요성에는 공감하지만 민간과 정부 부문 간 레버리지 차이가 확장적 재정정책의 충분조건이 되기는 어려워 보인다고 언급하면서, 이에 관해서는 재정과 통화 정책의 최적조합 측면에서도 접근해 볼 필요가 있어 보인다는 견해를 나타내었음.\n이에 대해 관련부서에서는 민간부문에 비해 정부부문의 레버리지가 낮은 점만이 확장적 재정정책 필요성의 충분한 근거라고 할 수는 없지만, 민간부문 레버리지를 안정시키고자 하는 경우 정부부문 레버리지가 높지 않은 점을 고려하면 재정정책의 경기대응 역할을 확대할 수 있는 여력이 있는 것으로 보인다고 답변하였음.\n다른 일부 위원은 매크로 레버리지의 구성과 총량 중 어느 쪽이 더 문제인지, 그리고 민간부문 레버리지의 지속가능성은 어느 정도이며 그렇게 평가할 수 있는 근거는 무엇인지에 대해 먼저 명확히 분석한 후에 정책대응 방식을 논의하는 것이 적절해 보인다는 의견을 제시하였음.\n이어서 동 위원은 최근의 가계부문 레버리지 상승이 어느 정도로 위험한 것인지 가늠해 보려면 과거 주택가격이 상승하였다가 하락한 시기의 레버리지 변화와 충격에 따른 영향 등을 살펴보면서 최근 상황과 비교·분석해 볼 필요가 있다는 견해를 표명하였음.\n또 다른 일부 위원은 현재 가계부채가 관리 가능한 수준이지만 동시에 관리가 필요한 시점이라고 언급하면서, 사전적으로 금융감독 등을 통해 가계부채 증가를 억제하는 데는 한계가 있을 수 있으므로 통화정책을 통해 가계부채가 억제될 수 있는 환경을 조성하는 것이 필요하다는 의견을 피력하였음.\n아울러 동 위원은 기업부문 레버리지가 정부의 혁신성장, 중소기업정책 등의 영향으로 상승할 가능성이 있는지에 대해 분석해 볼 필요가 있으며, 재정정책의 경우 확장적인 방향으로 운용되는 것이 바람직하겠지만 복지지출 확대 등이 장기적으로 재정건전성에 미치는 영향에 대해서도 살펴볼 필요가 있다고 당부하였음.\n한편 일부 위원은 가계부채 결정요인을 모형으로 분석함에 있어 금리의 내생성, 주택가격과의 인과관계, 금리와 가계부채 간의 시차 등을 주의 깊게 고려할 필요가 있으며, 부동산 관련 규제를 나타내는 변수도 좀 더 포괄적으로 반영해야 할 것이라는 의견을 나타내었음.\n다른 일부 위원은 기존 연구들이 제시하는 레버리지의 임계치는 시장에 의해 강제적으로 디레버리징이 시작되는 단계의 레버리지 수준이라기보다는 경제성장을 제약하는 수준으로 해석하는 것이 타당해 보인다고 언급하였음.\n또한 동 위원은 과거 여러 국가의 사례를 살펴보면 민간부문 디레버리징이 통상 정부부채의 증가를 수반하였다는 점에서 민간부문 레버리지를 안정시켜 나가는 과정에서 나타날 수 있는 소비제약 등의 부작용에 정부의 지출여력을 활용하여 대응하는 것은 적절한 것으로 보인다고 덧붙였음.\n또 다른 일부 위원은 기업부문 레버리지 분석 시 중소기업과 대기업을 나누어 접근해 볼 수 있으며, 임계치 분석 등에 국가별 산업구조가 상이한 점 등을 감안할 필요가 있다는 의견을 제시하였음.",
        },
        {
            "id": "BOK_20181130_20181218_S5",
            "filename": "BOK_20181130_20181218",
            "mdate": "2018-11-30 10:00:00",
            "rdate": "2018-12-18 16:00:00",
            "section": "Participants’ Views",
            "text": "일부 위원은 최근 실물경제 성장경로의 하방위험이 다소 커진 것으로 보이고 물가도 상승률의 확대속도가 여전히 완만한 가운데 내년도 경로의 하방위험이 미세하게 커진 모습이므로, 이에 금번 기준금리는 현 수준에서 동결하는 것이 적절하다는 견해를 표명하였음.\n최근 우리 경제의 실물 측면을 살펴보면 대외환경의 불확실성이 한층 높아지는 가운데 민간소비의 기저 회복세에 둔화 조짐이 있어 내년도 성장의 하방위험이 다소 증가한 모습임.\n대외적으로는 선진국의 경우 미국경제는 견조한 성장세를 유지하고 있으나 일본, 유로 경제는 성장흐름이 약화된 모습임.\n미국 이외 선진국의 3/4분기 성장률 둔화에는 독일의 자동차관련 인증절차 지연으로 인한 마이너스 성장 등 일시적 요인의 영향도 있었음.\n그러나 전반적으로는 올해 중반 이후 전산업 PMI 등 투자관련 지표가 꾸준히 둔화추세를 보여온 것이 성장률에 실현되는 것으로 보이며, 이에 기조적인 향후 성장전망이 약화되는 모습임.\n한편 중국경제도 올해 2/4분기 이후 투자가 정체된 흐름을 지속하여 온 것에 더하여 10월 소비지표가 상당 폭 둔화되어 성장세 약화가 뚜렷해지는 모습임.\n세계 주요금융시장과 원자재 시장의 가격동향에서는 내년 및 중기 세계경제 전망과 관련한 불확실성이 최근 크게 높아지고 있음이 관찰됨.\n그 중심에는 미국의 호황 지속에 대한 의문과 더불어 보다 중요하게는 중국의 성장세 둔화 정도, 그리고 이와 상호작용하고 있는 중국과 미국의 무역분쟁 전개방향을 둘러싼 불확실성 상승이 있다고 보임.\n두 요인 모두 현재로서는 향후 전개양상에 대한 예측의 불확실성이 매우 높지만, 그 만큼 내년도 세계경제의 성장과 교역증가율에 하방위험이 다소 커진 상황이라 판단됨.\n이러한 대외환경의 전개가 우리나라 수출에 미칠 수 있는 부정적인 영향은 아직 지표에서는 나타나지 않고 있음.\n9∼10월 수출은 올해 중의 양호한 흐름을 지속하고 있으며 11월에도 비슷한 흐름인 것으로 보임.\n지역별로 보면 신흥국 수출의 증가세는 둔화되었으나 미국 등 선진국으로의 수출이 2/4분기 대비 오히려 반등하며 전체적으로 양호한 증가세를 이끄는 모습임.\n선진국으로의 수출호조는 최근 선진국 실물경기의 흐름과는 상응하지 않는 특이한 현상임.\n최근 수출호조는 교역분쟁의 당사국인 중국에서도 발견되는 현상인데, 무역분쟁 악화를 대비한 수입 의존 선진국 기업들의 선수입이 원인이라는 해석도 제기되고 있음.\n이 가능성을 배제할 수 없으므로 최근 수출호조는 내년 중 반락으로 상쇄될 위험이 있음에도 유의하여야 하겠음.\n내수를 보면 소매판매가 9월 이후 감소세에서 회복되지 못한 것으로 추정되고 서비스업 추이도 보합세로 보여 최근 실적지표는 민간소비의 회복세 둔화를 시사하고 있음.\n민간소비는 작년도 하반기 장기간의 침체에서 회복세로 전환하였고 올해 상반기까지 전년대비 GDP성장률을 상회하였음.\n가계부분의 저축률이 여전히 높게 유지되고 있는 것으로 보여 지난 1년 높았던 민간소비 증가세가 향후에도 유지될 수 있을 것인지에는 상당한 불확실성이 있음.\n최근 지표는 일단 가계소비성향의 회복이 충분하지 않음을 시사하는 방향이며, 이런 의미에서 민간소비의 기조적 회복세는 둔화위험이 커졌다 하겠음.\n향후 실제 소비증가율이 어느 정도 수준에서 실현될지의 여부는 승용차 개별소비세 인하, 정부소비 증가 등 정부의 소비진작정책이 가져올 효과에 의존할 것으로 보임.\n내수의 다른 항목인 투자는 건설투자의 기조적인 하락조정을 배경으로 전반적으로 조정추세가 계속되는 모습임.\n한편 물가는 소비자물가 상승률의 경우 농산물가격 급등 등에 기인해 9월 1.9%에 이어 10월 2.0%를 기록하였음.\n반면 최근 상승요인이 모두 공급측 요인이었음에 따라 근원물가 상승률은 9월 1.0%, 10월 0.9%로 유례없이 낮은 수준을 지속하고 있음.\n올해 중 임금상승세가 상당히 높게 유지되고 있어 물가상승률의 확대를 기대하여 왔으나, 현실의 물가흐름에서는 여전히 상승률의 확대기조는 나타나지 않는 모습임.\n오히려 경기민감품목의 물가상승률이 10월 다소 하락하였고, 올해 자주 거론되었던 관리물가품목을 제외한 근원물가 흐름도 10월 소폭 둔화되었음.\n모두 수요측 물가상승압력은 미세하게 약화됨을 시사하며 최근 실물경기의 경로와도 부합됨.\n조사국은 지난 10월 전망에서 내년도 근원물가 상승률을 1.6%, 내년도 소비자물가 상승률을 1.7%로 예상한 바 있음.\n지난 금통위 회의에서 동 전망치는 하방위험이 있다는 의견을 피력하였는데, 이후 이 평가의견을 수정하여야할 상황 변화는 없었다는 판단임.\n이상 실물경제와 물가의 흐름에 의하는 한 통화정책 금리의 인상을 거론하는 것은 쉽지 않다는 판단임.\n그럼에도 금리인상의 논거를 찾고자 한다면 금융안정에 대한 고려일 수밖에 없을 것으로 보임.\n최근 적지 않은 사회적 공감을 얻고 있는 견해는 가계부채의 증가속도가 과도하므로 이에 대한 통화정책적 대응이 필요하다는 것으로 보임.\n우리나라 가계부채 증가율은 2008년 세계 금융위기 이후 10년의 장기간 꾸준히 소득증가율을 상회하여 왔고 특히 2015년, 2016년은 두 자리수로 증가하였음.\n이 추세는 우려되며 증가속도를 제어할 정책대응이 필요하다는 데 동의함.\n그러나 그 대응수단이 학계와 주요 중앙은행에서 중론으로 되어 있는 건전성정책이 아니라 통화정책이어야 하는 것인지는 의문임.\n특히 우리나라의 경우 다음의 이유로 금융안정은 건전성정책으로 대응하고 통화정책은 인플레이션 타게팅에 충실하게 거시상황에 초점을 두어 운용하는 것이 바람직하다는 의견임.\n최근 가계부채의 추이를 보면 작년에 비해 증가세가 둔화되었음.\n그럼에도 여전히 부채증가 속도는 소득증가 속도를 넘어선 상태에 머무르고 있는데, 최근 부채증가를 주도하고 있는 것은 은행의 전세자금대출과 신용대출임.\n2014∼16년 가계부채 급등기에 집단대출, 전세대출을 제외한 순수한 개별 주택담보대출이 두 자리수 증가율을 기록하며 급증을 주도하였던 것과는 크게 다른 양상임.\n신규분양시장과 연계된 은행 집단대출은 증가율이 작년 이후 둔화세로 전환되었고 신규분양흐름을 볼 때 향후 하락폭이 커질 전망임.\n은행의 전세대출을 제외한 개별 주택담보대출은 증가율 하락세가 한층 빠르게 진행되어 올해 10월 기준 전년동기대비 잔액 증가율은 마이너스, 즉 감소세로 전환되었음.\n신규분양 또는 주택매매와 관련된 대출 증가율이 안정화되는 반면 전세대출과 신용대출이 크게 증가하는 최근 현상은 주목되어야 함.\n보증, DSR규제 등 건전성 규제의 적용이 전세대출 등에 대해 완화 적용되고 있고, 이를 은행이 적극적으로 대출세일즈에 활용하고 있음이 주된 원인이라고 생각됨.\n그렇다면 이에 대한 1차 대응은 미시 건전성정책에 의함이 적당함.\n관련 정책당국도 이 점을 인식하여 지난 9.13대책에는 전세대출에 대한 보증요건 강화가 포함되었으며 이 방안이 10월 15일부터 시행되고 있음.\n이 정책의 효과를 지켜보되 효과가 제한적인 경우 전세대출과 신용대출에 의한 DSR규제 등의 강화방안 검토를 신중히 제안함.\n이상을 종합하면 최근 선진국경제와 중국경제의 성장세 둔화와 무역분쟁의 지속으로 대외환경의 불확실성이 한층 높아지고, 내수의 둔화 조짐도 나타나면서 실물경제 성장경로의 하방위험이 다소 커진 것으로 보임.\n물가도 상승률의 확대속도가 여전히 완만한 가운데 내년도 경로의 하방위험이 미세하게 커진 모습임.\n이에 금번 기준금리는 현 수준에서 동결하는 것이 적절하다는 판단임.\n다른 일부 위원은 경기와 물가 등 실물경제를 고려하여 통화정책의 완화기조는 유지하되, 이번에는 좀 더 금융안정에 중점을 두어 기준금리를 현 1.50%에서 1.75%로 상향 조정하는 것이 바람직하다는 견해를 밝혔음.\n지난 통화정책방향 회의 이후 경제동향을 살펴보면, 세계경제는 미국경제가 견조한 성장흐름을 보였지만, 중국, 유로지역, 일본의 경우 성장세가 주춤한 모습임.\n국제유가가 큰 폭의 하락세를 보인 가운데 글로벌 교역증가세도 소폭 둔화되는 듯 하나, 이러한 세계 성장 및 교역 둔화는 일시적 요인에 기인하는 바도 있어 전체적으로는 아직까지 지난 10월 전망경로를 크게 이탈하지 않고 있는 것으로 보임.\n국제금융시장에서는 미국 등 선진국의 주식, 외환시장 변동성이 확대되었으나, 신흥국 금융불안은 일부 안정세를 시현하였음.\n국내경제는 민간소비가 고용 부진에도 불구하고 10월 전망 수준에서 크게 벗어나지 않는 증가세를 이어가고, 수출은 반도체수요 호조 등으로 높은 증가세를 나타내었음.\n최근 반도체 가격이 조정되는 움직임을 나타내는 등 전기전자업종 중심의 우리 수출에 대한 우려도 꾸준히 제기되고 있어 향후 동향을 면밀히 분석해야 하겠음.\n설비 및 건설투자의 경우 조정양상을 이어가고 있음.\n우리의 주요산업은 내년에도 전망이 밝지 않고 주요국대비 경쟁력 약화가 우려된다는 지적이 꾸준히 제기되고 있음.\n중장기적인 성장잠재력 확충을 위해서는 산업의 구조적 문제에 대한 검토가 있어야 할 것임.\n특히 자동차, 조선 등 구조조정 과정에 있는 산업들의 전망과 부동산시장 안정화 정책 등이 건설업과 투자에 미칠 영향에 대해서도 점검해야 한다고 봄.\n고용은 10월 중 취업자수 증가규모가 소폭 확대되는 등 다소 호전되었으나, 내년에도 크게 개선되기는 어려운 상황임.\n고용이 일부 개선된 것은 주로 정부의 일자리지원정책 등에 기인한 보건복지업, 그리고 아파트 입주물량 증가에 따른 건설업의 고용 증가에 힘입은 바 크고, 제조업과 도소매, 음식숙박업 등에서의 부진은 지속되고 있기 때문임.\n이와 같이 경기와 고용상황이 10월 전망한 수준에서 크게 벗어나지 않고 있는 것으로 보이나, 경기지표와 심리지표의 하락세 지속에서 보듯 성장의 하방위험에 대한 우려도 커지고 있다 하겠음.\n금년과 내년의 물가도 대체로 10월 전망 수준과 부합할 것으로 예상됨.\n소비자물가 상승률은 2% 물가목표에 더 근접하였으나, 근원인플레이션율은 1% 내외 수준에 머무르고 있음.\n앞으로는 급등 후 급락세를 보이고 있는 국제유가와 일부 공공요금 인상 추이 등을 면밀히 지켜보며 기조적 물가흐름을 판단해 나가야 하겠음.\n이와 함께 지난 8월 이후 하락추세를 보이고 있는 기대인플레이션율에 대한 모니터링도 필요할 것임.\n실물경제 상황을 고려하면 GDP갭이 소폭이나마 마이너스일 가능성이 있는 것으로 추정되고, 수요 측면에서의 물가상승압력도 크지 않을 것으로 보여 완화적 통화정책기조는 앞으로도 당분간 유지해야 할 것으로 판단됨.\n다만, 그동안 지적해온 대로 지금은 금융안정에 보다 유의해야 할 시점이라고 생각함.\n금융불균형 누적문제에 대한 대응과 관련, 가계부채는 정부정책 등의 영향으로 안정세를 보일 것으로 예상하지만, 여전히 증가속도가 소득증가율을 웃돌고 있음.\n금리가 인상될 경우 취약차주 등의 이자상환부담이 더 커질 우려가 있으나 스트레스 테스트 결과에 비추어 볼 때 감내가능한 수준으로 추정됨.\n정부의 적극적인 서민금융과 취약차주 대책 등도 이와 같은 취약차주 부담완화에 기여할 것으로 기대함.\n마지막으로, 대외부분 금융안정과 관련해서는 건실한 거시경제상황, 높은 대외신인도 등을 감안할 때, 한·미 간 금리역전이 당장 대규모 자본유출로 이어지지는 않을 것으로 여겨짐.\n다만 미 연준의 금리인상에 따라 경제여건이 취약하고 대외부채가 많은 일부 신흥국들의 불안이 이어질 수 있고, 미·중 무역분쟁의 진행과정에서 중국관련 리스크가 부각될 우려도 있음.\n주요국 통화정책 정상화, 미·중 무역분쟁 등과 관련한 다양한 시나리오에 대해 검토해야 할 것임.\n이상의 상황을 종합해보면 경기와 물가 등 실물경제를 고려하여 통화정책의 완화기조는 유지하되, 이번에는 좀 더 금융안정에 중점을 두어 기준금리를 현 1.50%에서 1.75%로 상향 조정하는 것이 바람직하다고 판단함.\n또 다른 일부 위원은 금번 회의에서는 기준금리를 1.50%에서 1.75%로 인상하는 것이 바람직하다는 견해를 나타내었음.\n글로벌 시장환경이 지속적으로 급변하고 국내적으로는 고령화가 빠르게 진행되고 있어서 우리 경제에 대한 근본적 구조적 대응이 요구되는 상황임.\n이런 새로운 국면을 접하며 우리 경제는 지난 수년 간 저성장 기조 하에 낮은 수준의 경기변동성을 유지하며 횡보하고 있음.\n또한 이와 연관되어 경기적 선순환이 원활하지 않아 완화적 통화정책이 내수는 진작하지만 비효율적 자원배분을 수반하는 경향이 있어 금융불균형을 확대시키고 있는 것으로 판단됨.\n그러므로 물가의 수요압력이 미약하나마 유지되고 있는 현 상황에서 저축과 투자유인을 중립적 금리수준 방향으로 조정하여 금융불균형을 억제하는 것이 바람직하다고 생각됨.\n아울러 취약계층에 대한 재정지출이 확대되는 시기와 맞물리기 때문에 총수요 압력에는 큰 변화가 없을 것으로 예상됨.\n물가를 보면 관리물가를 제외한 물가지수는 상승 움직임을 나타내고 있어서 물가의 수요압력은 유지되고 있는 것으로 평가됨.\n10월 인플레이션은 석유류제품과 농축수산물의 상승에 힘입어 물가목표치 2.0%에 도달하였음.\n근원물가와 개인서비스물가의 상승률이 둔화되었지만 이는 관리품목인 학교급식비, 병원검사료 등의 하락에 주로 기인함.\n동 품목들을 제외할 경우 개인서비스물가의 상승률은 2.2%에서 2.7%로, 그리고 근원물가 상승률은 0.9%에서 1.4%로 각각 상향 조정됨.\n반면 금융불균형은 계속 누증되고 있는 것으로 판단됨.\n금융불균형은 내수 증대에 소요된 자원이 시간을 두고 지속가능한 부가가치 창출로 이어지지 않은 만큼 낭비된 부분이라고도 정의할 수 있겠음.\n금융시장에서의 금융불균형은 실물경제의 펀더멘탈 이상의 수익추구(search for yield)에 따른 위험에 대한 미스프라이싱(mispricing)에서 출발하며, 결국 금융기관의 대차대조표에 축적됨.\n하지만 그 정도를 측정하는 데에는 현실적으로 어려움이 있음.\n이는 미시데이터의 부족뿐만 아니라 과도한 담보활용 관행, 풍부한 유동성 상황에서 공정가액 측정의 애로, 그리고 금융상품 간의 복잡한 연계성 확대로 인한 거래상대방 위험 추적의 어려움 등에 기인함.\n한편 실물경제 측면에서는 소득대비 부채의 과도한 확대로 금융불균형을 상대적으로 용이하게 가늠할 수 있음.\n최근 우리 경제의 불균형은 부동산시장으로의 쏠림현상에서 잘 드러남. 고령화에 대비한 저축 부족이 부각되면서 임대수익이나 자산가격 차익 추구를 목적으로 레버리지를 통한 부동산투자가 중장년층을 중심으로 확대되었음.\n여기에 차입제약이 완화되면서 주택마련이나 업그레이드의 기회를 노리는 유인들이 합세하며, 지역별로 큰 편차가 있어 공급이 제한된 일부 수도권지역에서는 가파른 가격상승도 있었지만, 전국적으로는 주택의 과잉공급이 발생하였음.\n이로 말미암아 미분양이 확대되고 전세 및 월세 가격이 하락하고 있음.\n아울러 주거용 부동산에 대한 규제가 강화된 2017년 이후에는 상업용 부동산시장에 대한 금융기관의 대출이 빠르게 증가하고, 자산가격 상승에 대한 기대로 수익률이 예금금리를 크게 상회하며 투자유인으로 작용하였음.\n그 결과, 주택시장과 유사하게 과잉공급이 나타났고, 전년도에 잠시 하락하였던 오피스 공실률이 다시 상승세로 전환되었음.\n특히 상업용 부동산의 임차인이 자영업자 중심으로 증가하고 있어서 향후 업황이 침체될 경우 관련 대출의 부실화가 우려됨.\n일부 실증분석에서 중·고위 기술 제조업에서만 지속적인 순고용확대 효과가 발생하는 것으로 나타나듯이, 기술적 배경없이 단기적 투자유인만을 통한 창업 활성화는 중기적으로 더 큰 사회적 비용을 초래할 수 있음에 유의해야 함.\n정부의 다양한 규제정책으로 인하여 주택시장은 다소 안정을 찾고 있는 것으로 판단됨.\n다만 누적된 금융불균형이 근본적으로 해소되기 위해서는 부채확대에 따른 이자부담을 실질소득 증가로 상환할 수 있는 역량이 형성되어야 함.\n현재로서는 우리 경제의 구조적 상태와 상기 언급한 부동산의 과잉공급 상황으로 보아 부채증가가 잠재성장률 확대를 견인하지 못하고 있는 것으로 보임.\n이는 결국 상대적 가격조정, 즉 인플레이션 확대나 부실채권 증가로 이어질 가능성이 크다는 것임.\n결론적으로 물가압력이 미약하나마 유지되고 있는 가운데 금융불균형의 누적이 지속되는 현 상황에서는 근본적 구조조정과 더불어 투자유인을 중립금리 방향으로 조정해 나가야 한다고 생각됨.\n이를 통해 효율성을 증대하고 풍선효과를 선제적으로 차단하여 중장기적 경제손실을 축소할 수 있을 것으로 생각함.\n그러므로 본 회의에서는 기준금리를 1.50%에서 1.75%로 인상하는 것이 바람직하다고 판단함.\n한편 일부 위원은 기준금리를 현 수준에서 유지하는 것이 바람직하다는 견해를 밝혔음.\n우리 경제의 성장세는 2/4분기 이후 전분기 대비 0.6%, 즉, 연율 2%대 중반 수준으로 서서히 둔화되어 왔음.\n성장의 내용도 반도체 수출과 재정지출에 의존하는 다소 불안한 모습이며, 민간부문에 의한 고용은 매우 저조한 상태임.\n향후 경기도 낙관하기 어려움.\n호조세를 유지해 온 세계경제의 산업생산과 교역량 증가율, 제조업심리지수와 경기선행지수 등 대부분의 경기지표들이 일제히 정점을 지나 둔화추세로 전환되고 있음.\n특히 중국의 경기 둔화가 무역분쟁 및 부채조정 과정에서 심화될 위험이 있어, 중국과의 관계가 밀접한 우리 경제에 부정적인 요인으로 작용할 가능성도 배제할 수 없음.\n내수의 경우에도 설비투자와 건설투자의 조정이 지속되는 가운데, 상대적으로 양호한 모습을 보여 왔던 민간소비의 증가세도 완만해지는 모습임.\n임금상승률의 확대에도 불구하고 고용 부진과 자영업 사업소득 감소, 그리고 비소비지출 급증 등이 처분가능소득의 증가와 민간소비를 제약하는 상황에서, 정부소비만 높은 증가세를 유지하는 불균형적 상황이 지속되고 있음.\n내수 부진을 반영하여 기조적 물가상승률은 낮은 수준에 머물러 있음.\n근원물가 상승률은 1% 내외까지 하락하였으며, 여기에서 관리물가를 제외해도 1%대 중반의 상승률에 머물러 있음.\n특히 근원물가 상승의 주된 기여항목인 개인서비스 가격의 경우에도 노동집약적 품목의 가격이 상대적으로 많이 오른 것으로 나타나고 있어, 수요 측면의 압력보다는 노동비용 상승 충격에 상당한 영향을 받고 있음을 시사함.\n향후에도 경기여건이 낙관적이지 않다는 점을 감안할 때, 기조적인 물가가 빠르게 상승할 가능성은 높지 않은 것으로 판단됨.\n전체 소비자물가 상승률의 경우에는 전년동기대비 유가상승과 기저효과 등에 의해 당분간 2% 내외를 기록할 것으로 보이나, 일시적 요인들이 해소되면서 점차 하락하여 근원물가 상승률에 수렴해 갈 것으로 예상됨.\n금년 중반 이후 큰 폭으로 하락한 국채금리는 이와 같은 거시경제 하방위험에 대한 시장의 우려를 반영하고 있는 것으로 보임.\n최근 우리나라의 금리는 기준금리뿐만 아니라 모든 만기의 국채금리가 동일한 만기의 미국금리를 상당 폭 하회함.\n장단기 금리의 격차도 경기 둔화 신호여부 논란을 촉발하고 있는 미국의 장단기 금리격차와 비슷한 수준까지 축소되고 있음.\n미국과의 금리 역전에도 불구하고 환율 및 외환시장이 안정적인 모습을 유지하고 있는 것은 금융시장이 우리 경제의 대외건전성보다 내부적인 거시경제 위축 가능성을 더욱 우려하고 있음을 나타내는 것으로 해석됨.\n한편, 지난 수년 간 큰 폭으로 증가한 가계부채가 여전히 소득보다 빠른 속도로 증가하고 있어 대표적인 금융불균형 문제로 지적되고 있음.\n다만 다양한 스트레스 테스트들은 현재의 가계부채가 주요 금융기관의 건전성 훼손을 통해 금융시장 전반을 위협할 가능성은 높지 않은 것으로 평가하고 있음.\n아울러 단기적 등락은 있으나 작년 이후 가계부채 증가세는 둔화되는 추세에 있으며, 동 추세가 지속된다면 내년에는 소득증가율을 크게 상회하지 않는 수준까지 둔화될 가능성이 있는 것으로 보임.\n특히 부동산 관련 대출을 적극적으로 제한한 9.13 대책과 DSR 규제의 전면 시행이 효과적으로 정착된다면, 그 가능성은 커질 것임.\n이와 같은 여건들을 감안할 때, 기준금리를 현 수준에서 유지하는 것이 바람직하다고 생각함.\n최근 시행된 강력한 거시건전성 대책이 가계부채 증가세를 둔화시킬 것으로 예상되고 있는 현 시점에, 그 둔화추세를 가속화시키기 위한 금리인상의 필요성 혹은 경제·사회적 편익에 대해서는 논란의 여지가 있는 반면, 거시경제의 하방위험 확대라는 금리인상의 비용은 명확해 보임.\n아울러 금융자원의 쏠림현상과 같은 미시적 자원배분의 비효율성보다 높은 실업률과 낮은 가동률로 대표되는 총량적 차원의 유휴 실물자원에 의해 초래되는 비용을 축소시키는 것이, 그리고 이를 통해 궁극적으로 경기 및 물가를 안정시키는 것이, 거시경제 정책으로서 통화정책에 기대되는 본연의 역할인 것으로 이해하고 있음.\n따라서 기조적 물가상승률이 목표수준을 지속적으로 하회하고 경기 둔화에 대한 우려가 점증하는 현 시점에서의 통화정책은 금융불균형 완화보다 거시경제의 하방위험 완충에 초점을 두고 수행되는 것이 바람직하다고 판단함.\n다른 일부 위원은 금번 회의에서는 기준금리를 현재의 1.50%에서 1.75%로 상향 조정하는 것이 바람직하다는 견해를 나타내었음.\n지난 통화정책방향 결정회의 이후 경제여건 변화를 살펴보면, 대내외 금융시장의 변동성이 크게 확대되었으나 실물경제의 움직임은 비교적 안정적이어서 10월 조사국 전망경로에서 크게 벗어나지 않고 있음.\n세계경제의 경우 미국을 중심으로 잠재성장률을 웃도는 성장세를 지속하고 있는 것으로 추정됨.\n유로지역 성장세가 예상을 하회하고 있지만, 일본경제는 일시적 부진에서 회복되고 아시아 신흥국의 제조업 관련 지표들도 개선되는 모습임.\n국내경제는 잠재성장률 수준의 성장세를 유지하고 있으나 성장의 동인이 주로 수출에 집중되어 있는 가운데 투자의 조정국면이 장기화되고 있음.\n최근 발표된 경제지표를 기초로 판단해 볼 때, 건설투자의 부진은 앞으로도 한동안 계속될 것이지만 설비투자의 하락세가 진정되는 가운데 소비는 완만한 상승세를 지속할 것으로 보임.\n이에 따라 수출과 내수의 격차는 다소 축소될 것으로 예상됨.\n금융안정 상황을 보면 가계부채 증가세가 둔화되고 있으나 소득증가율을 상회하고 있고 10월에는 규제 강화를 앞둔 선수요가 더해지면서 증가율이 다소 높아졌음.\n부동산 및 가계대출 규제가 지속적으로 강화되고 있어 주택담보대출 증가율이 크게 확대될 가능성은 높지 않은 것으로 보이나, 다른 부문으로의 풍선효과 가능성은 배제할 수 없음.\n실제 규제의 영향력이 상대적으로 크지 않은 가계의 신용대출이나 개인사업자대출의 증가세는 최근까지도 높은 수준을 유지하고 있는 것으로 보임.\n수 차례의 미시적 규제에도 불구하고 민간부문의 레버리지가 계속 상승해 오고 있다는 것은 어느 정도는 완화적 금융상황을 배경으로 작지 않은 대출수요가 잠재해 있음을 시사하는 것으로 판단됨.\n이에 따라 성장과 물가 흐름이 크게 미흡하지 않다면 통화정책 완화기조의 적절성을 판단함에 있어 금융불균형 누적 가능성에 대한 보다 비중 있는 고려가 필요하다고 생각됨.\n물론 향후 국내외 경제의 성장경로 상에 있어 불확실성은 여전히 높은 상황임.\n미·중 무역갈등, 주요국 통화정책 정상화, 유로지역 정치상황 등의 불안요인이 상존하는 가운데 최근 금융시장에서는 글로벌 경기국면의 전환 가능성에 대한 우려가 관심을 받고 있음.\n지난 수년간 세계경제 성장의 버팀목이 되었던 미국 경기가 이미 성숙국면에 진입했고 중국경제의 성장세도 구조적으로 둔화되고 있다는 것이 많이 인용되고 있는 논거임.\n반면 세계경제의 성장세가 완만해지는 것은 지난 2년 동안 잠재성장률을 큰 폭으로 상회했던 성장세가 보다 지속 가능한 수준으로 정착하기 위한 자연스러운 조정이라고 보는 견해도 상당함.\n과거 사례를 볼 때, 세계경제가 본격적인 둔화국면으로 진입하게 될 가능성을 제한하는 여러 완충 요소들이 아직 유효함을 주목할 필요가 있음.\n무엇보다 글로벌 유동성이 실물경기에 여전히 우호적인 상황임.\n주요국 통화정책 정상화가 꾸준히 지속되고 일부 신흥국의 금융상황이 긴축적으로 돌아서고는 있지만, 세계경제 전체적으로 본다면 글로벌 통화정책과 금융상황은 여전히 완화적인 것으로 추정됨.\n또한 주요국 노동시장 상황이 빠르게 개선되면서 실업률이 크게 낮아졌지만 아직은 임금 상승 정도가 생산성 증가를 추월하지 않고 있어 기업수익 전망을 크게 악화시키지 않고 있음.\n그 외에도 제조업 재고수준이 크게 부담스럽지 않다는 점이나 대외건전성이 양호한 국가들 위주로 내수 부양정책이 나오고 있다는 점 등을 고려한다면 세계경제 성장률이 어느 정도의 조정과정을 거칠 수는 있겠으나 내년에도 양호한 수준을 지속할 것이라는 전망이 여전히 지지되고 있는 것으로 판단됨.\n국내경제의 경우 여러 구조적 제약과 미시적 정책 변동으로 인해 경기전망의 시계를 길게 갖고 가기가 쉽지 않으나, 세계경제 호조에 힘입은 수출 증가세 지속과 확장적 재정정책 운용으로 조사국 전망치에 부합하는 2%대 중후반의 성장세가 유지될 것으로 보임.\n다만 지난 10월 이후 금융시장의 변동성이 재차 확대되고 있다는 점에서 경기 판단 및 전망에 있어 계속적 주의가 필요함.\n금융시장의 높은 변동성이 장기화될 경우 경제주체들의 불안심리를 자극할 수 있고 이는 다시 자기실현적 과정을 통해 실물경제에 부정적으로 작용하는 경향이 있기 때문임.\n물가상황을 보면 소비자물가 상승률은 3/4분기 중 1.6%에서 10월에 2.0%로 상승한 반면 근원인플레이션율은 3/4분기 1.0%에서 10월 0.9%로 소폭 하락하였음.\n최근 소비자물가 상승세 확대는 농산물가격에 따른 상방압력이, 근원인플레이션율 둔화는 한시적 자동차세 인하 등과 같은 정책적 요인에 의한 하방압력이 작용하였던 것으로 판단됨.\n이와 같이 여러 불규칙 요인들이 상반된 방향으로 영향을 미치고 있어 기저물가 추이를 판단하는 것이 쉽지는 않으나, 중장기적 측면에서 보면 지난 수년 간 물가상승 흐름을 제한해 왔던 요인들이 대체적으로 후퇴해 오고 있음을 주목하고 있음.\n최근의 국제유가 하락과 한시적 유류세 인하 등으로 소비자물가 오름세가 당분간 조정될 것으로 보이나, 일시적·정책적 요인들의 영향이 상대적으로 적은 품목들의 가격 움직임을 참고해 볼 때 소비자물가는 중기적으로 물가목표에 다시 근접해갈 것으로 예상됨.\n상기 요소들을 종합적으로 고려할 때 통화정책의 완화적 기조를 다소 축소하는 것이 적절하다고 생각함.\n대내외 경제환경이 유동적이긴 하지만 국내경제의 성장률 추이가 전망경로에서 크게 이탈하지는 않을 것으로 판단되며, 물가상승률도 여러 불규칙 요인들로 인해 단기적으로는 변동성이 확대될 수 있지만 중장기적으로는 점차 목표치에 근접한 수준에 정착할 것으로 전망되고 있음.\n또한 통화정책 완화 정도를 점진적으로 조정하는 것은 금융불균형 확대에 대한 우려를 제한함과 동시에 보다 장기적인 시계에서 통화정책 여력을 확보하는 데에도 도움이 될 것으로 기대하고 있음.\n이에 따라 이번에는 기준금리를 현 1.50%에서 1.75%로 인상하는 것이 바람직하다고 판단함.\n또 다른 일부 위원은 금번 회의에서는 기준금리를 현재의 1.50%에서 1.75%로 상향 조정하는 것이 바람직하다는 견해를 표명하였음.\n먼저 지난 10월 18일 통화정책방향 결정회의 이후의 여건 변화를 감안하여 최근의 경제상황을 살펴보면, 경기상황은 월별 데이터의 기복은 있지만 큰 흐름은 투자의 큰 폭 조정에도 불구하고 수출이 호조를 지속하고 소비도 꾸준한 증가세를 이어가면서 대체로 잠재수준에서 크게 벗어나지 않는 성장세가 유지되고 있는 것으로 판단됨.\n다만 대외적으로 미·중 무역분쟁과 중국의 성장세 둔화에 대한 우려, 국내적으로 일부 업종 불안과 고용 부진 지속 등 경제여건의 불확실성이 높아진 상황임.\n물가상황을 보면 소비자물가 상승률이 목표수준의 오름세를 나타내고 있음.\n최근의 기조적 물가흐름과 유가하락 등을 감안할 때 향후 소비자물가 상승률은 다소 낮아지겠지만 목표수준에서 크게 벗어나지 않는 1%대 중후반 수준에서 등락할 것으로 예상됨.\n국내 금융시장은 10월 중 대외 리스크 증대 등으로 주식시장을 중심으로 변동성이 크게 높아졌다가 11월 들어 다소 진정되는 모습임.\n기업의 자금조달 여건도 양호한 상황이 지속되고 있음.\n다만 주요 대내외 리스크 요인이 상존하고 있고 이에 따라 신용위험에 대한 경계감도 높아질 가능성에 유의해야 하겠음.\n금융안정 측면에서는 최근 정부대책 등으로 주택가격 상승 기대가 크게 약화되었음에도 불구하고 10월 중 은행과 비은행 가계대출이 금년 들어 가장 큰 규모로 증가하였음.\n물론 향후 가계대출은 그동안의 증가 폭 둔화추세를 이어갈 것으로 보이나 여전히 소득증가율을 상회할 것으로 예상되는 만큼, 금융불균형 누적 위험은 쉽게 가시지 않을 것으로 보임.\n다음은 이에 대응한 통화정책방향에 대해 살펴보면, 그동안의 통화정책방향 커뮤니케이션은 성장과 물가상황이 각각 잠재수준과 목표수준에서 크게 벗어나지 않는다면 금융안정 상황에 보다 유의하여 완화기조를 축소 조정하는 것이 바람직하다는 것임.\n앞서 언급한 바와 같이 최근의 경제․금융상황은 이러한 조건에 부합된다고 판단됨.\n또한 기준금리가 인상되더라도 이는 시장의 기대와 부합할 뿐 아니라 완화기조 범위 내에서 소폭 조정이라는 점에서 금융시장과 실물경제에 대한 부정적 영향이 제한적일 것으로 보임.\n반면 금융안정 측면에서는 그동안의 각종 거시건전성 정책 대응과 보완적으로 작용하면서 가계부채 누증 등 금융불균형 위험을 완화시키는 긍정적 효과가 상당히 있을 것으로 기대됨.\n따라서 이번 금통위 회의에서는 기준금리를 현재의 1.50%에서 1.75%로 상향 조정하는 것이 좋겠다는 의견임.\n이는 보다 먼 시계에서의 통화정책 여력 확보, 그리고 미 연준 정책금리와의 격차 확대에 따른 일반의 불안심리 완화 차원에서도 바람직하다고 생각함.",
        },
    ],
    "fomc_minutes": [
        {
            "date": "1993-02-03",
            "speaker": "Alan Greenspan",
            "title": "FOMC Meeting Minutes",
            "text": "A meeting of the Federal Open Market Committee was held in the offices of the Board of Governors of the Federal Reserve System in Washington, D.C., on Tuesday, February 2, 1993, at 2:30 p.m. and was continued on Wednesday, February 3, 1993, at 9:00 a.m.\n\nPRESENT:\n\nMr. Greenspan, Chairman Mr. Corrigan, Vice Chairman Mr. Angell Mr. Boehne Mr. Keehn Mr. Kelley Mr. LaWare Mr. Lindsey Mr. McTeer Mr. Mullins Ms. Phillips Mr. Stern\n\nMessrs. Broaddus, Jordan, Forrestal, and Parry, Alternate Members of the Federal Open Market Committee\n\nMessrs. Hoenig, Melzer, and Syron, Presidents of the Federal Reserve Banks of Kansas City, St. Louis, and Boston, respectively\n\nMr. Kohn, Secretary and Economist Mr. Bernard, Deputy Secretary Mr. Coyne, Assistant Secretary Mr. Gillum, Assistant Secretary Mr. Mattingly, General Counsel Mr. Patrikis,1 Deputy General Counsel Mr. Prell, Economist Mr. Truman, Economist\n\nMessrs. R. Davis, Lang, Lindsey, Promisel, Rosenblum, Scheld, Siegman, Simpson, and Slifman, Associate Economists\n\nMr. McDonough, Manager of the System Open Market Account\n\nMs. Greene, Deputy Manager for Foreign Operations\n\nMs. Lovett,2 Deputy Manager for Domestic Operations\n\nMr. Ettin, Deputy Director, Division of Research and Statistics, Board of Governors\n\nMr. Stockton, Associate Director, Division of Research and Statistics, Board of Governors\n\nMr. Madigan, Assistant Director, Division of Monetary Affairs, Board of Governors\n\nMr. Brady,3 Section Chief, Division of Monetary Affairs, Board of Governors\n\nMr. Rosine,3 Senior Economist, Division of Research and Statistics, Board of Governors\n\nMr. Wiles,4 Secretary of the Board, Office of the Secretary, Board of Governors\n\nMr. Winn,4 Assistant to the Board, Office of Board Members, Board of Governors\n\nMs. Werneke,4 Special Assistant to the Board, Office of Board Members, Board of Governors\n\nMr. Siciliano,4 Special Assistant to the General Counsel, Legal Division, Board of Governors\n\nMs. Low, Open Market Secretariat Assistant, Division of Monetary Affairs, Board of Governors\n\nMessrs. Beebe, T. Davis, Dewald, Goodfriend, and Ms. Tschinkel, Senior Vice Presidents, Federal Reserve Banks of San Francisco, Kansas City, St. Louis, Richmond, and Atlanta, respectively\n\nMr. McNees, Vice President, Federal Reserve Bank of Boston\n\nMr. Gavin, Assistant Vice President, Federal Reserve Bank of Cleveland\n\nMr. Weber, Senior Research Officer, Federal Reserve Bank of Minneapolis\n\nMs. Meulendyke, Manager, Open Market Operations, Federal Reserve Bank of New York\n\n1. Attended Wednesday session only.\n\n2. Attended Tuesday session only.\n\n3. Attended portion of meeting relating to the Committee's discussion of the economic outlook and its longer-run objectives for monetary and debt aggregates.\n\n4. Attended portion of the meeting relating to the release of FOMC information to the public.\n\nThe Secretary reported that advices of the election of the Reserve Bank members and alternate members of the Federal Open Market Committee for the period commencing January 1, 1993, and ending December 31, 1993, had been received and that these individuals had executed their oaths of office. The elected members and alternate members were as follows:\n\nE. Gerald Corrigan, President of the Federal Reserve Bank of New York, with James H. Oltman, First Vice President of the Federal Reserve Bank of New York, as alternate;\n\nEdward G. Boehne, President of the Federal Reserve Bank of Philadelphia, with J. Alfred Broaddus, Jr., President of the Federal Reserve Bank of Richmond, as alternate;\n\nSilas Keehn, President of the Federal Reserve Bank of Chicago, with Jerry L. Jordan, President of the Federal Reserve Bank of Cleveland, as alternate;\n\nRobert D. McTeer, Jr., President of the Federal Reserve Bank of Dallas, with Robert P. Forrestal, President of the Federal Reserve Bank of Atlanta, as alternate;\n\nGary H. Stern, President of the Federal Reserve Bank of Minneapolis, with Robert T. Parry, President of the Federal Reserve Bank of San Francisco, as alternate.\n\nBy unanimous vote, the Committee elected the following officers of the Federal Open Market Committee to serve until the election of their successors at the first meeting of the Committee after December 31, 1993, with the understanding that in the event of the discontinuance of their official connection with the Board of Governors or with a Federal Reserve Bank, they would cease to have any official connection with the Federal Open Market Committee:\n\nBy unanimous vote, the Federal Reserve Bank of New York was selected to execute transactions for the System Open Market Account until the adjournment of the first meeting of the Committee after December 31, 1993.\n\nBy unanimous vote, William J. McDonough, Margaret L. Greene, and Joan E. Lovett were selected to serve at the pleasure of the Committee in the capacities of Manager of the System Open Market Account, Deputy Manager for Foreign Operations, System Open Market Account, and Deputy Manager for Domestic Operations, System Open Market Account respectively, on the understanding that their selection was subject to their being satisfactory to the Federal Reserve Bank of New York.\n\nSecretary's note: Advice subsequently was received that the selections indicated above were satisfactory to the board of directors of the Federal Reserve Bank of New York.\n\nOn January 15, 1993, the continuing rules, regulations, authorizations, and other instruments of the Committee listed below were distributed with the advice that, in accordance with procedures approved by the Committee, they were being called to the Committee's attention before the February 2-3 organization meeting to give members an opportunity to raise any questions they might have concerning them. Members were asked to indicate if they wished to have any of the instruments in question placed on the agenda for consideration at this meeting. No requests for substantive consideration were received.\n\nAt the meeting; the Committee voted unanimously to update the references to the Management of the System Open Market Account that were contained in the following: (1) Procedures for allocation of securities in the System Open Market Account and (2) Program for Security of FOMC Information. Apart from the indicated updating of titles, all of the instruments listed below remained in effect in their existing forms.\n\n1. Procedures for allocation of securities in the System Open Market Account.\n\n2. Authority for the Chairman to appoint a Federal Reserve Bank as agent to operate the System Account in case the New York Bank is unable to function.\n\n3. Resolution of FOMC to provide for the continued operation of the Committee during an emergency; Resolution of FOMC authorizing certain actions by Federal Reserve Banks during an emergency.\n\n4. Resolution relating to examinations of the System Open Market Account.\n\n5. Guidelines for the conduct of System operations in Federal agency issues.\n\n6. Regulation relating to Open Market Operations of Federal Reserve Banks.\n\n7. Program for Security of FOMC Information.\n\n8. Federal Open Market Committee Rules.\n\nBy unanimous vote, the Authorization for Domestic Open Market Operations, as shown below, was reaffirmed:\n\n1. The Federal Open Market Committee authorizes and directs the Federal Reserve Bank of New York, to the extent necessary to carry out the most recent domestic policy directive adopted at a meeting of the Committee:\n\n(a) To buy or sell U. S. Government securities, including securities of the Federal Financing Bank, and securities that are direct obligations of, or fully guaranteed as to principal and interest by, any agency of the United States in the open market, from or to securities dealers and foreign and international accounts maintained at the Federal Reserve Bank of New York, on a cash, regular, or deferred delivery basis, for the System Open Market Account at market prices, and, for such Account, to exchange maturing U. S. Government and Federal agency securities with the Treasury or the individual agencies or to allow them to mature without replacement; provided that the aggregate amount of U. S. Government and Federal agency securities held in such Account (including forward commitments) at the close of business on the day of a meeting of the Committee at which action is taken with respect to a domestic policy directive shall not be increased or decreased by more than $8.0 billion during the period commencing with the opening of business on the day following such meeting and ending with the close of business on the day of the next such meeting;\n\n(b) When appropriate, to buy or sell in the open market, from or to acceptance dealers and foreign accounts maintained at the Federal Reserve Bank of New York, on a cash, regular, or deferred delivery basis, for the account of the Federal Reserve Bank of New York at market discount rates, prime bankers acceptances with maturities of up to nine months at the time of acceptance that (1) arise out of the current shipment of goods between countries or within the United States, or (2) arise out of the storage within the United States of goods under contract of sale or expected to move into the channels of trade within a reasonable time and that are secured throughout their life by a warehouse receipt or similar document conveying title to the underlying goods; provided that the aggregate amount of bankers acceptances held at any one time shall not exceed $100 million;\n\n(c) To buy U. S. Government securities, obligations that are direct obligations of, or fully guaranteed as to principal and interest by, any agency of the United States, and prime bankers acceptances of the types authorized for purchase under l(b) above, from dealers for the account of the Federal Reserve Bank of New York under agreements for repurchase of such securities, obligations, or acceptances in 15 calendar days or less, at rates that, unless otherwise expressly authorized by the Committee, shall be determined by competitive bidding, after applying reasonable limitations on the volume of agreements with individual dealers; provided that in the event Government securities or agency issues covered by any such agreement are not repurchased by the dealer pursuant to the agreement or a renewal thereof, they shall be sold in the market or transferred to the System Open Market Account; and provided further that in the event bankers acceptances covered by any such agreement are not repurchased by the seller, they shall continue to be held by the Federal Reserve Bank or shall be sold in the open market.\n\n2. In order to ensure the effective conduct of open market operations, the Federal Open Market Committee authorizes and directs the Federal Reserve Banks to lend U. S. Government securities held in the System Open Market Account to Government securities dealers and to banks participating in Government securities clearing arrangements conducted through a Federal Reserve Bank, under such instructions as the Committee may specify from time to time.\n\n3. In order to ensure the effective conduct of open market operations, while assisting in the provision of short-term investments for foreign and international accounts maintained at the Federal Reserve Bank of New York, the Federal Open Market Committee authorizes and directs the Federal Reserve Bank of New York (a) for System Open Market Account, to sell U. S. Government securities to such foreign and international accounts on the bases set forth in paragraph l(a) under agreements providing for the resale by such accounts of those securities within 15 calendar days on terms comparable to those available on such transactions in the market; and (b) for New York Bank account, when appropriate, to undertake with dealers, subject to the conditions imposed on purchases and sales of securities in paragraph l(c), repurchase agreements in U. S. Government and agency securities, and to arrange corresponding sale and repurchase agreements between its own account and foreign and international accounts maintained at the Bank. Transactions undertaken with such accounts under the provisions of this paragraph may provide for a service fee when appropriate.\n\nBy unanimous vote, the Authorization for Foreign Currency Operations was amended to update the title of the Manager of the System Open Market Account. The Authorization, as amended, is shown below:\n\n1. The Federal Open Market Committee authorizes and directs the Federal Reserve Bank of New York, for System Open Market Account, to the extent necessary to carry out the Committee's foreign currency directive and express authorizations by the Committee pursuant thereto, and in conformity with such procedural instructions as the Committee may issue from time to time:\n\nA. To purchase and sell the following foreign currencies in the form of cable transfers through spot or forward transactions on the open market at home and abroad, including transactions with the U. S. Treasury, with the U. S. Exchange Stabilization Fund established by Section 10 of the Gold Reserve Act of 1934, with foreign monetary authorities, with the Bank for International Settlements, and with other international financial institutions:\n\nAustrian schillings Belgian francs Canadian dollars Danish kroner Pounds sterling French francs German marks Italian lire Japanese yen Mexican pesos Netherlands guilders Norwegian kroner Swedish kronor Swiss francs\n\nB. To hold balances of, and to have outstanding forward contracts to receive or to deliver, the foreign currencies listed in paragraph A above.\n\nC. To draw foreign currencies and to permit foreign banks to draw dollars under the reciprocal currency arrangements listed in paragraph 2 below, provided that drawings by either party to any such arrangement shall be fully liquidated within 12 months after any amount outstanding at that time was first drawn, unless the Committee, because of exceptional circumstances, specifically authorizes a delay.\n\nD. To maintain an overall open position in all foreign currencies not exceeding $25.0 billion. For this purpose, the overall open position in all foreign currencies is defined as the sum (disregarding signs) of net positions in individual currencies. The net position in a single foreign currency is defined as holdings of balances in that currency, plus outstanding contracts for future receipt, minus outstanding contracts for future delivery of that currency, i.e., as the sum of these elements with due regard to sign.\n\n2. The Federal Open Market Committee directs the Federal Reserve Bank of New York to maintain reciprocal currency arrangements (\"swap\" arrangements) for the System Open Market Account for periods up to a maximum of 12 months with the following foreign banks, which are among those designated by the Board of Governors of the Federal Reserve System under Section 214.5 of Regulation N, Relations with Foreign Banks and Bankers, and with the approval of the Committee to renew such arrangements on maturity:\n\nAny changes in the terms of existing swap arrangements, and the proposed terms of any new arrangements that may be authorized, shall be referred for review and approval to the Committee.\n\n3. All transactions in foreign currencies undertaken under paragraph I(A) above shall, unless otherwise expressly authorized by the Committee, be at prevailing market rates. For the purpose of providing an investment return on System holdings of foreign currencies, or for the purpose of adjusting interest rates paid or received in connection with swap drawings, transactions with foreign central banks may be undertaken at non-market exchange rates.\n\n4. It shall be the normal practice to arrange with foreign central banks for the coordination of foreign currency transactions. In making operating arrangements with foreign central banks on System holdings of foreign currencies, the Federal Reserve Bank of New York shall not commit itself to maintain any specific balance, unless authorized by the Federal Open Market Committee. Any agreements or understandings concerning the administration of the accounts maintained by the Federal Reserve Bank of New York with the foreign banks designated by the Board of Governors under Section 214.5 of Regulation N shall be referred for review and approval to the Committee.\n\n5. Foreign currency holdings shall be invested insofar as practicable, considering needs for minimum working balances. Such investments shall be in liquid form, and generally have no more than 12 months remaining to maturity. When appropriate in connection with arrangements to provide investment facilities for foreign currency holdings, U. S. Government securities may be purchased from foreign central banks under agreements for repurchase of such securities within 30 calendar days.\n\n6. All operations undertaken pursuant to the preceding paragraphs shall be reported promptly to the Foreign Currency Subcommittee and the Committee. The Foreign Currency Subcommittee consists of the Chairman and Vice Chairman of the Committee, the Vice Chairman of the Board of Governors, and such other member of the Board as the Chairman may designate (or in the absence of members of the Board serving on the Subcommittee, other Board Members designated by the Chairman as alternates, and in the absence of the Vice Chairman of the Committee, his alternate). Meetings of the Subcommittee shall be called at the request of any member, or at the request of the Manager of the System Open Market Account, for the purposes of reviewing recent or contemplated operations and of consulting with the Manager on other matters relating to his responsibilities. At the request of any member of the Subcommittee, questions arising from such reviews and consultations shall be referred for determination to the Federal Open Market Committee.\n\n7. The Chairman is authorized:\n\nA. With the approval of the Committee, to enter into any needed agreement or understanding with the Secretary of the Treasury about the division of responsibility for foreign currency operations between the System and the Treasury;\n\nB. To keep the Secretary of the Treasury fully advised concerning System foreign currency operations, and to consult with the Secretary on policy matters relating to foreign currency operations;\n\nC. From time to time, to transmit appropriate reports and information to the National Advisory Council on International Monetary and Financial Policies.\n\n8. Staff officers of the Committee are authorized to transmit pertinent information on System foreign currency operations to appropriate officials of the Treasury Department.\n\n9. All Federal Reserve Banks shall participate in the foreign currency operations for System Account in accordance with paragraph 3 G(1) of the Board of Governors' Statement of Procedure with Respect to Foreign Relationships of Federal Reserve Banks dated January 1, 1944.\n\nBy unanimous vote, the Foreign Currency Directive, as shown below, was reaffirmed:\n\n1. System operations in foreign currencies shall generally be directed at countering disorderly market conditions, provided that market exchange rates for the U. S. dollar reflect actions and behavior consistent with the IMF Article IV, Section 1.\n\n2. To achieve this end the System shall:\n\nA. Undertake spot and forward purchases and sales of foreign exchange.\n\nB. Maintain reciprocal currency (\"swap\") arrangements with selected foreign central banks and with the Bank for International Settlements.\n\nC. Cooperate in other respects with central banks of other countries and with international monetary institutions.\n\n3. Transactions may also be undertaken:\n\nA. To adjust System balances in light of probable future needs for currencies.\n\nB. To provide means for meeting System and Treasury commitments in particular currencies, and to facilitate operations of the Exchange Stabilization Fund.\n\nC. For such other purposes as may be expressly authorized by the Committee.\n\n4. System foreign currency operations shall be conducted:\n\nA. In close and continuous consultation and cooperation with the United States Treasury;\n\nB. In cooperation, as appropriate, with foreign monetary authorities; and\n\nC. In a manner consistent with the obligations of the United States in the International Monetary Fund regarding exchange arrangements under the IMF Article IV.\n\nBy unanimous vote, the Procedural Instructions with respect to Foreign Currency Operations were amended to update the title of the Manager of the System Open Market Account. The Procedural Instructions, as amended, are shown below:\n\nIn conducting operations pursuant to the authorization and direction of the Federal Open Market Committee as set forth in the Authorization for Foreign Currency Operations and the Foreign Currency Directive, the Federal Reserve Bank of New York, through the Manager of the System Open Market Account (\"Manager\"), shall be guided by the following procedural understandings with respect to consultations and clearance with the Committee, the Foreign Currency Subcommittee, and the Chairman of the Committee. All operations undertaken pursuant to such clearances shall be reported promptly to the Committee.\n\n1. The Manager shall clear with the Subcommittee (or with the Chairman, if the Chairman believes that consultation with the Subcommittee is not feasible in the time available):\n\nA. Any operation that would result in a change in the System's overall open position in foreign currencies exceeding $300 million on any day or $600 million since the most recent regular meeting of the Committee.\n\nB. Any operation that would result in a change on any day in the System's net position in a single foreign currency exceeding $150 million, or $300 million when the operation is associated with repayment of swap drawings.\n\nC. Any operation that might generate a substantial volume of trading in a particular currency by the System, even though the change in the System's net position in that currency might be less than the limits specified in 1B.\n\nD. Any swap drawing proposed by a foreign bank not exceeding the larger of (i) $200 million or (ii) 15 percent of the size of the swap arrangement.\n\n2. The Manager shall clear with the Committee (or with the Subcommittee, if the Subcommittee believes that consultation with the full Committee is not feasible in the time available, or with the Chairman, if the Chairman believes that consultation with the Subcommittee is not feasible in the time available):\n\nA. Any operation that would result in a change in the System's overall open position in foreign currencies exceeding $1.5 billion since the most recent regular meeting of the Committee.\n\nB. Any swap drawing proposed by a foreign bank exceeding the larger of (i) $200 million or (ii) 15 percent of the size of the swap arrangement.\n\n3. The Manager shall also consult with the Subcommittee or the Chairman about proposed swap drawings by the System, and about any operations that are not of a routine character.\n\nThe Report of Examination of the System Open Market Account, conducted by the Board's Division of Reserve Bank Operations and Payments Systems as of the close of business on July 31, 1992, was accepted.\n\nBy unanimous vote, the minutes of actions taken at the meeting of the Federal Open Market Committee held on December 22, 1992, were approved.\n\nThe Deputy Manager for Foreign Operations reported on developments in foreign exchange markets during the period December 22, 1992, through February 2, 1993. There were no System open market transactions in foreign currencies during this period, and thus no vote was required of the Committee.\n\nThe Manager of the System Open Market Account reported on developments in domestic financial markets and on System open market transactions in government securities and federal agency obligations during the period December 22, 1992, through February 2, 1993. By unanimous vote, the Committee ratified these transactions.\n\nThe Committee then turned to a discussion of the economic outlook, the ranges for the growth of money and debt in 1993, and the implementation of monetary policy over the intermeeting period ahead. A summary of the economic and financial information available at the time of the meeting and of the Committee's discussion is provided below, followed by the domestic policy directive that was approved by the Committee and issued to the Federal Reserve Bank of New York.\n\nThe information reviewed at this meeting indicated that economic activity rose appreciably further in the fourth quarter. Final demands were buoyed by strength in consumption, business spending for durable equipment, and residential construction. Manufacturing activity also increased considerably, and employment appeared to be on a modest upward trajectory, despite a continuing flow of announcements of layoffs by large corporations. Although recent data on wages and prices had been mixed, on balance they suggested that inflation was trending gradually lower.\n\nTotal nonfarm payroll employment registered a small increase in December for the fourth consecutive month. Service industries, notably business and health services, and retail trade accounted for nearly all of the rise in jobs. Manufacturing and construction payrolls changed little, and government employment fell as temporary election workers were dropped from payrolls. The civilian unemployment rate remained at 7.3 percent, almost 1/2 percentage point below its mid-year peak but slightly above its level at the beginning of the year.\n\nIndustrial production advanced further in December and was up considerably over the fourth quarter as a whole. Motor vehicle assemblies rose sharply during the quarter; strong gains also were registered in business equipment, partly reflecting a further jump in output of computers, and in nondurable consumer goods. By contrast, the production of durable consumer goods other than motor vehicles was lower on balance after changing little over the third quarter, and the output of defense and space equipment remained on a downward trend. Total utilization of industrial capacity increased significantly in the fourth quarter and for the year as a whole.\n\nConsumer spending was up substantially in the fourth quarter. Retail sales, after rising sharply in October and changing little in November, posted a further sizable increase in December. The largest sales gains in the fourth quarter were reported at automotive dealers and at building material and supply outlets, but most other types of retail stores also recorded higher sales. By contrast, consumer spending for services, as indicated by data on personal consumption expenditures, rose more slowly. Housing starts surged in December, with single family starts reaching their highest level in nearly three years and multifamily starts picking up slightly from the very low levels of October and November. Sales of new and existing homes remained on a strong upward trend in December.\n\nReal outlays for business fixed investment apparently registered a notable gain in the fourth quarter, particularly for producers' durable equipment. Shipments of nondefense capital goods rose in November and December after changing little in October; for the quarter as a whole, shipments advanced substantially, with increases widespread by category. Business purchases of cars and trucks were up sharply in the fourth quarter, while nonresidential construction activity retraced a small part of a third-quarter decline.\n\nBusiness inventories expanded moderately in November as a sizable drop in manufacturing inventories was more than offset by increases in wholesale and retail inventories. At the manufacturing level, the drawdown of stocks was associated with strong shipments of durable goods, and inventory-to-shipments ratios in most industries were at or near the bottom of their recent ranges. In the wholesale sector, sizable inventory increases were reported in November for a second straight month; most of the buildup was limited to machinery, motor vehicles, and miscellaneous nondurable goods. With stocks rising in line with sales since September, the stock-to-sales ratio in wholesaling remained at the low end of its range over the past year. Retail inventories increased moderately further in November; the inventory-to-sales ratio for the sector was slightly below its average for previous months of the year.\n\nThe nominal U.S. merchandise trade deficit widened slightly in November. For October and November together, however, the deficit narrowed a little from its average rate in the third quarter, as the value of exports rose more than the value of imports. Most of the increase in exports was in capital goods, both machinery and aircraft, and in consumer goods. Passenger cars accounted for a considerable part of the rise in imports, while the inflow of consumer goods eased from the very strong pace of the third quarter. Recent indicators suggested that economic activity had remained weak in the major foreign industrial countries and that unemployment rates had increased further in most of those countries. The recovery in Canada appeared to be continuing, but the downturn in western Germany and Japan evidently had persisted into the fourth quarter.\n\nA small November decline in producer prices of finished goods was reversed in December, with a rebound in prices of finished foods outweighing a further drop in energy prices. For finished items other than food and energy, producer prices rose in December, but the advance followed six months of no change on balance; for 1992 as a whole, this measure of prices increased by a considerably smaller amount than in 1991. At the consumer level, the index for prices of nonfood, non-energy items edged higher in December after somewhat larger increases in the two preceding months. The rise in this index in 1992 was the smallest for any year since the early 1970s, when wage and price controls were in effect. Hourly compensation of private industry workers advanced a little more rapidly in the fourth quarter than in the two previous quarters, but the rise in total compensation over the year as a whole was considerably smaller than in 1991. The slowing of labor cost increases last year occurred in both the wages and benefits components.\n\nAt its meeting on December 22, the Committee adopted a directive that called for maintaining the existing degree of pressure on reserve positions and that did not include a presumption about the likely direction of any adjustments to policy during the intermeeting period. Accordingly, the directive indicated that in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint or slightly lesser reserve restraint would be acceptable during the intermeeting period. The reserve conditions associated with this directive were expected to be consistent with expansion of M2 at an annual rate of about 1-1/2 percent and with M3 remaining about unchanged on balance over the four-month period from November through March.\n\nOpen market operations during the intermeeting period were directed toward maintaining the existing degree of pressure on reserve positions. Adjustment plus seasonal borrowing was well above expected levels in the first two full reserve maintenance periods in the intermeeting interval; borrowing was sizable over the long New Year's weekend and also later when unusually heavy Treasury tax receipts drained reserves from the banking system. The federal funds rate averaged close to expected levels over the intermeeting period. However, the rate was somewhat volatile in late December as a result of sizable swings in market factors affecting reserves and of shifting market anticipations regarding year-end pressures.\n\nMost other short-term interest rates declined somewhat over the intermeeting period, in part reflecting the passing of year-end pressures. Intermediate- and long-term rates, including those on fixed-rate mortgages, also moved somewhat lower; the declines occurred in response to growing indications that any proposed near-term fiscal stimulus would be quite moderate and that the new Administration intended to recommend steps, possibly including new taxes, to lower the trajectory of the fiscal deficit appreciably over time. Broad indexes of stock prices exhibited mixed results over the intermeeting period: Indexes giving heavy weight to large companies changed little, while those primarily reflecting smaller companies rose significantly.\n\nIn foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies rose on balance over the intermeeting period. Through early January, the dollar appreciated against both the yen and the mark, especially the latter, in response to actual and expected further declines in interest rates in Japan and Germany. Subsequently, the dollar's gains were partially erased as the prospects for near-term easing in Germany diminished somewhat and perceptions grew that fiscal initiatives in the United States would lower the deficit and reduce the chances that monetary policy might be tightened in the months ahead.\n\nAfter expanding at a moderate pace over the course of earlier months, M2 contracted in December and January. Some of the weakness reflected a slowdown in Ml growth associated with lower mortgage refinancing activity. Within M2's nontransaction component, the expansion of savings and money market deposit accounts slowed abruptly, perhaps owing in part to the wider spread that had developed during the fall between market rates and those paid on these accounts, as well as to the use of monies in these accounts to fund a step-up in consumer purchases and nonwithheld tax payments. In addition, the continued attractiveness to investors of bond and stock mutual funds might have contributed to a quickening of the runoff of holdings of money market mutual funds and to the persisting weakness in other M2 accounts. Appreciable declines in M3 in December and January reflected both the contraction in M2 and reduced needs by banks for managed liabilities at a time of weak overall credit demand. From the fourth quarter of 1991 to the fourth quarter of 1992, both M2 and M3 grew at rates somewhat below the lower ends of the Committee's annual ranges. Total domestic nonfinancial debt appeared to have expanded at the lower end of the Committee's monitoring range for 1992.\n\nThe staff projection prepared for this meeting suggested that economic activity would expand over the year ahead at a pace that would be sufficient to reduce gradually margins of unemployed labor and capital. Recent declines in long-term interest rates and more optimistic attitudes on the part of businesses and households were expected to support further solid gains in business fixed investment and in homebuying. Continuing progress in reducing debt service burdens and a gradual lessening of concerns regarding job security were projected to foster an expansion of consumer spending a shade faster than the growth in incomes. Export demand would be damped for some period of time by the appreciation of the dollar since mid-1992, but an anticipated pickup in growth abroad later this year would begin to counteract the effects of the higher dollar. Against the background of considerable uncertainties associated with still unannounced fiscal policy initiatives, the staff retained for this forecast the assumption contained in several previous forecasts that fiscal policy would remain mildly restrictive, largely because of declining defense outlays. The persisting slack in resource utilization over the forecast horizon was expected to be associated with some additional progress in reducing inflation.\n\nIn the Committee's discussion of current and prospective economic developments, the members were encouraged by the mounting evidence of appreciable momentum in the economic expansion. On the whole, recent developments tended to reinforce their forecasts of continuing growth at a moderate pace over the year ahead, especially in light of the improvement in business and consumer confidence. The impact of some retarding influences on the expansion, notably various balance sheet adjustment activities, appeared to be waning. In addition, while some major sectors of the economy such as defense spending and commercial construction remained weak, the economy was benefitting from considerable growth in consumer spending, from rising business expenditures for producer equipment, and from increasing outlays for housing. In one view, the recent behavior of commodity prices also tended to indicate some strengthening in the economy's expansion. Despite various indications of a more firmly established expansion, however, the members felt that the outlook remained subject to a good deal of uncertainty, and some commented that substantial deviations--in either direction--from their current forecasts could not be ruled out. It was noted in this connection that the specifics of the President's fiscal policy proposals were still unknown, and their reception by the public and the Congress would have a major influence on confidence, interest rates, and the performance of the economy. Other sources of uncertainty related to the outlook for further restructuring activities that involved cutbacks in operations and employment by many firms, and the prospective lending policies of banking institutions. With regard to the outlook for inflation, most of the members believed that some further progress toward stable prices was likely over the year ahead, given an economic outcome about in line with their forecasts of continued, albeit reduced, margins of unutilized or underutilized productive resources. Some members also referred to the extended period of relatively sluggish growth in the broad measures of money as a favorable indicator in the outlook for inflation.\n\nIn keeping with the practice at meetings when the Committee establishes its long-run ranges for growth of the money and debt aggregates, the Committee members and the Federal Reserve Bank presidents not currently serving as members had prepared projections of economic activity, the rate of unemployment, and inflation for 1993. The central tendencies of the forecasts pointed to slightly faster economic growth this year than currently seemed to have occurred in 1992. The anticipated rate of economic expansion would be at a pace that was rapid enough to reduce the rate of unemployment a little further. Nonetheless, with some slack in productive resources persisting, price and cost pressures would remain subdued and modest additional moderation in inflation was expected by most members. Measured from the fourth quarter of 1992 to the fourth quarter of 1993, the forecasts for growth of real GDP had a central tendency of 3 to 3-1/4 percent within a full range of 2-1/2 to 4 percent. Projections of the civilian rate of unemployment in the fourth quarter of 1993 were concentrated in the upper half of a 6-1/2 to 7 percent range. For the CPI, the central tendency of the forecasts for the period from the fourth quarter of 1992 to the fourth quarter of 1993 was centered on increases in a range of 2-1/2 to 2-3/4 percent, and for nominal GDP the forecasts were clustered in a range of 5-1/2 to 6 percent for the year.\n\nIn the course of the Committee's discussion of various factors underlying the outlook for economic activity, the members observed that on the whole the effects of a number of structural impediments to the expansion seemed to be diminishing as the financial condition of households, business firms, and financial institutions continued to improve. Household and business debt-service burdens had eased substantially, but it remained difficult to predict to what extent and for how long the ongoing balance sheet adjustments would continue to divert an unusual proportion of cash flows from spending to balance sheet repair. Improved profitability and new capital-market issuance had strengthened the capital positions of banking institutions, and in general they were now in a much better position to augment their lending activities. However, there were few indications thus far of any easing in terms or standards on business loans, and the depressed and uncertain values of commercial mortgages and real estate held in bank portfolios might continue to exert an inhibiting effect on the willingness of banks to lend. Another negative factor was the persistence of downsizing and other restructuring activities by numerous firms, notably large businesses. Such restructuring activities had not fully run their course as many firms continued to pare excess production capacity and to modernize production facilities to meet strong competition in domestic and foreign markets. The resulting layoffs had damped overall job growth.\n\nDespite tepid job growth, retail sales had strengthened markedly during the closing months of 1992, and several members commented that such sales had continued to display surprising vigor in some parts of the country during the early weeks of 1993. Apart from the improvement in consumer sentiment, other favorable factors cited with regard to the outlook for consumer spending included lower debt-service burdens and the capital gains or enhanced cash flows now being realized as sales of homes picked up and mortgage refinancings again strengthened. Some members nonetheless expressed a degree of concern about the sustainability of the gains in consumer spending unless there were faster growth in employment and income to support such spending. Announcements by prominent firms of cutbacks in their workforces had continued into the new year, and while job gains at other firms, especially smaller ones, were contributing to modest net growth in overall employment, the publicity surrounding the persisting job cutbacks and a tendency for many new jobs to be lower-paying added an element of caution to the outlook for consumer expenditures. On balance, with the measured saving rate already at a low level, though an argument could be made that the actual rate was somewhat higher than indicated by the currently published data, consumer spending seemed likely to expand about in line with the growth in consumer incomes over the coming year.\n\nThe growth in consumer incomes in turn was likely to depend importantly on the expansion in business investment spending, and members cited a number of factors that were expected to provide a favorable setting for sustained momentum in such spending over the year ahead. These included the strengthening of final demands, the recent declines in intermediate -and long-term interest rates, the greater leeway for financial intermediaries to increase their lending to businesses, and a continuing desire by business firms to improve their operating efficiencies. Commercial construction activity, however, was likely to remain quite sluggish. There were indications that commercial real estate values had stabilized in a number of areas, but at low levels, and given the persistence of marked imbalances in numerous real estate markets that were the result of several years of overbuilding, a significant rebound in commercial building activity for the nation as a whole might well be several years away. The outlook for housing construction was much more promising. Against the background of a general upswing in consumer confidence and the improved balance sheets of many households, the declines that had occurred in mortgage interest rates had fostered a marked strengthening in the demand for single-family housing as evidenced by reports from many parts of the country as well as the overall statistics on housing. On the basis of these developments, the members anticipated a continuing impetus to the economic expansion from housing construction and from related industries over the year ahead. In addition, the current indications of generally lean business inventories, associated in part with strong final demands over the past several months, suggested that the prospects for further gains in overall spending were likely to stimulate efforts by business firms to build up inventories over the quarters ahead.\n\nThe increasing signs of slow growth or recession in a number of foreign nations represented a greater downside risk to the demand for U.S. exports than had been apparent earlier. It was noted, for example, that firms engaged in business activities abroad were reporting substantial deterioration in markets for U.S. goods in many foreign countries. Growth in U.S. exports might remain positive over the year ahead, but against the background of a relatively expansive U.S. economy and the dollar's recent appreciation, the value of exports might well fall increasingly short of that of imports with adverse effects on the growth of U.S. economic activity.\n\nTurning to the outlook for fiscal policy, members were encouraged by the prospect that the President would soon propose a program that would produce substantial reductions in the federal deficit over the years ahead. Such a deficit-reduction program, if deemed credible, could result in lower intermediate- and long-term interest rates than would otherwise prevail--even before the program was enacted--with very positive implications for interest-sensitive expenditures. For the nearer term, the President was expected to announce some modest fiscal stimulus relative to what was currently in train. However, the specifics of the President's proposals were not yet known and there was little current basis on which to judge prospective public and Congressional reactions. Members emphasized the critical need for long-term deficit reduction, and some expressed concern about the adverse effects on financial markets if fiscal stimulus measures were to be enacted for the short run without the assurance of further legislation to cut federal deficits over time.\n\nWith regard to the outlook for inflation, most of the members anticipated that the trend toward lower price and wage inflation would be sustained over the year ahead, and one member observed that the disinflationary momentum in the economy might well be underestimated. Favorable developments relating to the outlook for inflation included evidence of slowing increases in labor costs and continued aggressive efforts by many business firms to improve productivity and reduce costs in the face of intense competition from domestic and foreign producers. Indeed, anecdotal reports from around the country continued to suggest little or no upward pressure on prices in many regions. In addition, the behavior of interest rates in longer-term debt markets was consistent with spreading expectations of gradually diminishing inflation. Some members believed, however, that little or no further progress in reducing inflation was a more likely outcome in the year ahead, though none anticipated higher inflation. Some commodity price indexes had edged higher recently, apparently in response to growing demands related to strengthening activity in several sectors of the economy. Lumber prices in particular had risen considerably in conjunction with the uptrend in single-family housing construction and various constraints on lumber supplies. Some business contacts reported for the first time in a long while that they were experiencing or anticipated some upward pressure on their raw materials prices. Further, while most business contacts saw or anticipated little or no upward pressure on prices in their own industries, many continued to expect rising inflation more generally. The still relatively steep slope of the yield curve and its implications with regard to expectations of future increases in interest rates also suggested that investors remained concerned about the possibility of higher inflation over the longer run, even though such concerns might have abated somewhat recently and did not appear to extend to the next year or two. In general, however, the members viewed the inflation outlook with considerable optimism on the presumption of favorable fiscal and monetary policy developments.\n\nIn keeping with the requirements of the Full Employment and Balanced Growth Act of 1978 (the Humphrey-Hawkins Act), the Committee at this meeting reviewed the ranges for growth of the monetary and debt aggregates in 1993 that it had established on a tentative basis at its meeting on June 30-July 1, 1992. The tentative ranges included expansion of 2-1/2 to 6-1/2 percent for M2 and 1 to 5 percent for M3, measured from the fourth quarter of 1992 to the fourth quarter of 1993. The monitoring range for growth of total domestic nonfinancial debt had been set provisionally at 4-1/2 to 8-1/2 percent for 1993. All of these ranges were unchanged from those that the Committee had set for 1992 at its meeting in February of last year and had reaffirmed at mid-year. When the provisional ranges for money growth were established, the Committee had noted that they were especially tentative and subject to revision in the latter part of 1992 or early 1993 owing to the considerable uncertainty about the evolving relationship of money to income.\n\nIn the event, the velocities of M2 and M3 had increased appreciably in the second half of 1992 and analysis of the factors behind this development suggested further increases in the year ahead. Consequently, in the Committee's discussion, which tended to focus on M2, all the members indicated that they could support a proposal to lower the tentative ranges for growth of the broad monetary aggregates by one-half percentage point for 1993. At the same time, a number of members indicated that they preferred somewhat different ranges including the retention of the tentative ranges, lowering the ranges by more than the proposal, and widening or narrowing them. All the members were in firm agreement that the purpose of the proposed reductions was not to signal or implement any change in monetary policy or to convey any intention to move away from the Committee's commitment to maximum sustainable economic expansion. Rather, the reductions were motivated by the persistence of marked shortfalls in the growth of M2 and M3 from their historical relationships with various measures of aggregate economic performance; those shortfalls appeared to be the technical result of forces that are altering the relationship between money and income. Members of the Committee urged that the Board's report to Congress and the Chairman's accompanying testimony make clear the reasons for the unusual behavior of money and its consequences for the Committee's choice of ranges.\n\nThe deviations in monetary growth from historical norms reflected a number of developments whose relative importance and intensity had shifted to some extent over the course of recent years, but in general they had served to rechannel funds away from depository institutions, and the associated weakness in deposit growth had raised velocity--the ratio of nominal GDP to money. The result was the need for lower money growth than in the past to support a given rate of income growth. Among the developments that had tended to retard the relative growth of M2 and M3 was the unprecedented steepness of the yield curve that had prompted large shifts of funds by savers from M2 accounts to higher-yielding intermediate- and long-term assets. At the same time, credit growth at bank and thrift depository institutions had been weak, partly as a result of efforts by these institutions to improve capital and liquidity positions, and partly owing to weak demand. As a consequence, they also had maintained relatively low offering rates on deposits that had provided consumers with an incentive to reduce or hold down their deposit holdings in order to pay down relatively high cost mortgages and other debts. In 1992, sluggish growth of M2 and M3 had been associated with a considerable acceleration in nominal spending. Indeed, despite growth of both M2 and M3 at rates below the Committee's ranges, the expansion of the economy had exceeded most forecasts.\n\nThe members generally anticipated that the intensity of these forces might diminish in 1993 as borrowers and lending institutions achieved more comfortable balance sheet positions. Nonetheless, the relative weakness in money growth was seen as likely to persist to a marked extent. The yield curve, while it had flattened a bit recently, was still expected to provide a considerable incentive for many savers to shift funds out of M2 assets, especially as relatively high-yielding time deposits continued to mature. In addition, banks were likely to remain generally unaggressive in bidding for deposits, in part because their substantial earlier acquisitions of securities would permit them to accommodate some of the anticipated growth in loan demand by selling securities or limiting purchases. In these circumstances, restrained money growth seemed likely to remain consistent with relative strength in the economic expansion.\n\nThe members recognized that the strength of the factors that were expected to continue to depress broad money growth in relation to income in 1993 was still subject to considerable uncertainty, and this implied the need for flexibility in assessing the implications of money growth relative to the Committee's ranges. Should the factors influencing the behavior of the broad aggregates persist in holding down money growth to the extent seen in 1992, expansion of M2 and M3 in the lower portion of their reduced ranges would be consistent with considerable further growth in nominal spending. Indeed, a shortfall from the reduced ranges could not be ruled out, and one member felt that the potential for such a development warranted consideration of a somewhat larger reduction in the M2 range; such a reduction also would signal more clearly the Committee's commitment to price stability. On the other hand, the upper portions of the reduced ranges would still accommodate an ample provision of liquidity to support further economic expansion even if the growth of money and of income were to move toward an historically more normal alignment and velocity were to slow from its high rate of increase. In one view, widening the tentative M2 range by reducing its lower limit while retaining its upper limit would help the Committee to convey its views regarding the potential for a continuing but acceptable sluggishness in M2 growth while leaving room for the possibility of faster M2 expansion should changing circumstances foster diminishing strength in velocity. Another member expressed a preference for narrowing the tentative range by lowering only its upper limit as a means of signaling the Committee's intent to resist both inflationary and recessionary developments. In light of the uncertainties that were involved, the informational content of the aggregates probably had diminished and in any event the Committee would need to continue to evaluate monetary growth developments in the context of a careful assessment of a wide variety of other financial, economic, and price developments. In this connection, one member observed that the uncertainties were of such a magnitude that, while plausible arguments could be made for a number of different ranges, retention of the tentative ranges would be appropriate in light of the Committee's willingness to review the ranges in the event that unanticipated developments were to unfold.\n\nAll of the members agreed that it would be desirable to retain the monitoring range of 4-1/2 to 8-1/2 percent that the Committee had established on a provisional basis for the growth of total domestic nonfinancial debt in 1993. The expansion in such debt had not been damped by special forces to the same extent as the broad monetary aggregates in 1992. Over the year ahead, growth in the federal debt was likely to remain substantial, and the expansion of debt in the nonfederal sectors was projected to accelerate somewhat given the continued improvement in borrower balance sheets and an anticipated increase in the willingness of financial institutions to lend as the economy continued to expand. Nonetheless, in the context of still cautious attitudes on the part of both borrowers and lenders, the growth of nonfederal debt probably would remain below that of nominal GDP in the year ahead.\n\nAt the conclusion of the Committee's discussion, all of the members indicated that they favored or could accept a technical downward adjustment of one-half percentage point in the tentative ranges for the broader monetary aggregates for 1993 to rates of 2 to 6 percent for M2 and 1/2 to 4-1/2 percent for M3. It was agreed that there should be no change from the tentative range for total domestic nonfinancial debt. In keeping with the Committee's usual procedures under the Humphrey-Hawkins Act, the ranges would be reviewed at midyear, or sooner if deemed necessary, in light of the growth and velocity behavior of the aggregates and ongoing economic and financial developments. Accordingly, by unanimous vote, the following longer-run policy for 1993 was approved by the Committee for inclusion in the domestic policy directive:\n\nThe Federal Open Market Committee seeks monetary and financial conditions that will foster price stability and promote sustainable growth in output. In furtherance of these objectives, the Committee at this meeting established ranges for growth of M2 and M3 of 2 to 6 percent and 1/2 to 4-1/2 percent respectively, measured from the fourth quarter of 1992 to the fourth quarter of 1993. The Committee expects that developments contributing to unusual velocity increases are likely to persist during the year. The monitoring range for growth of total domestic nonfinancial debt was set at 4-1/2 to 8-1/2 percent for the year. The behavior of the monetary aggregates will continue to be evaluated in the light of progress toward price level stability, movements in their velocities, and developments in the economy and financial markets.\n\nTurning to policy for the intermeeting period ahead, all of the members endorsed a proposal to maintain unchanged conditions in reserve markets, and all indicated that they could accept a directive that did not incorporate any presumption with regard to the likely direction of possible intermeeting adjustments to policy. While there was concern about the weakness in the monetary aggregates, the members generally agreed that recent economic developments tended to reinforce the view that monetary policy was on an appropriate course. The economy seemed to be on a stronger growth track than earlier in the expansion, and inflation remained quite subdued--only a bit above some estimates of price stability--and likely to moderate further in coming quarters in the view of most members. Some commented that a further easing move at this juncture might well have adverse effects on inflation sentiment and on interest rates in intermediate- and long-term debt markets. A few referred to the recent firming in some commodity prices and the consensus among private forecasters that inflation could drift higher over the next few years. In the view of one member, these developments might argue for a tilt in the directive toward possible restraint, but they did not call for an immediate tightening in reserve conditions.\n\nA staff analysis prepared for this meeting suggested a resumption of some growth in the broad measures of money later in the first quarter but a decline in both M2 and M3 for the quarter as a whole. While part of the declines appeared to reflect difficulties with seasonal adjustments and the ebbing of special factors that previously had boosted growth, the uncertainties surrounding the behavior of these aggregates tended to reduce their role in current monetary policy. Nevertheless, there was concern about the persisting weakness in the broad aggregates, including the likelihood that they would fall well short of the Committee's new ranges over the first part of the year. Some members also noted that the growth of Ml, while still fairly robust in December and January, was markedly below its pace over most of 1992. On the other hand, bank loans had increased in recent months, and the weakness in the monetary aggregates did not appear to reflect underlying softness in the economy. In these circumstances, a number of members believed that any effort to stimulate monetary growth under immediately prevailing economic conditions and market expectations might well prove to be counterproductive. An easing at this time could accelerate outflows from interest-sensitive M2 assets if the easing were seen as signaling a weakening of the System's anti-inflationary resolve and were to result in higher rates on intermediate- and long-term debt securities.\n\nAt the conclusion of the Committee's discussion, all of the members indicated that they favored a directive that called for maintaining the existing degree of pressure on reserve positions. They also noted their preference for, or acceptance of, a directive that did not include a presumption about the likely direction of any adjustment to policy over the intermeeting period. Accordingly, in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, the Committee decided that slightly greater or slightly lesser reserve restraint would be acceptable during the intermeeting period. The reserve conditions contemplated at this meeting were expected to be consistent with little change in the levels of M2 and M3 over the two-month period from January through March.\n\nBy unanimous vote, the Federal Reserve Bank of New York was authorized and directed, until otherwise directed by the Committee, to execute transactions in the System Account in accordance with the following domestic policy directive:\n\nThe information reviewed at this meeting indicates that economic activity rose appreciably further in the fourth quarter. Total nonfarm payroll employment registered another small increase in December, and the civilian unemployment rate remained at 7.3 percent. Industrial production posted solid gains over the closing months of the year. Retail sales were up substantially in the fourth quarter, and residential construction activity increased sharply. Indicators of business fixed investment suggest a notable gain in recent months, particularly for producers' durable equipment. The nominal U.S. merchandise trade deficit narrowed slightly in October-November from its average rate in the third quarter. Recent data on wages and prices have been mixed but they continue to suggest on balance a trend toward lower inflation.\n\nInterest rates have declined somewhat since the Committee meeting on December 22. In foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies rose on balance over the intermeeting period.\n\nM2 appears to have contracted in December and January, after expanding at a moderate pace over the course of previous months; M3 is estimated to have declined appreciably in both months. From the fourth quarter of 1991 to the fourth quarter of 1992, both M2 and M3 grew at rates somewhat below the lower ends of the Committee's annual ranges for 1992. Total domestic nonfinancial debt appears to have expanded at the lower end of the Committee's monitoring range for the year.\n\nThe Federal Open Market Committee seeks monetary and financial conditions that will foster price stability and promote sustainable growth in output. In furtherance of these objectives, the Committee at this meeting established ranges for growth of M2 and M3 of 2 to 6 percent and 1/2 to 4-1/2 percent respectively, measured from the fourth quarter of 1992 to the fourth quarter of 1993. The Committee expects that developments contributing to unusual velocity increases are likely to persist during the year. The monitoring range for growth of total domestic nonfinancial debt was set at 4-1/2 to 8-1/2 percent for the year. The behavior of the monetary aggregates will continue to be evaluated in the light of progress toward price level stability, movements in their velocities, and developments in the economy and financial markets.\n\nIn the implementation of policy for the immediate future, the Committee seeks to maintain the existing degree of pressure on reserve positions. In the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint or slightly lesser reserve restraint would be acceptable in the intermeeting period. The contemplated reserve conditions are expected to be consistent with little change in M2 and M3 over the period from January to March.\n\nAt this meeting the Committee discussed a preliminary report of a subcommittee that had been established to examine various issues relating to the release of information about Committee meetings and decisions. All of the members agreed that the Committee should keep the public as fully informed as possible about its monetary policy decisions and their rationale. Such information could reduce uncertainty about the stance of policy and about the factors the Committee takes into account in reaching its decisions. However, release of information should not be allowed to compromise the overriding objective of making and implementing the best possible decisions. In that regard, the Committee noted that its deliberative process requires a free flow of ideas, including the ability to advance or question hypotheses, to speculate on alternative outcomes, and to change opinions in response to the views expressed by other members. The members also needed to feel at liberty during meetings to use a wide array of information that is obtained on a confidential basis; at least some of that information would no longer be provided to the Committee if there were a risk of public disclosure. Moreover, the Committee wanted to give further consideration to the risk that the adoption of a different schedule for releasing information about policy decisions might have the effect, in difficult circumstances, of reducing its willingness to make needed policy adjustments promptly. No decisions were made at this meeting concerning various options for apprising the public more fully or promptly of the Committee's actions, and it was understood that the subcommittee would continue to study the matter.\n\nIt was agreed that the next meeting of the Committee would be held on Tuesday, March 23, 1993.\n\nThe meeting adjourned.\n\nSecretary\n\nFOMC",
            "content_type": "fomc_minutes",
        },
        {
            "date": "1993-03-23",
            "speaker": "Alan Greenspan",
            "title": "FOMC Meeting Minutes",
            "text": "A meeting of the Federal Open Market Committee was held in the offices of the Board of Governors of the Federal Reserve System in Washington, D.C., on Tuesday, March 23, 1993, at 9:00 a.m.\n\nPRESENT:\n\nMr. Greenspan, Chairman Mr. Corrigan, Vice Chairman Mr. Angell Mr. Boehne Mr. Keehn Mr. Kelley Mr. LaWare Mr. Lindsey Mr. McTeer Mr. Mullins Ms. Phillips Mr. Stern\n\nMessrs. Broaddus, Jordan, Forrestal, and Parry, Alternate Members of the Federal Open Market Committee\n\nMessrs. Hoenig, Melzer, and Syron, Presidents of the Federal Reserve Banks of Kansas City, St. Louis, and Boston, respectively\n\nMr. Bernard, Deputy Secretary Mr. Coyne, Assistant Secretary Mr. Gillum, Assistant Secretary Mr. Mattingly, General Counsel Mr. Patrikis, Deputy General Counsel Mr. Prell, Economist Mr. Truman, Economist\n\nMessrs. R. Davis, Lang, Lindsey, Rolnick, Rosenblum, Scheld, Siegman, Simpson, and Slifman, Associate Economists\n\nMr. McDonough, Manager of the System Open Market Account\n\nMs. Greene, Deputy Manager for Foreign Operations\n\nMs. Lovett, Deputy Manager for Domestic Operations\n\nMr. Ettin, Deputy Director, Division of Research and Statistics, Board of Governors\n\nMr. Winn,1 Assistant to the Board, Office of Board Members, Board of Governors\n\nMr. Madigan, Assistant Director, Division of Monetary Affairs, Board of Governors\n\nMr. Hooper, Assistant Director, Division of International Finance, Board of Governors\n\nMs. Low, Open Market Secretariat Assistant, Division of Monetary Affairs, Board of Governors\n\nMessrs. Beebe, T. Davis, Dewald, Goodfriend, and Ms. Tschinkel, Senior Vice Presidents, Federal Reserve Banks of San Francisco, Kansas City, St. Louis, Richmond, and Atlanta, respectively\n\nMs. Browne, and Mr. Sniderman, Vice Presidents, Federal Reserve Banks of Boston and Cleveland, respectively\n\nMs. Krieger, Manager, Open Market Operations, Federal Reserve Bank of New York\n\n1. Attended portion of meeting relating to discussion of merging minutes of action and policy record into one document.\n\nAt the start of the meeting, the subcommittee established to review policies relating to the release of Committee information reported on its further deliberations and proposed a merging of the current \"Minutes of Actions\" and the \"Record of Policy Actions\" into a new document to be designated \"Minutes of the Federal Open Market Committee Meeting.\" Merging the two documents would put in convenient form all the information that is released pertaining to FOMC meetings, and the new document would be made public on the same schedule as its predecessor documents. The Committee members endorsed the subcommittee's proposal and by unanimous vote the Committee approved the \"Minutes of the Federal Open Market Committee Meeting\" held on February 2-3, 1993; this merged document was scheduled to be released on March 26, 1993.\n\nThe Manager of the System Open Market Account reported on developments in foreign exchange markets since the previous meeting on February 2-3, 1993. There were no System open market transactions in foreign currencies during this intermeeting period, and thus no vote was required of the Committee.\n\nThe Deputy Manager for Domestic Operations reported on developments in domestic financial markets and on System open market transactions in government securities and federal agency obligations during the period February 3, 1993, through March 22, 1993. By unanimous vote, the Committee ratified these transactions.\n\nThe Committee then turned to a discussion of the economic outlook and the implementation of monetary policy over the inter-meeting period ahead. A summary of the economic and financial information available at the time of the meeting and of the Committee's discussion is provided below, followed by the domestic policy directive that was approved by the Committee and issued to the Federal Reserve Bank of New York.\n\nThe information reviewed at this meeting suggested that economic activity was expanding at a more moderate pace in the early months of 1993 after increasing substantially in the fourth quarter. Although outlays for business equipment apparently remained on a strong upward trajectory, sales of new homes had slackened and consumer spending was rising less rapidly. Indicators of production activity also were mixed: Industrial output had continued to post solid gains, but homebuilding had been less robust since year-end. Payroll employment had strengthened, and the unemployment rate had moved down further. Increases in wages had remained subdued in recent months, but advances in consumer and producer prices had been larger than those recorded in the latter part of 1992.\n\nTotal nonfarm payroll employment rose sharply in February, following generally small advances in previous months, and the length of the average workweek remained at the fourth-quarter level. The strong job gains in construction, services, and retail trade in February apparently reflected to some extent a partial reversal of the special factors that had depressed reported employment in these sectors in previous months. Since December, initial claims for unemployment insurance had fluctuated in a range that was consistent with further modest growth in employment. The civilian unemployment rate edged lower again in February, to 7.0 percent.\n\nIndustrial production continued to rise at a fairly brisk pace in January and February. Changes in mining and utilities were about offsetting on balance over the two months, but increases in manufacturing were fairly widespread. Although motor vehicle assemblies fell in February from a relatively high January level, the production of consumer durables and computers turned up sharply. In addition, increases in output were recorded in several other categories, including non-energy materials and construction supplies. Recent surveys indicated that new orders for durable goods increased further in February, and lean factory inventories coupled with reports of lengthening delivery times suggested further gains in industrial output in coming months. Total utilization of industrial capacity rose again in February.\n\nRetail sales advanced in February after a fourth-quarter surge and a pause in January. Sales at automotive dealers weakened in February. However, there were sharp increases in sales of building materials and supplies, miscellaneous durable goods, and nondurable goods other than apparel. After registering sizable gains late last year, housing starts fell substantially in January and retraced only part of that decline in February. The slowdown was concentrated in single-family housing starts; multifamily starts were up in February from a historically low level in January. Although mortgage interest rates had dropped to the lowest levels in decades, sales of both new and existing homes turned down in January from their high December levels.\n\nIncoming data on orders and shipments of nondefense capital goods suggested a further brisk advance in outlays for business equipment in coming months. In January, a decline in shipments of nondefense capital goods only partially reversed a large December rise, as a surge in shipments of computing equipment helped sustain the overall level. Shipments of complete civilian aircraft posted a solid gain in January. The increase appeared to be concentrated in sales to foreign purchasers; in the domestic airline industry, intense competition was forcing cutbacks of unprofitable routes and reductions in both the number of planes in service and orders for new planes. Shipments of durable equipment other than computers and aircraft fell in January to about the level of the fourth quarter. On the other hand, the January reading on new orders for these goods was well above the average for the fourth quarter, suggesting that additional advances in shipments might lie ahead. Nonresidential construction activity was down slightly further in January, reflecting persisting declines in office and industrial building in an environment of excess supply and some continuing, though perhaps lessening, downward pressure on the prices of such structures.\n\nBusiness inventories appeared to have edged lower in January. In manufacturing, factory stocks were drawn down further, and most industries had relatively low stocks-to-shipments ratios. Among wholesalers, strong January sales pulled down inventories at many types of establishments; in numerous cases, a large accumulation of stocks in the fourth quarter was reversed. For the wholesale sector as a whole, the inventories-to-sales ratio in January was near the bottom of the range of the past two years. Retail inventories rose somewhat further in January after a large December increase. Stocks at automotive dealers accounted for all of the January accumulation. At retail stores other than auto dealers, the ratio of inventories-to-sales remained within the narrow range observed over the past year.\n\nThe nominal U.S. merchandise trade deficit widened slightly in January but was little changed from its average level in the fourth quarter. The value of both exports and imports dropped sharply in January from the December levels. The decline in imports was spread widely among major trade categories, but the decrease in exports largely reflected a reduction in shipments of aircraft after a strong December rise. Among the major foreign industrialized countries, the level of real activity contracted further in the fourth quarter in Japan, western Germany, and France; for the first quarter, the limited data available were generally weak for Japan and France but somewhat more mixed for western Germany. By contrast, economic activity appeared to be increasing in Canada and the United Kingdom.\n\nProducer prices of finished goods were up in January and February after changing little over the fourth quarter. Producer prices of finished foods declined over the first two months of the year, but prices of finished energy products climbed rapidly, and prices of other finished items rose at a faster rate than in 1992. At the consumer level, price increases in January and February also were on the high side of the past year's advances. Food prices jumped in January and rose slightly further in February, while energy prices retraced most of a sharp January rise. Excluding food and energy items, consumer prices advanced at a substantially faster pace over the January-February period than in 1992. Increases in wages, as measured by average hourly earnings of production or nonsupervisory workers, remained subdued in recent months. The advance in average hourly earnings slowed in February, and the rise over the twelve months ended in February was considerably smaller than over the previous twelve-month period.\n\nAt its meeting on February 2-3, 1993, the Committee adopted a directive that called for maintaining the existing degree of pressure on reserve positions and that did not include a presumption about the likely direction of any adjustments to policy during the intermeeting period. Accordingly, the directive indicated that in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint or slightly lesser reserve restraint would be acceptable during the intermeeting period. The reserve conditions associated with this directive were expected to be consistent with little change in the levels of M2 and M3 over the two-month period from January through March.\n\nOpen market operations during the intermeeting period were directed toward maintaining the existing degree of pressure on reserve positions. Adjustment plus seasonal borrowing averaged only slightly above expected levels in the three full reserve maintenance periods in the intermeeting interval. For the period as a whole, the federal funds rate remained close to the 3 percent level that had prevailed in previous months.\n\nOther short-term interest rates changed little over the intermeeting period, while long-term rates fell appreciably on balance. Bond markets rallied over most of the period, reflecting market assessments of improved prospects for significant reductions in the federal budget deficit in coming years and the consequences for overall spending. Prices of Treasury notes and bonds also were boosted by municipal defeasance activity and by perceptions of heightened prepayment risks in mortgage-backed securities. In early March, interest rates on long-term Treasury bonds and conventional fixed-rate mortgages reached their lowest levels since 1973, but some of the decline in bond and mortgage rates subsequently was reversed in response to increased apprehension about inflation. Equity prices generally responded favorably to the drop in long-term interest rates, but concerns about future changes in government policy toward a number of industries, including health care, led to lower prices in some segments of the equities market.\n\nIn foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies fell on balance over the intermeeting period. The dollar depreciated through late February, partly in response to declines in U.S. long-term interest rates and incoming data that were seen as pointing to some slowing of the expansion in the United States. Subsequently, the dollar rebounded in the wake of unexpectedly strong U.S. employment statistics, disappointing inflation numbers, and further signs of weakening economic activity abroad. Near the end of the period, the dollar again dropped against the German mark and other European currencies, following a cut by the Bundesbank of its discount rate that apparently was less than market participants were expecting. On balance over the period, the dollar was marginally lower against the mark and other European currencies, but it declined substantially against the Japanese yen, reaching an all-time low.\n\nM2 and M3 contracted in January and February. Part of the weakness apparently reflected temporary factors, such as distortions in seasonal adjustment factors and a lull in prepayments of mortgage-backed securities that reduced deposits held in association with this activity. More fundamentally, relatively attractive returns on capital market instruments continued to prompt households to shift large amounts of liquid balances into market investments, such as bond and stock mutual fund shares. In addition, banks continued to issue subordinated debt and equity to improve their balance sheets at a time when the expansion of bank credit was slowing noticeably; in particular, bank lending to businesses had been depressed by paydowns from the proceeds of heavy bond and stock issuance by nonfinancial corporations. Total domestic nonfinancial debt appeared to have expanded somewhat further in January.\n\nThe staff projection prepared for this meeting suggested that economic activity would grow over the year ahead at a pace that would foster a further gradual reduction in margins of unemployed labor and capital. The projection incorporated the essential elements of the fiscal proposals recently set forth by the Administration; the effects on aggregate demand, all other things equal, were expected to be small over the next several quarters. However, the appreciable declines in long-term interest rates that had occurred in recent months--evidently partly in response to anticipations of intermediate-term deficit reduction--were expected to support substantial additional gains in business and residential investment. Consumer spending would be bolstered by the progress already achieved in reducing debt service burdens and by a gradual lessening of concerns regarding job security, although the higher personal income taxes now envisioned for upper-income taxpayers were expected to be an inhibiting factor. Increases in export demand would be limited in the near term by the continuing weakness in the economies of the major industrialized countries. The persisting slack in resource utilization was expected to be associated with a return to more subdued price increases after a spurt earlier this year.\n\nIn the Committee's discussion of current and prospective economic conditions, the members remained encouraged by recent developments that they viewed on the whole as tending to confirm their forecasts of sustained economic expansion, though at a pace appreciably below that now indicated for the fourth quarter of 1992. If realized, such economic growth would be associated over time with a further gradual decline in unemployment. While the expansion appeared to have generated some momentum, a number of factors were likely to limit its strength, including ongoing balance sheet and business restructuring activities, the outlook for a more restrictive federal budget, and continuing weakness in key foreign markets. At the same time, greatly reduced interest rates and much improved, if still vulnerable, business and consumer confidence were positive factors in the outlook. Some members cautioned that even though a moderate rate of economic growth could be viewed as the most likely outcome over the forecast horizon, the current expansion differed in important respects from earlier cyclical recoveries and in light of the attendant uncertainties a considerably different result--in either direction--could not be ruled out. With regard to the outlook for inflation, the faster increases in consumer prices in recent months and a sharp upturn in the prices of certain producer materials tended to raise concerns, or at least a degree of unease, with regard to underlying inflation trends. While these developments might well prove to be an aberration rather than a signal of intensifying inflation, they did suggest the need to reassess the likelihood of a further decline in inflation and to be alert to further signs of a sustained upturn. For now, however, the favorable trends in underlying unit labor costs, which were associated in turn with ongoing gains in productivity and the absence of any firming in wage pressures, led many members to conclude that recent price developments did not provide persuasive evidence of a change in the inflation outlook.\n\nMembers continued to report somewhat uneven business conditions across the nation. Steady economic growth characterized many parts of the country, but business activity remained depressed in some areas and industries, notably those related to defense, aerospace, and nonresidential construction. While business sentiment was generally positive, many business contacts were uncertain about the outlook for demand in their own industries or the potential strength of the overall expansion, and recent fiscal policy developments appeared to have introduced a further note of caution. This uncertainty helped to account for the continuing reliance of numerous firms on overtime work to meet growing demand rather than incurring the considerable costs of adding new workers. Even so, an increasing number of contacts were reporting worker recalls or new hires. One member commented that job growth could be viewed both as a measure of business sentiment and as a necessary element in building or maintaining consumer confidence and thus helping to ensure an enduring economic expansion.\n\nThe quickening recovery during 1992, especially in the second half of the year, had received considerable impetus from consumer spending, and while growth in such spending could be expected to moderate from its pace in recent quarters, the consumer sector was viewed as likely to play a key role in sustaining the expansion this year. Many consumers had taken advantage of steep declines in interest rates to strengthen their balance sheets and reduce their debt service burdens, and they were now in a much improved position to finance further growth in their expenditures. The members took note of recent indications of a decline in consumer confidence and of some softening in retail sales since early in the year. However, the latter appeared to be in part the result of recently adverse weather conditions in some major parts of the country, and consumer confidence was still much improved on balance since earlier in the recovery. Accordingly, recent developments were not seen in themselves as harbingers of a weakening consumer spending trend over the next several quarters.\n\nBusiness spending on producers' durable equipment also was believed likely to continue to provide appreciable stimulus to the expansion assuming that the much reduced interest rates and currently favorable business attitudes would be sustained and that proposed investment tax credit legislation eventually would be enacted. At the same time, business spending for nonresidential structures probably would continue to be held back by weakness in office construction stemming from widespread overcapacity. While office building activity was likely to be restrained for an extended period, members saw some positive signs that pointed to a degree of stabilization in this sector, including the leveling out or even a marginal pickup in rents and occupancy rates in some markets that previously had been severely depressed. A slow turnaround in other building activity was reported in some regions, notably for industrial and retail structures.\n\nWhile the available data on starts of single-family houses in January and February were somewhat disappointing, the members felt that housing construction activity had held up relatively well thus far this year, after allowing for the adverse weather conditions that had retarded construction in some areas. The greatly reduced cost of mortgage financing pointed to continuing gains in housing construction despite a rise in costs associated with the sharp jump in lumber prices and a scarcity of finished building lots in some areas.\n\nThe members agreed that the prospects for overall spending on business capital goods and housing were vulnerable to shifts in attitudes that might be triggered, for example, by increases in market interest rates associated with an absence of progress in reducing the federal budget deficit. The outlook for a significant contraction in the federal deficit was subject to considerable uncertainty, especially in light of the still pending decisions to be made with regard to health care programs and their financing. The members recognized that the direct effects of appreciable deficit reduction would tend to constrain economic activity, as evidenced by the impact in many areas of the defense cutbacks that were already being implemented. Business contacts had expressed concerns about the potential effects on their industries and local markets of various provisions in the proposed legislation. Even so, a more encompassing assessment of the effects of deficit reduction needed to take account of its favorable implications for domestic interest rates. Moreover, insofar as the nearer-term outlook was concerned, the fiscal legislation now under consideration included new spending initiatives and the investment tax credit that were intended to provide some temporary stimulus to an economic expansion that, in the view of many observers, might still be in the process of gathering sustainable momentum. On balance, substantial deficit reduction in line with the currently proposed legislation was seen as likely to have a positive effect on business and consumer confidence, financial markets, and the longer-term health of the economy.\n\nSeveral members observed that the outlook for exports had worsened as a result of weakening economic trends in a number of major industrialized nations. Members also commented on the uncertainties in the outlook for foreign trade associated with a variety of political risks abroad and the persisting protectionism that currently was highlighted by strong opposition to key trade agreements now under negotiation or under consideration for ratification. Anecdotal information from business contacts involved in export markets continued to suggest lagging foreign demands for many U.S. goods; however, backlogs for other products, such as labor-saving capital equipment, remained sizable.\n\nThe members devoted considerable attention to the discussion of various factors underlying the outlook for inflation. The consumer and producer price indexes had been less favorable in January and February than in the latter part of 1992. Also, prices of various industrial and construction materials had firmed since the start of the year in apparent response to rising production and, in some industries, to import or environmental restrictions. Anecdotal reports of increasing costs and prices had begun to appear with somewhat greater frequency in some areas, and pressures to widen profit margins reportedly were strong in a number of industries. In their evaluation of recent inflation developments, however, the members generally gave more weight to the behavior of unit labor costs, which indicated that much of the economy's underlying cost structure did not reflect any signs of a pickup in inflationary pressures. Moreover, from a financial perspective, extensions of credit and growth in overall nonfinancial sector debt were not consistent with an economy that was generating significant inflationary pressures, and the recent behavior of long-term debt markets suggested expectations of more subdued inflation. Against this background, the recent upturn in consumer and certain commodity prices might well represent a temporary development such as had occurred previously during the current cyclical upswing. In support of this view, members cited various fundamentals that seemed inconsistent with accelerating inflation, including the considerable slack in the utilization of labor and capital resources, strong competitive conditions in many markets, the absence of significant lengthening in supplier delivery schedules, and an extended period of weak expansion in the broader monetary aggregates that now encompassed some recent deceleration in M1. Nonetheless, the members acknowledged that recent price developments had raised a degree of unease in their minds, and their concerns would rise if the recent pace of price advances were sustained in the months immediately ahead. One member observed that a somewhat faster economic expansion than currently was expected by most members might well serve to intensify inflation pressures. While price developments were notably difficult to predict, most of the members concluded that the evidence at this point did not confirm a resurgence in inflationary pressures, and some commented that further modest disinflation remained a reasonable expectation for the next several quarters.\n\nIn the Committee's discussion of policy for the intermeeting period ahead, most of the members endorsed a proposal to maintain an unchanged degree of pressure on reserve positions, while two members supported an immediate move to tighten reserve conditions. In the majority view, the current degree, of reserve pressure continued to represent a policy stance that was appropriately balanced in light of the opposing risks of a faltering economic expansion and a resurgence of inflation. Conditions in credit markets did not provide confirming evidence of the emergence of greater inflationary pressures and the need to restrain the growth in credit. Indeed, the continuation of balance sheet restructuring activities by financial institutions and the associated caution on the part of these institutions with regard to extending loans still appeared to be exerting a significantly inhibiting effect on the overall growth in spending and economic activity. Several members acknowledged that a policy of maintaining unchanged reserve conditions and an associated federal funds rate around current levels, which implied that real short-term rates were near zero or even slightly negative, could have inflationary consequences in the event of a strengthening of the expansion and a sustained pickup in credit demands. The Committee would need to remain alert to such a development. In present circumstances, however, an unchanged policy stance seemed most consistent with achieving sustained economic expansion in an environment of subdued inflation. The members who favored a prompt move toward restraint were persuaded that a steady policy incurred an unacceptable risk of halting the progress toward price stability and indeed of intensifying inflation as the current expansion matured. In this view, a policy tightening move at this point was likely to counter the need for more substantial and potentially disruptive tightening actions later.\n\nIn the course of the Committee discussion, the members took account of a staff analysis that pointed to a resumption of M2 and M3 growth over the months ahead. This analysis suggested that the temporary factors depressing the broader monetary aggregates likely would be reversing, but that the other influences causing a rechanneling of credit flows away from depository institutions and boosting the velocity of money undoubtedly would persist, though probably with diminishing force. Accordingly, the staff foresaw moderate growth of M2 and M3 that at midyear would leave these aggregates below the lower ends of the Committee's ranges for 1993. Under prevailing circumstances, such continuing weakness in the broader aggregates was not viewed as indicating inadequate monetary stimulus. Indeed, a number of members commented that other indicators suggested that current monetary policy was in fact quite accommodative as evidenced for example by low short-term interest rates, especially on an inflation-adjusted basis. Moreover, Ml, reserves, and the monetary base had continued to expand in the first quarter, though at much reduced rates. One member commented that the slowdown in these narrower monetary measures, which he viewed as important indicators of the thrust of monetary policy, had favorable implications with regard to bringing inflation under control. The members agreed that the considerable uncertainty that continued to surround the outlook for broad money relative to spending implied that forming precise expectations for monetary growth over the months ahead was not feasible.\n\nIn the Committee's discussion of possible intermeeting adjustments to the degree of reserve pressure, members who favored an unchanged policy stance also expressed a preference for retaining the symmetry of the existing directive. Some observed that a policy change during the intermeeting period, if any, might well be in the direction of a tightening move. However, because there was no compelling case in the view of most members for such a move at this time and any intermeeting adjustment would be made in the light of emerging developments, a symmetric directive was warranted. In this connection, one member commented that, given the Committee's assessment of current economic and financial conditions, a tilt in the directive toward restraint would give a misleading indication of the Committee's current intentions. Members also noted that a change in policy, should one be called for by intermeeting developments, would represent a shift in the direction of policy and would be likely to have an especially pronounced impact on financial markets.\n\nAt the conclusion of the Committee's discussion, a majority of the members indicated that they favored a directive that called for maintaining the existing degree of pressure on reserve positions. These members also expressed a preference for a directive that did not include a presumption about the likely direction of any adjustment to policy during the intermeeting period. Accordingly, in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, the Committee decided that slightly greater or slightly lesser reserve restraint would be acceptable during the intermeeting period. The reserve conditions contemplated at this meeting were expected to be consistent with a resumption of moderate growth in M2 and M3 over the second quarter.\n\nAt the conclusion of the meeting, the Federal Reserve Bank of New York was authorized and directed, until instructed otherwise by the Committee, to execute transactions in the System account in accordance with the following domestic policy directive:\n\nThe information reviewed at this meeting suggests that economic activity has increased at a more moderate pace in the early months of 1993 after expanding robustly in the fourth quarter. Total nonfarm payroll employment registered a sharp increase in February following generally small advances in previous months, and the civilian unemployment rate edged down further to 7.0 percent. Industrial production continued to post solid gains in January and February. Retail sales increased somewhat further over the first two months of the year after a fourth-quarter surge. Housing starts slipped in early 1993 after registering sizable gains late last year. Incoming data on orders and shipments of nondefense capital goods suggest a further brisk advance in outlays for business equipment, while nonresidential construction has remained soft. The nominal U.S. merchandise trade deficit was essentially unchanged in January from its average level in the fourth quarter, but both exports and imports were substantially lower. Increases in wages have remained subdued, but recent advances in consumer and producer prices have been larger than those recorded in the latter part of 1992.\n\nShort-term interest rates have changed little since the Committee meeting in early February while bond yields have fallen appreciably on balance. In foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies declined on balance over the intermeeting period.\n\nM2 and M3 contracted in January and February, apparently reflecting transitory factors and further shifts into market investments. Total domestic nonfinancial debt appears to have expanded somewhat further in January.\n\nThe Federal Open Market Committee seeks monetary and financial conditions that will foster price stability and promote sustainable growth in output. In furtherance of these objectives, the Committee at its meeting in February established ranges for growth of M2 and M3 of 2 to 6 percent and 1/2 to 4-1/2 percent respectively, measured from the fourth quarter of 1992 to the fourth quarter of 1993. The Committee expects that developments contributing to unusual velocity increases are likely to persist during the year. The monitoring range for growth of total domestic nonfinancial debt was set at 4-1/2 to 8-1/2 percent for the year. The behavior of the monetary aggregates will continue to be evaluated in the light of progress toward price level stability, movements in their velocities, and developments in the economy and financial markets.\n\nIn the implementation of policy for the immediate future, the Committee seeks to maintain the existing degree of pressure on reserve positions. In the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint or slightly lesser reserve restraint would be acceptable in the intermeeting period. The contemplated reserve conditions are expected to be consistent with a resumption of moderate growth in the broader monetary aggregates over the second quarter.\n\nVotes for this action: Messrs. Greenspan, Corrigan, Boehne, Keehn, Kelley, LaWare, McTeer, Mullins, Ms. Phillips, and Mr. Stern.\n\nVotes against this action: Messrs. Angell and Lindsey.\n\nMessrs. Angell and Lindsey indicated that their concerns about the outlook for inflation prompted them to favor an immediate move to tighten reserve conditions. In their view, such an action was desirable not only to arrest the possible emergence of greater inflation but especially to promote further disinflation. They were persuaded that monetary policy currently was overly accommodative as suggested by various indicators such as recent data on consumer and producer prices, the upswing in commodity prices, the low level of real short-term interest rates, and what in their judgment was a relatively depressed foreign exchange value of the dollar given the comparative strength of the U.S. economy and international interest rate trends. They noted that the current federal funds rate was probably not sustainable in the long term and that a tightening move at this time might well avoid the need for more sizable and potentially disruptive policy actions later.\n\nMr. Angell also emphasized the risks associated with any policy that did not firmly maintain a disinflationary trend. As he interpreted historical precedents, the typical result of a policy that tolerated some inflation was an eventual rise in inflation leading to permanently higher interest rates with adverse effects on economic activity. Indeed, he supported unpegging the federal funds rate to counter incipient price pressures showing through in commodity and finished goods prices. He believed that a clear signal of the Committee's commitment to price level stability would stabilize the price of gold along with the exchange value of the dollar and thereby provide a climate for further reductions in long-and intermediate-term interest rates. Such an approach to policy not only would assure a continuing disinflationary trend and eventual price stability, with very favorable implications for financial markets and economic growth, but it would in his view preclude an unsettling tendency for the debt markets to weaken every time newly available data appeared to suggest that economic growth was strengthening and that further monetary policy tightening actions therefore might be in the offing. In sum, such a policy would provide for the achievement of the Committee's objective of sustaining progress toward price stability which he believed was necessary for maintaining recent higher labor productivity, a permanently higher savings rate, and a prolonged period of economic expansion.\n\nIt was agreed that the next meeting of the Committee would be held on Tuesday, May 18, 1993.\n\nThe meeting adjourned.\n\nNormand Bernard Deputy Secretary\n\nFOMC",
            "content_type": "fomc_minutes",
        },
        {
            "date": "1993-05-18",
            "speaker": "Alan Greenspan",
            "title": "FOMC Meeting Minutes",
            "text": "A meeting of the Federal Open Market Committee was held in the offices of the Board of Governors of the Federal Reserve System in Washington, D.C., on Tuesday, May 18, 1993, at 9:00 a.m.\n\nPRESENT:\n\nMr. Greenspan, Chairman Mr. Corrigan, Vice Chairman Mr. Angell Mr. Boehne Mr. Keehn Mr. Kelley Mr. LaWare Mr. Lindsey Mr. McTeer Mr. Mullins Ms. Phillips Mr. Stern\n\nMessrs. Broaddus, Jordan, Forrestal, and Parry, Alternate Members of the Federal Open Market Committee\n\nMessrs. Hoenig, Melzer, and Syron, Presidents of the Federal Reserve Banks of Kansas City, St. Louis, and Boston, respectively\n\nMr. Bernard, Deputy Secretary Mr. Coyne, Assistant Secretary Mr. Gillum, Assistant Secretary Mr. Mattingly, General Counsel Mr. Prell, Economist\n\nMessrs. R. Davis, Lang, Lindsey, Promisel, Rolnick, Rosenblum, Scheld, Siegman, and Slifman, Associate Economists\n\nMr. McDonough, Manager of the System Open Market Account\n\nMs. Greene, Deputy Manager for Foreign Operations\n\nMs. Lovett, Deputy Manager for Domestic Operations\n\nMr. Ettin, Deputy Director, Division of Research and Statistics, Board of Governors\n\nMr. Madigan, Associate Director, Division of Monetary Affairs, Board of Governors\n\nMr. Stockton, Associate Director, Division of Research and Statistics, Board of Governors\n\nMr. Hooper, Assistant Director, Division of International Finance, Board of Governors\n\nMr. Small,1 Section Chief, Division of Monetary Affairs, Board of Governors\n\nMs. Low, Open Market Secretariat Assistant, Division of Monetary Affairs, Board of Governors\n\nMessrs. T. Davis, Dewald, and Goodfriend, Senior Vice Presidents, Federal Reserve Banks of Kansas City, St. Louis, and Richmond, respectively\n\nMs. Browne, Mr. Judd, and Mses. Rosenbaum and White, Vice Presidents, Federal Reserve Banks of Boston, San Francisco, Atlanta, and New York, respectively\n\nMr. Eberts, Assistant Vice President, Federal Reserve Bank of Cleveland\n\n1. Attended portion of meeting relating to a report on a study entitled \"Operating Procedures and the Conduct of Monetary Policy: Conference Proceedings,\" edited by Marvin Goodfriend and David Small. This two-volume study has been designated Working Studies 1, Parts 1 and 2, of the Federal Reserve Board's Finance and Economic Discussion Series.\n\nBy unanimous vote, the minutes for the meeting of the Federal Open Market Committee held on March 23, 1993, were approved.\n\nThe Deputy Manager for Foreign Operations reported on developments in foreign exchange markets and on System transactions in foreign currencies during the period March 23, 1993, through May 17, 1993. By unanimous vote, the Committee ratified these transactions.\n\nThe Manager of the System Open Market Account reported on developments in domestic financial markets and on System open market transactions in government securities and federal agency obligations during the period March 23, 1993, through May 17, 1993. By unanimous vote, the Committee ratified these transactions.\n\nThe Committee then turned to a discussion of the economic outlook and the implementation of monetary policy over the intermeeting period ahead. A summary of the economic and financial information available at the time of the meeting and of the Committee's discussion is provided below, followed by the domestic policy directive that was approved by the Committee and issued to the Federal Reserve Bank of New York.\n\nThe information reviewed at this meeting suggested that the pace of the economic expansion had slowed in recent months. Business outlays for durable equipment had remained strong, but consumer spending had been quite sluggish, reflecting limited gains in employment and real labor income and diminished optimism about near-term economic prospects. Additionally, U.S. exports continued to be constrained by the disappointing performance of the major foreign industrial economies. Available data indicated relatively modest growth in payroll employment and industrial production over recent months. Despite the considerable slack in the economy, increases in wages and prices had been appreciably larger thus far in 1993 than in the second half of last year.\n\nTotal nonfarm payroll employment rose only slightly on balance over March and April after registering sizable increases in the first two months of the year. Strong job gains in the services industry, notably in business and health services, were offset in considerable measure by job losses in manufacturing and construction in March and April. In manufacturing, reductions in payrolls were widespread, with particularly large declines at manufacturers of transportation equipment. Construction employment recovered only partially in April from the weather-related decline in March. The civilian unemployment rate remained at 7.0 percent.\n\nIndustrial production, after having posted solid gains in previous months, was little changed in March and April. Part of the recent sluggishness reflected a decline in utility output following a weather-related runup in February, but manufacturing output also grew more slowly. In the transportation industry, motor vehicle assemblies edged down and production of civilian aircraft remained weak over March and April. Elsewhere, the output of consumer goods other than motor vehicles was about unchanged, and the continuing strength in the computer industry contrasted with sluggish production of other types of business equipment. Total utilization of industrial capacity changed little over the two months.\n\nRetail sales increased substantially in April, reversing the weather-related decline in March; automotive dealers reported large sales gains in April, and expenditures at other retail outlets retraced part of the March decrease. For the year to date, however, retail sales had been lackluster after the strong increases of the latter part of 1992. Housing starts picked up in April; both single-family and multifamily starts rebounded from weather-depressed March levels.\n\nBusiness fixed investment advanced further during the first quarter of 1993, with another sizable rise in outlays for equipment outweighing continued weakness in nonresidential construction. Shipments of nondefense capital goods during the first quarter were paced by another sharp increase in shipments of office and computing equipment. By contrast, business spending for transportation equipment generally exhibited little strength; although sales of heavy trucks continued to trend up, outlays for complete aircraft apparently edged down further. Recent data on orders for nondefense capital goods other than aircraft suggested further expansion in business spending for equipment in the near term. Nonresidential construction activity was mixed in the first quarter. Office construction declined considerably further in response to the depressing effects of a continuing overhang of unoccupied space. On the other hand, building activity in the public utilities sector continued to trend up, and the construction of commercial structures other than office buildings increased for a second consecutive quarter.\n\nBusiness inventories appeared to have risen in the first quarter. Manufacturing inventories expanded in both February and March after a series of declines that began early in the fall; much of the recent advance occurred in the durable goods sector, where shipments were strong, and the ratio of inventories to shipments fell for manufacturing as a whole. Wholesale inventories increased appreciably in March. However, the inventory-to-sales ratio for the sector moved up only slightly, and it remained near the low end of its range over the past two years. In the retail sector, available data indicated that inventories rose appreciably over January and February but that the inventory-to-sales ratio remained in the narrow range that had prevailed over the preceding year.\n\nThe nominal U.S. merchandise trade deficit in February was unchanged from its January level, reflecting little change in total exports and total imports. For January-February combined, however, the trade deficit was slightly below its average level for the fourth quarter, with both exports and imports down considerably from their fourth-quarter levels. Much of the drop in exports reflected a reversal of an earlier, largely transitory runup in aircraft and automotive products. The decline in imports was spread across all major trade categories; imports of aircraft and miscellaneous industrial supplies dropped appreciably, and imports of consumer goods fell further. Recent indicators pointed to further weakness in economic activity in continental Europe and Japan through the first quarter. Elsewhere, the recovery in the United Kingdom appeared to be firming, and growth continued at a modest pace in Canada.\n\nProducer prices of finished goods rose more rapidly in March and April, partly as a result of sharp increases in the prices of finished energy goods in March and in the prices of finished foods in April. Excluding the food and energy components, producer prices advanced over the first four months of 1993 at a faster pace than in 1992. At the consumer level, the increase in prices of nonfood, non-energy items over the March-April period was smaller than the outsized change over the first two months of the year; nevertheless, averaging over the first four months of the year, the rate of increase in consumer prices was higher than in 1992. The deceleration of labor costs also appeared to have stalled in 1993. Average hourly earnings of production or nonsupervisory workers had grown more rapidly thus far this year than in 1992, and total hourly compensation of private industry workers rose at a faster pace in the first quarter of 1993 than in any quarter of last year.\n\nAt its meeting on March 23, the Committee adopted a directive that called for maintaining the existing degree of pressure on reserve positions and that did not include a presumption about the likely direction of any adjustments to policy during the intermeeting period. Accordingly, the directive indicated that in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint or slightly lesser reserve restraint would be acceptable during the intermeeting period. The reserve conditions associated with this directive were expected to be consistent with a resumption of moderate growth in M2 and M3 over the second quarter.\n\nOpen market operations during the intermeeting period were directed toward maintaining the existing degree of pressure on reserve positions. Expected levels of adjustment plus seasonal borrowing were raised during the period in anticipation of some increase in seasonal borrowing. Adjustment plus seasonal borrowing was near or a little above expected levels, except for a surge at the end of the first quarter, and the federal funds rate remained close to the 3 percent level that had prevailed for an extended period.\n\nShort-term interest rates changed little over the period since the March meeting. Long-term rates rose considerably early in the period when a sharp increase in average hourly earnings and some upward pressure on commodity prices sparked fears among market participants of a buildup in inflation pressures. Subsequently, despite growing doubts about the fate of the deficit reduction program, bond yields declined in response to a series of more favorable readings on price behavior and to indications of a slowing of the economic expansion. Adverse news about consumer and producer prices rekindled inflation concerns late in the period, and bond rates once again moved higher. On balance, most long-term market rates rose somewhat over the period. Despite unexpectedly favorable earnings reports for many firms, major indexes of stock prices were narrowly mixed over the period.\n\nIn foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies declined somewhat on balance over the intermeeting period. The dollar depreciated considerably more against the Japanese yen than against the German mark. A variety of factors contributed to the dollar's weakness, including indications of renewed sluggishness in U.S. economic activity, diminished prospects for a fiscal stimulus package, and market perceptions over much of the intermeeting period of limited official support for concerted actions to support the dollar against the yen. After falling to a historical low against the yen in late April, the dollar tended to stabilize following Treasury Secretary Bentsen's clarification of the Administration's exchange rate policy and intervention purchases of dollars against yen in a coordinated operation. Later in the period, the dollar rose somewhat against European currencies as the outlook for economic activity in Europe became more pessimistic.\n\nM2 contracted slightly on balance over March and April, while M3 was unchanged over the two months; both monetary aggregates increased substantially in early May. Much of the weakness in M2 over the March-April period owed to a smaller volume of nonwithheld tax payments in April of this year that reduced the need for a buildup in deposits to fund these payments. Abstracting from this temporary depressant, weak underlying growth in M2 continued to reflect the relatively attractive returns available on capital market instruments such as bond and stock mutual funds, which experienced heavy inflows during the two-month period. Total domestic nonfinancial debt expanded somewhat further through March.\n\nThe staff projection prepared for this meeting suggested that economic activity would grow at a moderate pace and that such growth would foster a gradual reduction in margins of unemployed labor and capital. The projection continued to incorporate the essential elements of the Administration's fiscal package, excluding that portion of the short-run stimulus initiative that seemed unlikely to be enacted by the Congress. Although the outlook for fiscal policy now seemed somewhat more contractionary than earlier, the sizable declines in long-term interest rates that had occurred in recent months were expected to support substantial additional gains in business and residential investment. Moreover, the increasingly favorable financial environment associated with expected further easing of credit supply constraints and the ongoing strengthening of balance sheets would tend to buttress private spending on housing, consumer durables, and business equipment. Increases in export demand would be damped in the near term by the continuing weakness in the economies of the major industrialized countries. The persisting slack in resource utilization was expected to be associated with a return to more subdued price increases after a spurt earlier in the year.\n\nIn the Committee's discussion of current and prospective economic conditions, the members focused with some concern on the evidence of a slower economic expansion and a higher rate of inflation since late 1992. While recent indicators of economic activity were disappointing, the expansion nonetheless appeared to have sustainable momentum and the members generally viewed moderate growth in line with, or perhaps a bit below, their February forecasts as a reasonable expectation. At the same time, several emphasized that the outlook was subject to substantial uncertainty stemming to an important extent from the unsettled course of legislation aimed at reducing the federal deficit. Members expressed particular concern about the rise in various measures of inflation over the past several months. The increase seemed to reflect temporary factors and a worsening in inflationary expectations rather than any significant change in economic fundamentals. Accordingly, it was premature in the view of many members to conclude that the inflationary trend had tilted upward. Even so, higher inflation expectations, if sustained, would be detrimental to economic performance, and the risks of an uptrend in inflation clearly had increased.\n\nIn their review of business developments across the nation, members continued to report uneven conditions ranging from apparently moderate gains in some parts of the country to mixed or marginally declining activity in others. Business confidence had deteriorated in many areas and firms were trimming or putting on hold new or expanded spending programs pending a resolution of federal tax and spending proposals, including prospective health care reform, and the outcome of proposed tax legislation in some states as well. Cautious business attitudes were reflected in continuing efforts to constrain costs and to hold down or reduce employment levels, notably of permanent workers in light of the large non-wage costs associated with full-time workers. Accordingly, while some job growth was occurring, especially outside major firms and the defense sector, business firms generally appeared disposed to continue to meet increases in demand through overtime work and temporary workers, and members anticipated that such attitudes were likely to persist in the absence of a major improvement in business confidence.\n\nAs reflected in the available data for the national economy, anecdotal reports from around the country suggested generally lackluster retail sales over the first four months of the year. To an extent, this development probably involved some retrenchment in consumer spending following an unsustainable surge during the latter part of 1992. In some parts of the country, unusually severe weather conditions also had served to hold down retail sales earlier this year and recovery from that slowdown had tended to be limited thus far, especially outside the automotive sector. Looking ahead, the members continued to anticipate that consumer spending would provide moderate support for a sustained economic expansion.\n\nDespite the cautious business attitudes about the economic outlook, spending for business equipment had continued to help maintain the expansion. Encouraged in part by relatively low interest rates, receptive financial markets, and the more aggressive lending policies of some depository institutions, many firms were upgrading equipment to reduce costs and improve their product offerings. Concurrently, however, numerous firms reported that they were holding off on making major new investment commitments and in some cases were revising down earlier expansion plans in light of prevailing economic uncertainties, notably those generated by the current legislative debate about federal taxes and spending. Nonresidential construction remained uneven and on the whole relatively subdued across the nation. The construction of new office structures was likely to stay depressed in much of the country as overcapacity continued to be worked down, but members saw indications of some strengthening in industrial and commercial building activity and in public works projects in some areas.\n\nTurning to the outlook for the nation's trade balance, some members referred to quite gloomy assessments from business contacts and other sources regarding current economic conditions in a number of major industrial nations and the associated prospect of little or no growth in U.S. exports to such countries. While total U.S. exports might continue to expand, reflecting sizable gains in some parts of the world, imports probably would grow at a somewhat faster pace, given moderate expansion in domestic demand in line with the members' expectations. At the same time, members expressed concern about the potential impact of growing protectionist sentiment on current trade negotiations and on the longer-run outlook for domestic industries and parts of the country that relied on foreign trade.\n\nWith regard to the inflation situation, members commented that it remained difficult to find a satisfactory explanation for the faster-than-projected increases in price measures thus far this year. Although temporary anomalies seemed to be involved, including measurement problems and special factors boosting some prices, higher inflation expectations also might have been playing a key role. The latter seemed to have intensified in the last month or two, perhaps as a result of growing concerns that significant deficit-reduction legislation might not be enacted. Strong competitive pressures in many markets, including competition from foreign producers, still appeared to be restraining or precluding price increases by many business firms, but efforts to raise prices seemed to be encountering somewhat less resistance recently than earlier in the economic expansion. Some price increases appeared to be associated with the earlier surge in demand, and in the case of one key industry higher prices had been facilitated by the implementation of import restrictions. The downtrend in labor compensation inflation also seemed to have stalled in recent months. Against this background, a considerable degree of uncertainty surrounded the outlook for inflation and the members differed to some extent with regard to the most likely outcome. A number of members, while they did not rule out the possibility of a more favorable result, stressed the risk that a faster rate of inflation might well tend to be sustained. Others gave more emphasis to the still considerable slack in labor and product markets and to the restrained growth in broad measures of money and credit. In this view, an inflation rate in the quarters ahead more in line with their earlier forecasts was still a reasonable expectation even though the average rate for the year as a whole was likely to be higher than they had forecast at the start of the year.\n\nIn the Committee's discussion of policy for the intermeeting period ahead, many of the members commented that recent price and wage developments were troubling but did not point persuasively at this juncture toward an extended period of higher inflation. In light of underlying economic and financial conditions, the upturn in inflation expectations and the somewhat quickened pace of inflation might well prove to be temporary. The economy was expanding, but the pace had slowed in recent months. On the other hand, the potential for a sustained increase in the rate of inflation could not be dismissed. Waiting too long to counter any emerging uptrend in inflation or further worsening in inflationary expectations would exacerbate inflationary pressures and require more substantial and more disruptive policy moves later. Indeed, in one view sensitive commodity prices and other key measures of inflation already indicated the need for a prompt move toward restraint, especially in the context of the Committee's objective of fostering progress toward price stability. However, the other members all supported a proposal to maintain an unchanged degree of pressure on reserve positions at this time.\n\nIn the course of the Committee's discussion, the members took account of a staff analysis that pointed to a considerable pickup in the growth of M2 and M3 over the months of May and June. Such strengthening, which appeared to have emerged in early May, was associated in part with the reversal of earlier tax-related distortions and with a surge in prepayments of mortgage-backed securities. Monetary growth was expected to revert to a more modest pace over subsequent months, and the members recognized that in any event the interpretation of monetary growth rates needed to be approached with considerable caution in a period when traditional relationships of such growth to aggregate measures of economic performance were not reliable. In present circumstances, M2 and M3 no longer seemed to be good barometers of underlying liquidity, which appeared to be ample. One member expressed the view that the relatively robust growth of M1 and reserves served as a better indicator of the thrust of monetary policy than did the broader monetary aggregates.\n\nIn the view of a majority of the members, wage and price developments over recent months were sufficiently worrisome to warrant positioning policy for a move toward restraint should signs of intensifying inflation continue to multiply. In addition to new information on prices and costs, such signs could include developments in markets affected by inflation psychology, such as those for bonds, foreign exchange, and sensitive commodities, all of which would need to be monitored carefully. These members supported a directive that incorporated a greater predilection to tighten than to ease over the intermeeting period. Given the special nature of current inflation concerns and attendant uncertainties, however, the Committee agreed with a proposal by the Chairman that an intermeeting consultation would be appropriate in the event that a tightening move were to be contemplated during this period. If a policy tightening action were not needed, an asymmetric directive would nonetheless underscore the Committee's concern about recent inflation readings and its judgment that a policy to encourage progress toward price stability would promote sustained economic growth. In the event that a tightening action became necessary, such action could help to moderate inflationary expectations, with positive implications over time for long-term interest rates and the performance of the economy. Monetary policy would still be stimulative after a modest tightening move in that such a move would leave short-term interest rates close to or even below their year-ago levels in real terms, given the interim rise in inflation.\n\nSome members preferred to retain a directive that did not incorporate a presumption about the likely direction of a change in policy, if any, during the intermeeting period. They were concerned that adopting a biased directive might prove to be an overreaction to temporary factors and to a short-lived upturn in inflationary sentiment that was not warranted by underlying economic conditions. They noted that, if called for by intermeeting developments, a move toward restraint could be implemented from a symmetric directive. More fundamentally, they believed that the circumstances surrounding the recent performance of the economy and the uncertainties about price developments suggested the need for considerable caution before any policy tightening was implemented and that such a policy move should be carried out only in the light of information that pointed clearly to the emergence of higher inflation. Nonetheless, all but one of these members could accept an asymmetric directive on the understanding that the Committee would have a chance to discuss any possible policy action.\n\nAt the conclusion of the Committee's discussion, all but two of the members indicated that they preferred or could accept a directive that called for maintaining the existing degree of pressure on reserve positions and that included a bias toward possible firming of reserve conditions during the intermeeting period. Accordingly, in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, the Committee decided that slightly greater reserve restraint would be acceptable or slightly lesser reserve restraint might be acceptable during the intermeeting period. The reserve conditions contemplated at this meeting were expected to be consistent with appreciable growth in the broader monetary aggregates over the second quarter.\n\nAt the conclusion of the meeting, the Federal Reserve Bank of New York was authorized and directed, until instructed otherwise by the Committee, to execute transactions in the System account in accordance with the following domestic policy directive:\n\nThe information reviewed at this meeting suggests that the economic expansion has slowed in recent months. Total nonfarm payroll employment rose only slightly over March and April after registering sizable increases earlier in the year, and the civilian unemployment rate remained at 7.0 percent. Industrial production was little changed in March and April after posting solid gains in previous months. Retail sales increased substantially in April but were about unchanged on balance for the year to date. Housing starts picked up in April. Incoming data on orders and shipments of nondefense capital goods suggest a further brisk advance in outlays for business equipment, while nonresidential construction has remained soft. The nominal U.S. merchandise trade deficit in January-February was slightly below its average level in the fourth quarter. Increases in wages and prices have been appreciably larger this year than in the second half of 1992.\n\nShort-term interest rates have changed little since the Committee meeting on March 23 while bond yields have risen somewhat. In foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies declined somewhat on balance over the intermeeting period.\n\nAfter contracting during the first quarter, M2 was unchanged in April while M3 turned up; both aggregates increased substantially in early May. Total domestic nonfinancial debt expanded somewhat further through March.\n\nThe Federal Open Market Committee seeks monetary and financial conditions that will foster price stability and promote sustainable growth in output. In furtherance of these objectives, the Committee at its meeting in February established ranges for growth of M2 and M3 of 2 to 6 percent and 1/2 to 4-1/2 percent respectively, measured from the fourth quarter of 1992 to the fourth quarter of 1993. The Committee expects that developments contributing to unusual velocity increases are likely to persist during the year. The monitoring range for growth of total domestic nonfinancial debt was set at 4-1/2 to 8-1/2 percent for the year. The behavior of the monetary aggregates will continue to be evaluated in the light of progress toward price level stability, movements in their velocities, and developments in the economy and financial markets.\n\nIn the implementation of policy for the immediate future, the Committee seeks to maintain the existing degree of pressure on reserve positions. In the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint would or slightly lesser reserve restraint might be acceptable in the intermeeting period. The contemplated reserve conditions are expected to be consistent with appreciable growth in the broader monetary aggregates over the second quarter.\n\nVotes for this action: Messrs. Greenspan, Corrigan, Keehn, Kelley, LaWare, Lindsey, McTeer, Mullins, Ms. Phillips, and Mr. Stern.\n\nVotes against this action: Messrs. Angell and Boehne.\n\nMr. Angell dissented because he believed that the persisting indications of rising inflation and the related deterioration in inflationary psychology called for a prompt move to tighten monetary policy. In his view, low real interest rates, a very steep yield curve, a surprisingly weak exchange value of the dollar along with the confirming price behavior of inflation-sensitive commodities such as gold underscored the need for Committee action to signal the System's continuing commitment to the eventual achievement of price stability. In his opinion, progress toward lower inflation was not likely in 1993 and 1994 in the absence of a monetary policy that was sufficiently restrictive to check inflationary expectations. He added that history demonstrated that a monetary policy focused primarily on developments in the real economy ran the risk of waiting too long to counter a worsening in inflationary expectations and thus requiring more substantial and possibly more disruptive policy changes later.\n\nMr. Boehne supported a steady policy course, but he dissented because he objected to a directive that was biased toward tightening. Although recent developments suggested that inflation would be somewhat higher and real growth somewhat lower during the year than had been expected earlier, he did not believe recent data indicated a fundamental shift in the outlook for inflation or the economy. He was concerned that adopting a biased directive might prove to be an overreaction to temporary factors affecting the inflation rate and inflationary sentiment. In his view, underlying economic conditions did not point toward an extended period of higher inflation. While the pace of economic growth conceivably could quicken, it seemed just as likely that the tempo of business and consumer spending could diminish in the face of uncertainty about the stance of fiscal policy, particularly with regard to potential tax increases. Given these uncertainties, he had a strong preference for keeping an open mind about possible Committee action during the intermeeting period and, accordingly, favored a balanced policy directive.\n\nIt was agreed that the next meeting of the Committee would be held on Tuesday-Wednesday, July 6-7, 1993.\n\nThe meeting adjourned at 1:50 p.m.\n\nNormand Bernard Deputy Secretary\n\nFOMC",
            "content_type": "fomc_minutes",
        },
        {
            "date": "1993-07-07",
            "speaker": "Alan Greenspan",
            "title": "FOMC Meeting Minutes",
            "text": "A meeting of the Federal Open Market Committee was held in the offices of the Board of Governors of the Federal Reserve System in Washington, D.C., on Tuesday, July 6, 1993, at 2:30 p.m. and continued on Wednesday, July 7, 1993, at 9:00 a.m.\n\nPRESENT:\n\nMr. Greenspan. Chairman Mr. Mullins1 Mr. Angell Mr. Boehne Mr. Keehn Mr. Kelley Mr. LaWare Mr. Lindsey Mr. McTeer Mr. Oltman2 Ms. Phillips Mr. Stern\n\nMessrs. Broaddus, Jordan. Forrestal, and Parry, Alternate Members of the Committee\n\nMessrs. Hoenig, Melzer, and Syron, Presidents of the Federal Reserve Banks of Kansas City, St. Louis, and Boston, respectively\n\nMr. Kohn, Secretary and Economist Mr. Bernard, Deputy Secretary Mr. Coyne, Assistant Secretary Mr. Gillum, Assistant Secretary Mr. Mattingly, General Counsel Mr. Patrikis, Deputy General Counsel Mr. Prell, Economist Mr. Truman, Economist Messrs. R. Davis, Lang, Lindsey, Promisel, Rolnick, Rosenblum, Scheld, Siegman, Simpson, and Slifman, Associate Economists\n\nMr. McDonough, Manager of the System Open Market Account\n\nMs. Greene, Deputy Manager for Foreign Operations\n\nMs. Lovett, Deputy Manager for Domestic Operations\n\nMr. Madigan, Associate Director, Division of Monetary Affairs, Board of Governors\n\nMr. Stockton, Associate Director, Division of Research and Statistics, Board of Governors\n\nMs. Danker, Assistant Director, Division of Monetary Affairs, Board of Governors\n\nMessrs. Small,3 and Whitesell,4 Section Chiefs, Division of Monetary Affairs, Board of Governors\n\nMs. Kusko,4 Senior Economist, Division of Research and Statistics. Board of Governors\n\nMs. Low, Open Market Secretariat Assistant, Division of Monetary Affairs, Board of Governors\n\nMessrs. Beebe, J. Davis, T. Davis, Goodfriend, and Ms. Tschinkel, Senior Vice Presidents, Federal Reserve Banks of San Francisco. Cleveland, Kansas City, Richmond, and Atlanta, respectively\n\nMr. McNees, Vice President, Federal Reserve Bank of Boston,\n\nMessrs. Coughlin and Guentner, Assistant Vice Presidents. Federal Reserve Banks of St. Louis and New York, respectively\n\n1. Acting Vice Chairman in Mr. Corrigan's absence.\n\n2. First Vice President, Federal Reserve Bank of New York, attending as alternate member for Mr. Corrigan.\n\n3. Attended portion of meeting relating to a discussion of the uses of a broad monetary aggregate that includes bond and stock mutual funds.\n\n4. Attended portion of meeting relating to the Committee's discussion of the economic outlook and its longer-run growth objectives for monetary and debt aggregates.\n\nBy unanimous vote, the minutes for the meeting of the Federal Open Market Committee held on May 18, 1993, were approved.\n\nThe Manager of the System Open Market Account reported on developments in foreign exchange markets and on System transactions in foreign currencies during the period May 18, 1993, through July 6, 1993. By unanimous vote, the Committee ratified these transactions.\n\nThe Deputy Manager for Domestic Operations reported on developments in domestic financial markets and on System open market transactions in government securities and federal agency obligations during the period May 18, 1993, through July 6, 1993. By unanimous vote, the Committee ratified these transactions.\n\nThe Committee then turned to a discussion of the economic outlook, the ranges for the growth of money and debt in 1993 and 1994, and the implementation of monetary policy over the intermeeting period ahead. A summary of the economic and financial information available at the time of the meeting and of the Committee's discussion is provided below, followed by the domestic policy directive that was approved by the Committee and issued to the Federal Reserve Bank of New York.\n\nThe information reviewed at this meeting provided a mixed reading on the economy, but on balance the available data suggested that the expansion had picked up somewhat during the second quarter from the very slow pace of the first quarter. Employment statistics, while tending to soften in June, pointed to considerable strength for the second quarter as a whole, although recent spending indicators suggested a much more moderate expansion. Consumer and producer price inflation slowed substantially in May, but prices had risen at a faster rate over the first five months of the year than over the second half of 1992.\n\nTotal nonfarm payroll employment changed little in June after registering substantial gains in April and May. For the second quarter as a whole, the increase in employment matched that of the first quarter. Manufacturing employment, which was about unchanged over the first quarter, declined somewhat in June for a third straight month. Construction payrolls edged lower after rising appreciably over the preceding two months, and job gains in the services industries were considerably smaller in June than those recorded earlier in the year. The civilian unemployment rate backed up to 7.0 percent in June.\n\nIndustrial production increased in May at the relatively subdued rate recorded in March and April; for June, the limited data available suggested a modest decline in output. In May, assemblies of motor vehicles declined after holding steady over the two previous months. Among other manufactured goods, the production of business equipment, led by computers and industrial equipment, recorded another brisk advance whereas the output of non-auto consumer goods continued to expand sluggishly. Total utilization of industrial capacity was unchanged in May for a third straight month.\n\nReal personal consumption expenditures edged higher in May after a sizable rebound in April from weather-related weakness. On balance, however, consumer spending had increased only slightly thus far this year. Outlays for new cars and light trucks advanced in May to their highest level since January 1990 and apparently remained near that elevated level in June. In addition, spending for non-energy services had increased substantially in recent months. By contrast, energy consumption had fallen from the especially high levels of late winter, and outlays for nondurable goods in May were still below their fourth-quarter level. Housing starts increased in April and May from a depressed first-quarter pace, with most of the rise attributable to starts of single-family dwellings.\n\nShipments of nondefense capital goods in May retraced only a portion of a sizable April decline. However, for the two months combined, shipments of such goods were above the average for the first quarter and apparently remained on an upward trend that began early in 1992. The upward trajectory for spending on machinery and on electrical and communications equipment seemed to have reflected improved cash flows for the business sector and a declining cost of capital, and incoming data suggested that outlays for business equipment would increase further over the months ahead. Nonresidential construction activity was unchanged over the first quarter but picked up slightly on balance over April and May. Office building rose over the two months, while construction of non-office commercial structures was little changed and industrial building activity was down sharply.\n\nBusiness inventories recorded another appreciable rise in April, and available data pointed to a further increase in May. In manufacturing, inventory accumulation stepped up in April and May after changing little in the first quarter: the ratio of stocks to shipments edged higher in each month and was only slightly above the very low level reached early in 1993. In the wholesale trade sector, inventories advanced at a slower rate in May than in April, and the inventory-to-sales ratio fell to the low end of the range for the past three years. The buildup of retail inventories slowed considerably in April, and with sales rebounding from the effect of March storms, the inventory-to-sales ratio declined for the retail sector. Nonetheless, the ratio still was near the high end of its range for the past several years.\n\nThe nominal U.S. merchandise trade deficit for April was unchanged from March, with both imports and exports declining slightly. However, the April deficit was substantially above the average for the first quarter, reflecting sizable increases in imports of capital goods, automotive products, consumer goods, and oil. The value of exports in April was only slightly above the first-quarter average. Recent indicators pointed to further weakness in economic activity in continental Europe thus far this year. By contrast, economic recovery appeared to be continuing in the United Kingdom and Canada. In Japan, economic activity was up appreciably in the first quarter, but available data suggested that this strength had not carried over to the second quarter.\n\nChanges in producer and consumer prices were small in May following sizable increases earlier in the year. Producer prices of finished goods were unchanged in May, as declines in prices of finished consumer food and energy products offset small advances in prices of other finished goods. Excluding the food and energy components, producer prices had risen more rapidly thus far in 1993 than they had in the second half of 1992. At the consumer level, prices of items other than food and energy rose only slightly in May, but this measure of inflation also had risen faster this year than in the second half of last year. Labor costs likewise had evidenced a quickened pace of increases this year. Average hourly earnings of production or nonsupervisory workers rose substantially in May after edging lower in April, and these earnings had grown more rapidly over the first five months of 1993 than over the preceding six months.\n\nAt its meeting on May 18, the Committee adopted a directive that called for maintaining the existing degree of pressure on reserve positions but that included a tilt toward possible firming of reserve conditions during the intermeeting period. Accordingly, the directive indicated that in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint would be acceptable or slightly lesser reserve restraint might be acceptable during the intermeeting period. The reserve conditions associated with this directive were expected to be consistent with appreciable growth of the broader monetary aggregates over the second quarter.\n\nOpen market operations were directed toward maintaining the existing degree of pressure on reserve positions throughout the intermeeting period. Several upward adjustments were made to expected levels of adjustment plus seasonal borrowing during the period in anticipation of stepped-up demand for seasonal credit during the crop-growing season: borrowing averaged near expected levels over the period. The federal funds rate remained close to 3 percent over the period, although quarter-end pressures in money markets pushed the rate higher for a brief period at the end of June.\n\nOther short-term interest rates also were little changed on balance over the period since the May meeting. Early in the period, unexpectedly robust employment data for May, coupled with media reports about the monetary policy stance adopted at the May meeting, led to some upward pressure on money market interest rates. Subsequently, however, this pressure abated in response to the release of data suggesting slower inflation and a somewhat weaker outlook for the economy. These developments along with the progress in the Congress toward adoption of a deficit-reduction package fostered a decline in bond yields: buoyed by the drop in yields, major indexes of stock prices rose over the intermeeting period in spite of disappointing second-quarter earnings reports by several major companies.\n\nIn foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies increased on balance over the intermeeting period. After depreciating somewhat through the end of May, the dollar recovered in early June when U.S. money market interest rates moved higher. The dollar rose more strongly over the last half of June, principally in response to actual and expected monetary easing abroad. The rise in the dollar over the intermeeting interval reflected sizable appreciations against European currencies, especially the German mark. The dollar continued to fall against the Japanese yen through the middle of June, in the process setting several new lows, before recovering a little over the remainder of the period.\n\nAfter contracting during the first quarter, M2 and M3 expanded appreciably over the second quarter. Most of this growth, which was especially pronounced in May, reflected strength in Ml and occurred despite continued heavy outflows to bond and equity funds. The May surge resulted in part from a strong pickup in mortgage refinancing activity and a reversal of the depressing effect in April of relatively damped individual nonwithheld tax payments on the seasonally adjusted level of liquid deposits. The growth of the broader aggregates moderated substantially in June, and by more than might have been suggested by the waning of these mortgage and tax influences. For the year through June, growth of both M2 and M3 was below the lower ends of the ranges for 1993 that the Committee had established in February. This sluggishness reflected ongoing changes in asset preferences and financing patterns rather than restrictive financial conditions. The velocity of M2 was estimated to have increased at a rate of about 4-1/2 percent over the first half of the year after a 4 percent rise in 1992. Total domestic nonfinancial debt expanded somewhat further through April.\n\nThe staff projection prepared for this meeting suggested moderate growth in economic activity and modest reductions in margins of unemployed labor and capital through 1994. The projection assumed the enactment of a federal budget bill that implied a moderately restrictive fiscal policy over the forecast horizon. As in earlier staff projections, lower interest rates were expected to support appreciable gains in interest-sensitive expenditures, including housing, consumer durables, and business equipment. Private spending also would be buttressed by a favorable financial environment associated with strengthened balance sheets and reduced debt burdens and by the apparently increasing willingness of banking institutions to make new loans. Export demand was likely to remain constrained over the near term by the weakness in the economies of several major industrial countries, but some improvement in foreign demand was anticipated later as those economies started to strengthen. The outlook for moderate growth and continuing slack in resource utilization suggested considerably more subdued price increases than had occurred in the early months of 1993.\n\nIn the Committee's discussion, the members generally agreed that ongoing economic developments remained consistent with moderate but sustained growth in economic activity. No sector of the economy seemed poised at this juncture to provide strong impetus to the expansion, but a promising basis for further growth was seen in the much improved financial condition of many households and business firms. Lower long-term interest rates, which had contributed to the improvement in balance sheets, were likely as well to bolster housing and business capital spending more directly. While the expansion now appeared to be firmly established, a number of members cautioned that it was subject to some downside risks, notably those associated with the still uncertain outlook for government budget and other policies. The possibility of higher taxes, associated with the deficit reduction legislation currently under consideration in the Congress and with the forthcoming proposals for national health care reform, was widely reported to be damping spending. With regard to the outlook for inflation, the most recent data on prices offered some encouragement that the earlier upturn in key measures of inflation might prove to be temporary, especially in the context of still ample margins of unutilized labor and other production resources. Even so, given generally held expectations among the public that inflation was not likely to decline and might in fact trend higher, many members concluded that for now the disinflationary trend might have been arrested or, at least, that further progress toward price stability would be quite difficult to achieve over the next several quarters.\n\nIn conformance with the usual practice at meetings when the Committee considers its longer-run objectives for growth of the monetary and debt aggregates, the members of the Committee and the Federal Reserve Bank presidents not currently serving as members provided individual projections of growth in real and nominal GDP, the rate of unemployment, and the rate of inflation for the years 1993 and 1994. In light of the experience in the first half of the year, forecasts of real growth in 1993 had been revised down somewhat since February, while projections of inflation had been raised. The central tendency of the forecasts of the rate of expansion in real GDP in 1993 was now 2-1/4 to 2-3/4 percent for the year as a whole; for 1994, these projections had a central tendency of 2-1/2 to 3-1/4 percent. With regard to the expansion of nominal GDP, the forecasts converged on growth rates of 5 to 5-3/4 percent for 1993 and 5 to 6-1/2 percent for 1994. Given the projections of a moderate uptrend in economic activity and expectations of some further gains in labor productivity, the forecasts incorporated only a small decline in unemployment to rates of around 6-3/4 percent in the fourth quarter of 1993 and only slightly lower by the fourth quarter of 1994. For the rate of inflation, as measured by the CPI, the projections had a central tendency of 3 to 3-1/4 percent in 1993 and 3 to 3-1/2 percent in 1994, reflecting little change in both years from the rate of inflation experienced in 1992.\n\nMembers commented that the improved prospects for significant reductions in the federal deficit had played an important role in fostering the declines in longer-term interest rates that had occurred since the latter part of 1992: the lower rates were having positive effects on spending decisions in a number of interest-sensitive sectors of the economy as well as on balance sheets more generally. At the same time, the prospects for higher taxes--accentuated by uncertainties about their size and incidence--were widely reported to be inhibiting spending decisions by business firms and might also be adding to cautious consumer attitudes. Some of the anecdotal evidence suggested that uncertainties associated with the potential impact of the still unannounced proposals for health care reform were making many businesses especially cautious, notably in their hiring decisions. Adding to the effects of anticipated federal legislation were concerns in various parts of the country about further cuts in defense spending and the impact of additional reductions in state and local expenditures or of increases in state and local taxes. Some members observed that the fiscal policy legislation before the Congress appeared to have generated a perhaps exaggerated degree of concern, and passage of this legislation might have a generally favorable effect on business and consumer sentiment.\n\nTurning to the outlook for individual sectors of the economy, members referred to indications of an upturn in consumer spending in recent months, but they also noted that survey results and anecdotal reports still suggested generally cautious consumer attitudes. The prospects for increased taxes might be having some negative effect on consumer confidence, but consumers remained especially concerned about the outlook for jobs and incomes as defense cutbacks continued and many firms, notably larger business establishments, took further steps to restructure and downsize their operations. To an important extent the improvement in retail sales in the second quarter was associated with stronger sales of motor vehicles that, in the view of at least some members, appeared to reflect previously postponed replacement demand rather than a major shift in consumer attitudes. In any event, moderate growth in consumer spending was likely to be maintained in the context of the improved financial condition and the related reduction in debt-service burdens of many households. Further growth in overall employment, in line with that achieved in the first half of the year, would if it persisted provide important support toward sustaining the expansion of consumer spending and thus the growth of the economy more generally.\n\nWith regard to the outlook for business fixed investment, members reported that many firms were scaling back or putting on hold their capital spending plans pending a resolution of the business tax proposals under consideration in the Congress. Nonetheless, business spending for equipment still constituted a relatively robust sector of the economy, at least according to the data available to date. To a considerable extent, such spending reflected ongoing efforts to improve the quality of products and the efficiency of business operations while holding down the number of employees, and the members saw this trend as likely to continue. In general, other business capital spending had remained sluggish, although construction activity other than office building appeared to have picked up in parts of the country. The prospects for housing construction, though not ebullient, were viewed as more promising particularly in light of the declines in mortgage interest rates to relatively low levels. The improved financial position of many potential homebuyers also provided a basis for anticipating stronger housing markets. Despite these favorable factors, however, overall housing activity had improved only modestly in recent months as homebuyers tended to remain cautious, and at least in some areas housing developers continued to report that they were encountering difficulties in securing construction finance. On balance, housing construction seemed likely to provide some impetus to the expansion in coming quarters.\n\nRelatively weak economic conditions in a number of foreign industrial countries were likely to continue to limit U.S. exports, which had declined since the end of 1992. Indeed, available data supplemented by reports from a variety of contacts suggested that business conditions had remained quite weak or had worsened in a number of foreign industrial nations. Even so, business contacts in some parts of the United States indicated that foreign demand for their products was still quite robust. Business activity abroad, which already was trending higher in a few industrial nations, was viewed as likely to strengthen more generally over the year ahead, with positive effects on overall U.S. exports.\n\nTurning to the outlook for inflation, members commented that despite favorable readings recently, a wide range of price and wage data had suggested some acceleration in the rate of inflation during the early months of the year. To some extent, the indications of intensified inflation might have been the result of difficulties with seasonal adjustments or other temporary factors, but there were reports of some successful efforts by business firms to raise prices following the spurt in demand and the rise in capacity utilization toward the end of 1992. These price developments were disappointing and suggested to many members that the disinflationary trend might have been arrested, at least for now, though the economic fundamentals remained consistent with a resumption of some further downward movement in the rate of inflation. With regard to those fundamentals, many members saw significant, albeit diminished, slack in labor and product markets as likely to persist over the forecast horizon, given their current forecasts of moderate expansion in economic activity. Other favorable-factors in the inflation outlook included efforts by businesses to hold down costs and increase productivity by restructuring their operations and investing in new, more productive equipment. Unfortunately, these favorable elements in the underlying economic situation seemed at odds with the apparently widely held view by the public that inflation would not diminish and indeed was likely to increase and that in any event current inflation levels were tolerable. Such expectations and attitudes would tend to temper the gains against inflation, if any, over the forecast horizon by their effects on the pricing and wage behavior of business firms and employees and the reactions of consumers toward rising prices. This inflationary climate underscored the importance of credible government policies--monetary, fiscal, trade, and regulatory--that encouraged reduced inflation over time.\n\nIn keeping with the requirements of the Full Employment and Balanced Growth Act of 1978 (the Humphrey-Hawkins Act), the Committee at this meeting reviewed the ranges for growth in the monetary and debt aggregates that it had established in February for 1993 and it decided on tentative ranges for growth in those aggregates in 1994. The current ranges for the period from the fourth quarter of 1992 to the fourth quarter of 1993 included expansion of 2 to 6 percent for M2 and 1/2 to 4-1/2 percent for M3. A monitoring range for growth of total domestic nonfinancial debt had been set at 4-1/2 to 8-1/2 percent for 1993.\n\nIn the Committee's discussion, the members focused on the issue of whether or not to lower the ranges further. In February, the ranges for M2 and M3 had been reduced by 1/2 percentage point in the expectation that continuing rechanneling of credit demands and savings flows into securities markets and away from depository institutions would result in further increases in velocity, the ratio of nominal GDP to monetary measures such as M2 or M3. In fact, the strength of these forces was underestimated to some extent. Substantial increases occurred in the velocity of both M2 and M3, especially in the first quarter, that were reflected in weak bank credit and huge inflows into bond and stock mutual funds. In the circumstances, the expansion of both aggregates through midyear was below the lower ends of the reduced ranges established by the Committee for the year. According to a staff analysis, the developments boosting M2 and M3 velocity could be expected to persist over the balance of 1993. Such an outcome would imply monetary growth for the year as a whole slightly below the Committee's current ranges, even if the growth of nominal GDP picked up in the second half of the year as implied by the central tendency of the members' forecasts.\n\nIn light of this expectation, many of the members indicated their support of a proposal to lower the M2 and M3 ranges further for 1993 and on a tentative basis to retain the reduced ranges for 1994. It was emphasized during the discussion that the reductions were intended solely as technical adjustments to reflect expected increases in velocity and that the lower ranges did not imply any tightening of monetary policy. Rather, the reductions in the ranges would serve to align them with monetary growth rates that were more likely to be associated with a satisfactory economic performance. Indeed, M2 and M3 growth consistent with most members' forecasts might still leave the expansion of those aggregates near the lower ends of their reduced ranges for the year; at the same time, the probability of a surge in monetary growth to levels above the new ranges appeared remote. In this connection, some members commented that the uncertainties surrounding the behavior of M2 and M3 might well persist for some time. The value of these aggregates in guiding policy seemed to have diminished in 1992 and 1993, and the Committee needed to continue to rely on its evaluation of a broad array of other financial and economic developments in its assessment of an appropriate course for monetary policy. The members did not rule out the possibility that a more normal or predictable relationship between money and economic activity might be restored once the current process of balance sheet adjustments was completed, the yield curve flattened, and some stabilization in the intermediation function of depository institutions emerged. In the view of a few members, moreover, the lower range proposed for M2 might in fact be more consistent with the rate of monetary growth that would be needed over the long term to sustain price stability and satisfactory economic expansion, if the earlier relationships between broad money growth and economic performance were to reemerge.\n\nMany of these members commented that the considerations underlying the desirability of a technical adjustment to the ranges for this year applied to 1994 as well, and they therefore supported extending the reduced ranges to 1994 on a tentative basis subject to review early next year. Monetary growth outcomes somewhat higher within these ranges might be anticipated in association with the somewhat faster economic growth and essentially unchanged rate of inflation that most members had forecast for next year.\n\nSeveral members indicated that while they could accept reductions in the 1993 ranges, they nonetheless preferred to retain the existing ranges. One reason given for this preference was that the prospective performance of the broad monetary aggregates in relation to developments in the economy was not sufficiently understood to warrant the specification of new ranges. Indeed, a change might be misinterpreted as implying more knowledge about velocity relationships than the Committee in fact possessed and could set up expectations that the Committee would put greater and, depending on emerging circumstances, perhaps undesirable emphasis on achieving monetary growth within the new ranges. Moreover, to the extent that some observers interpreted the ranges as the Committee's proxies for presumed nominal GDP objectives, an erroneous conclusion could be reached that the Committee had decided on a reduced target level of nominal GDP even though the Committee had not in fact framed its objectives in terms of GDP targets. On balance, while these members did not view this choice as a matter of great consequence in current circumstances, they concluded that it was marginally preferable to retain the ranges for this year, and if necessary, to accept and explain the reasons for a shortfall once the latter were more clearly established. The members who preferred to retain the current ranges agreed that there were plausible arguments on both sides of this issue and they could accept a proposal to reduce the ranges for both 1993 and 1994.\n\nIn light of the limited reliance that the members felt they could place on the behavior of the current monetary aggregates, the Committee at this meeting reviewed the possible advantages of a newly constructed measure of money. This measure involved the addition of bond and stock mutual funds to M2 as currently defined. There were indications that the shares of such funds had become closer substitutes for M2, and large portfolio shifts into such funds seemed to account for much of the weakness in M2 and its uncertain relationship to income and the longer-run behavior of prices. After examining the properties of this measure and reviewing its past behavior in relation to key indicators of economic performance, the members concluded that it would not enhance the formulation or implementation of monetary policy, at least at this point. However, the members agreed that mutual funds flows should continue to be monitored for their effects on M2 and that the relevant data should be made available to outside analysts.\n\nAt the conclusion of its discussion, the Committee voted to lower the M2 range that it had established in February by an additional one percentage point and to reduce the M3 range by another 1/2 percentage point, bringing the M2 range to 1 to 5 percent and that for M3 to 0 to 4 percent for 1993. The Committee also voted to reduce the annual monitoring range for growth of total domestic nonfinancial debt by 1/2 percentage point to 4 to 8 percent. The members anticipated that this debt aggregate would continue to grow at a rate that was roughly in line with that of nominal GDP. The Committee approved the following statement for inclusion in its domestic policy directive.\n\nThe Federal Open Market Committee seeks monetary and financial conditions that will foster price stability and promote sustainable growth in output. In furtherance of these objectives, the Committee at this meeting lowered the ranges it had established in February for growth of M2 and M3 to ranges of 1 to 5 percent and 0 to 4 percent respectively, measured from the fourth quarter of 1992 to the fourth quarter of 1993. The Committee anticipated that developments contributing to unusual velocity increases would persist over the balance of the year and that money growth within these lower ranges would be consistent with its broad policy objectives. The monitoring range for growth of total domestic nonfinancial debt also was lowered to 4 to 8 percent for the year.\n\nVotes for this action: Messrs. Greenspan, Mullins, Angell, Boehne, Keehn, Kelley, LaWare, Lindsey, McTeer, Oltman, Ms. Phillips, and Mr. Stern.\n\nVotes against this action: None.\n\nAbsent: Mr. Corrigan. (Mr. Oltman voted as alternate for Mr. Corrigan.)\n\nFor the year 1994, the Committee approved provisional ranges that were unchanged from the reduced levels for 1993. Accordingly, the Committee voted to incorporate the following statement regarding the 1994 ranges in its domestic policy directive.\n\nFor 1994, the Committee agreed on tentative ranges for monetary growth, measured from the fourth quarter of 1993 to the fourth quarter of 1994, of 1 to 5 percent for M2 and 0 to 4 percent for M3. The Committee provisionally set the monitoring range for growth of total domestic nonfinancial debt at 4 to 8 percent for 1994. The behavior of the monetary aggregates will continue to be evaluated in the light of progress toward price level stability, movements in their velocities, and developments in the economy and financial markets.\n\nVotes for this action: Messrs. Greenspan, Mullins. Angell, Boehne, Keehn, Kelley, LaWare, Lindsey, McTeer, Oltman, Ms. Phillips. and Mr. Stern.\n\nVotes against this action: None.\n\nAbsent: Mr. Corrigan. (Mr. Oltman voted as alternate for Mr. Corrigan.)\n\nIn the Committee's discussion of policy for the period until the next meeting, most of the members indicated that they saw little or no reason to change monetary policy in either direction. The most recent information on the performance of the economy was mixed, and this together with questions about the course of fiscal policy contributed to considerable uncertainty about the outlook. Even so, the members felt that the evidence pointed on the whole to a sustained rate of economic expansion. The latest price statistics provided some encouragement that the apparent intensification of inflation in earlier months of the year might have abated. For now, therefore, nearly all the members saw the balance of factors bearing on the course of economic activity and the outlook for inflation as calling for an unchanged degree of pressure on reserve positions.\n\nAccording to a staff analysis prepared for this meeting, the growth of M2 could be expected to slow markedly in the months ahead from its pace over the second quarter. The projected deceleration was mainly associated with some unwinding of the second-quarter bulge in mortgage refinancings along with further heavy inflows to bond and stock mutual funds. The expansion of M3 appeared likely to be held down by weaker bank credit extensions as alternative sources of funds in the capital markets attracted more borrowers. On balance, modest growth of both M2 and M3 would keep them close to the lower ends of their downward-revised ranges through September.\n\nSome members cautioned that despite the very sluggish behavior of the broad measures of money thus far this year, monetary policy was relatively expansive as evidenced by a variety of other indicators including the growth in narrow measures of money and reserves and the very low levels of money market interest rates. Indeed, in the view of several members, in a period characterized by indications of some worsening in inflationary expectations, a policy course that maintained steady conditions in reserve markets could be said to have become more accommodative as the federal funds rate, in real terms after adjustment for expected inflation, moved down from an already low level. Accordingly, while current monetary policy seemed likely to support further economic-expansion, the Committee needed to remain alert to the potential for intensifying inflation. At some point the current policy stance could well begin to foster greater price pressures. One member urged a prompt move toward restraint, given the prospect in his view that further progress toward price stability was unlikely with the current, quite stimulative, stance of monetary policy.\n\nA majority of the members, taking account of the current stance of monetary policy, favored a proposal to retain the bias toward possible tightening that the Committee had adopted at the May meeting. In this connection, some commented that while the need for any policy adjustment during the period ahead seemed somewhat remote, the next policy move was more likely to be in the direction of some firming than toward easing. Other members suggested that a symmetrical directive might be more consistent with current economic conditions and the related outlook for a steady policy course over the near term. These members agreed, however, that a return to symmetry so soon after the adoption of a directive that was biased toward restraint could convey a misleading impression that recent developments had increased the Committee's concerns about the sustainability of the expansion or that the Committee had become less committed to a disinflationary policy course. Accordingly, these members indicated that they could support an asymmetric directive at this point. Several members observed that a number of key economic measures were scheduled for release during the intermeeting period and that the data in question should provide a firmer basis for evaluating the performance of the economy and a desirable course for monetary policy.\n\nAt the conclusion of the Committee's discussion, all but one of the members indicated that they preferred or found acceptable a directive that called for maintaining the existing degree of pressure on reserve positions and that retained a bias toward possible firming of reserve conditions during the intermeeting period. Accordingly, in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, the Committee decided that slightly greater reserve restraint would be acceptable or slightly lesser reserve restraint might be acceptable during the intermeeting period. The reserve conditions contemplated at this meeting were expected to be consistent with modest growth in the broader monetary aggregates over the third quarter.\n\nAt the conclusion of the meeting, the Federal Reserve Bank of New York was authorized and directed, until instructed otherwise by the Committee, to execute transactions in the System account in accordance with the following domestic policy directive:\n\nThe information reviewed at this meeting suggests that the economic expansion has picked up somewhat in recent months from the very slow pace of the first quarter. Total nonfarm payroll employment changed little in June after registering substantial gains in April and May, and the civilian unemployment rate edged up to 7.0 percent in June. Industrial production has changed little on balance over the last few months. Real consumer expenditures edged higher in May after a sizable rise in April but have increased only slightly thus far this year. Housing starts turned up in April from a depressed first-quarter pace and rose somewhat further in May. Incoming data suggest a continued brisk advance in outlays for business equipment, while nonresidential construction has remained soft. The nominal U.S. merchandise trade deficit was about unchanged in April but substantially larger than its average rate in the first quarter. Consumer and producer prices were about unchanged in May, but for the year to date inflation has been more rapid than in the second half of 1992.\n\nShort-term interest rates have changed little since the Committee meeting on May 18 while bond yields have declined somewhat. In foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies increased on balance over the intermeeting period.\n\nAfter contracting during the first quarter, M2 and M3 expanded appreciably over the second quarter. For the year through June, growth of the two aggregates was below the lower ends of the ranges established by the Committee for 1993. Total domestic nonfinancial debt expanded somewhat further through April.\n\nThe Federal Open Market Committee seeks monetary and financial conditions that will foster price stability and promote sustainable growth in output. In furtherance of these objectives, the Committee at this meeting lowered the ranges it had established in February for growth of M2 and M3 to ranges of 1 to 5 percent and 0 to 4 percent respectively, measured from the fourth quarter of 1992 to the fourth quarter of 1993. The Committee anticipated that developments contributing to unusual velocity increases would persist over the balance of the year and that money growth within these lower ranges would be consistent with its broad policy objectives. The monitoring range for growth of total domestic nonfinancial debt also was lowered to 4 to 8 percent for the year. For 1994, the Committee agreed on tentative ranges for monetary growth, measured from the fourth quarter of 1993 to the fourth quarter of 1994, of 1 to 5 percent for M2 and 0 to 4 percent for M3. The Committee provisionally set the monitoring range for growth of total domestic nonfinancial debt at 4 to 8 percent for 1994. The behavior of the monetary aggregates will continue to be evaluated in the light of progress toward price level stability, movements in their velocities, and developments in the economy and financial markets.\n\nIn the implementation of policy for the immediate future, the Committee seeks to maintain the existing degree of pressure on reserve positions. In the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint would or slightly lesser reserve restraint might be acceptable in the intermeeting period. The contemplated reserve conditions are expected to be consistent with modest growth in the broader monetary aggregates over the third quarter.\n\nVotes for this action: Messrs. Greenspan, Mullins. Boehne, Keehn, Kelley, LaWare, Lindsey. McTeer, Oltman, Ms. Phillips, and Mr. Stern.\n\nVote against this action: Mr. Angell.\n\nAbsent: Mr. Corrigan. (Mr. Oltman voted as alternate for Mr. Corrigan.)\n\nMr. Angell dissented because he favored a prompt move to tighten policy. In his view, monetary policy was overly expansive at this point as evidenced by what he viewed as excessive liquidity in financial markets, the negative level of real short-term interest rates, and the disappointing lack of progress toward lower inflation this year. Given indications of worsening inflationary expectations, such as the substantial rise in the price of gold, as well as projections of an increase in inflation, a policy that led to a steady federal funds rate in fact implied a further easing of an already stimulative monetary policy. In these circumstances, a tightening of policy would not involve any significant risk to the expansion but would foster changes in financial conditions and the outlook for inflation that would be more consistent with renewed progress toward price stability in 1994 and later. Declining inflation around the world and a stronger trend of productivity growth in the United States, among other factors, were providing a favorable environment for further disinflation, but those developments needed to be supported and validated by appropriate monetary policy action.\n\nAt this meeting, the Committee reviewed its practices with regard to the release of information to the public. This review was undertaken in response to media reports of the purported results of the May meeting before the Committee had made public any information about that meeting. In its discussion, the Committee reaffirmed its long-standing rules governing the confidentiality of FOMC information, including the schedule that calls for releasing the minutes of a Committee meeting, along with an explanation of the Committee's decisions, a few days after the next meeting. These rules are designed to safeguard the Committee's flexibility to make needed adjustments to policy and also to provide adequate time to prepare a full report of the context and rationale for its decisions. Committee members emphasized the potential for inadvertent leaks of information in the course of general conversations with representatives of the news media or others concerning the members' views about economic developments or monetary policy. The members agreed that particular care needed to be taken for some period before and after each of its meetings.\n\nIt was agreed that the next meeting of the Committee would be held on Tuesday, August 17, 1993.\n\nThe meeting adjourned at 12:25 p.m. on Wednesday, July 7, 1993.\n\nDonald L. Kohn Secretary\n\nFOMC",
            "content_type": "fomc_minutes",
        },
        {
            "date": "1993-08-17",
            "speaker": "Alan Greenspan",
            "title": "FOMC Meeting Minutes",
            "text": "A meeting of the Federal Open Market Committee was held in the offices of the Board of Governors of the Federal Reserve System in Washington, D.C., on Tuesday, August 17, 1993, at 9:00 a.m.\n\nPRESENT:\n\nMr. Greenspan, Chairman Mr. McDonough, Vice Chairman Mr. Angell Mr. Boehne Mr. Keehn Mr. Kelley Mr. LaWare Mr. Lindsey Mr. McTeer Mr. Mullins Ms. Phillips Mr. Stern\n\nMessrs. Broaddus, Jordan, Forrestal, and Parry, Alternate Members of the Federal Open Market Committee\n\nMessrs. Hoenig, Melzer, and Syron, Presidents of the Federal Reserve Banks of Kansas City, St. Louis, and Boston, respectively\n\nMr. Kohn, Secretary and Economist Mr. Bernard, Deputy Secretary Mr. Coyne, Assistant Secretary Mr. Gillum, Assistant Secretary Mr. Mattingly, General Counsel Mr. Patrikis, Deputy General Counsel Mr. Prell, Economist\n\nMessrs. R. Davis, Promisel, Rosenblum, Scheld, Siegman, Simpson, and Slifman, Associate Economists\n\nMs. Greene, Deputy Manager for Foreign Operations\n\nMs. Lovett, Deputy Manager for Domestic Operations\n\nMr. Ettin, Deputy Director. Division of Research and Statistics, Board of Governors\n\nMr. Madigan, Associate Director, Division of Monetary Affairs. Board of Governors\n\nMr. Stockton. Associate Director, Division of Research and Statistics, Board of Governors\n\nMs. Johnson, Assistant Director, Division of International Finance. Board of Governors\n\nMs. Low, Open Market Secretariat Assistant. Division of Monetary Affairs, Board of Governors\n\nMessrs. Beebe, J. Davis, T. Davis, Dewald, Goodfriend, and Ms. Tschinkel, Senior Vice Presidents, Federal Reserve Banks of San Francisco, Cleveland, Kansas City, St. Louis, Richmond, and Atlanta, respectively\n\nMessrs. McNees, Meyer, and Miller, Vice Presidents. Federal Reserve Banks of Boston, Philadelphia, and Minneapolis, respectively\n\nMs. Meulendyke, Manager, Open Market Operations. Federal Reserve Bank of New York.\n\nBy unanimous vote, the minutes for the meeting of the Federal Open Market Committee held on July 6-7, 1993, were approved.\n\nSecretary's Note: Advice had been received of the election of William J. McDonough by the Board of Directors of the Federal Reserve Bank of New York as a member of the Federal Open Market Committee for the period commencing July 19, 1993, and ending December 31, 1993, and that he had executed his oath of office.\n\nBy unanimous vote, the Committee elected William J. McDonough as Vice Chairman of the Committee to serve until the first meeting of the Committee after December 31, 1993.\n\nThe Deputy Manager for Foreign Operations reported on developments in foreign exchange markets during the period since the July meeting. There were no System open market transactions in foreign currencies during this period, and thus no vote was required of the Committee.\n\nThe Deputy Manager for Domestic Operations reported on developments in domestic financial markets and on System open market transactions in government securities and federal agency obligations during the period July 7, 1993, through August 16, 1993. By unanimous vote, the Committee ratified these transactions.\n\nThe Committee then turned to a discussion of the economic and financial outlook and the implementation of monetary policy over the intermeeting period ahead. A summary of the economic and financial information available at the time of the meeting and of the Committee's discussion is provided below, followed by the domestic policy directive that was approved by the Committee and issued to the Federal Reserve Bank of New York.\n\nThe information reviewed by the Committee suggested that economic activity was expanding at a moderate pace. The limited data available for the third quarter presented a mixed picture but on balance pointed to continued expansion in consumption, business fixed investment, and homebuilding. Employment remained on an uptrend, and industrial production recently had firmed somewhat. After rising at a faster rate in the early part of the year, consumer prices had changed little and producer prices had fallen in recent months.\n\nTotal nonfarm payroll employment, after a small gain in June, expanded in July at a rate close to its average advance in earlier months of the year. The services industries, led by business services, provided half of the July increase. Elsewhere, considerable hiring was evident in wholesale and retail trade, and construction employment moved up after a small decline in June. In manufacturing, more jobs were lost, although at a slower rate than earlier in the year. The civilian unemployment rate dropped to 6.8 percent in July.\n\nIndustrial production recovered in July from small declines in May and June. Manufacturing output rose in spite of a sizable cutback in motor vehicle assemblies: utility production registered a strong weather-related gain; and mining output declined further. Within manufacturing, the production of consumer durable goods other than automobiles and trucks rebounded in July, and the output of business equipment advanced further. Total utilization of industrial capacity edged higher in July, reflecting a substantial gain at electric utilities: utilization of manufacturing capacity was unchanged.\n\nRetail sales increased slightly further in July after a sizable rise in the second quarter. Spending on automobiles was down for a second straight month, but sales were strong at apparel, furniture and appliance, and general merchandise stores. Total housing starts, depressed by wet weather and floods in some areas of the country, were down somewhat in July; however, permit issuance moved up, suggesting that homebuilding activity remained in a mild uptrend. In addition, consumer surveys indicated that attitudes toward homebuying continued to be strongly positive during July, and builders' assessments of home sales improved substantially.\n\nBusiness fixed investment increased in the second quarter at about the rapid pace of the first quarter. Business spending for equipment remained strong, with solid increases in business purchases of motor vehicles, computers, and a wide range of machinery and equipment. However, outlays for aircraft declined in the second quarter, retracing some of the substantial first-quarter rise. The limited information available for the third quarter pointed to some slowing of the growth of business spending for equipment. In the second quarter, nonresidential building activity posted its largest advance in three years. Expenditures were up across a broad array of categories, with investment in institutional and public utilities structures being particularly strong.\n\nBusiness inventories expanded moderately during the second quarter, and inventory accumulation was broadly in line with sales over the first half of the year. In manufacturing, stocks edged lower in June, reflecting a further decline in inventories held by aircraft producers. Outside of the aircraft industry, inventory changes were mixed. For manufacturing as a whole, the ratio of inventories to shipments fell in June to one of the lowest levels in recent years. In the wholesale trade sector, inventories expanded modestly in June, and with sales lower, the inventory-to-sales ratio for the sector increased slightly. Retail inventories, after changing little in May, rose slightly more than sales in June, and the stocks-to-sales ratio for the retail sector remained near the high end of its range for the past several years.\n\nThe nominal U.S. merchandise trade deficit was considerably smaller in May than the deficits recorded in March and April: however, the deficit for April and May combined was larger than the average rate for the first quarter. The value of exports rose slightly in May; increases in sales abroad of industrial supplies, machinery, and consumer goods offset declines in agricultural products, civilian aircraft, and motor vehicles and parts. A drop in the value of imports was spread across a wide range of products, particularly automotive products, consumer goods, and oil. The economic performance of the major foreign industrial countries was mixed in the second quarter. Output continued to decline in western Germany, and economic activity in Japan appeared to have stalled after modest growth in the first quarter. In contrast, economic recovery continued in Canada and the United Kingdom.\n\nProducer prices of finished goods declined in July for a second consecutive month. Prices of finished foods edged lower, and prices of finished energy goods, particularly gasoline and fuel oil fell significantly; excluding the food and energy components, producer prices edged up in July and to that point in the year had risen at a slightly lower rate than was recorded in 1992. At the consumer level, prices for nonfood, non-energy items were up slightly in both June and July and for the year to date had increased a little more slowly than last year. Hourly compensation for private industry workers rose in the second quarter at about the rate seen last year. Average hourly earnings of production or nonsupervisory workers were unchanged on balance over June and July, but for the year through July these earnings had increased at the same pace as in 1992.\n\nAt its meeting on July 6-7, 1993, the Committee adopted a directive that called for maintaining the existing degree of pressure on reserve positions and that retained a tilt toward possible firming of reserve conditions during the intermeeting period. Accordingly, the directive indicated that in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint would be acceptable or slightly lesser reserve restraint might be acceptable during the intermeeting period. The reserve conditions associated with this directive were expected to be consistent with modest growth of the broader monetary aggregates over the third quarter.\n\nThroughout the intermeeting period, open market operations were directed toward maintaining the existing degree of pressure on reserve positions. Two upward revisions were made to expected levels of adjustment plus seasonal borrowing in anticipation of further increases in demand for seasonal credit. Borrowing averaged close to expected levels over most of the intermeeting interval, and the federal funds rate remained close to 3 percent.\n\nMoney market interest rates were little changed on balance over the intermeeting period, while rates on intermediate-term U.S. Treasury obligations and on fixed-rate mortgages dropped slightly. Yields on long-term Treasury and corporate bonds were down by more, with the rate on the 30-year Treasury bond falling below 6-1/2 percent. Many market interest rates moved higher after Chairman Greenspan's congressional testimony on July 20, which was perceived by market participants as suggesting a greater likelihood of some tightening of monetary policy in the future. Subsequently, interest rates generally retreated in reaction to incoming economic data indicating subdued inflation pressures and to the passage of the deficit-reduction legislation. Major indexes of stock prices increased somewhat over the intermeeting period.\n\nIn foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies was about unchanged on balance over the intermeeting period. The dollar strengthened slightly against the German mark, but it rose by significantly more against most other European currencies in the Exchange Rate Mechanism in the aftermath of a widening of the margins within which participating currencies are allowed to fluctuate relative to each other. The widening, which was in response to massive selling pressures on the French franc and several other currencies, followed sharp increases in short-term interest rates in the affected countries. With exchange market participants continuing to focus on Japan's trade surplus, the dollar fell substantially against the yen.\n\nM2 expanded only slightly in July after growing appreciably over the second quarter. The continued strength of inflows to bond and stock mutual funds suggested that households were still realigning their portfolios toward assets outside the monetary aggregates. Through July, M2 was estimated to have grown at a rate close to the lower end of the Committee's range for the year. M3 contracted slightly in June and July, owing in part to a substantial drop in institution-only money market mutual funds, whose returns had not kept pace with the increase in money market rates in late spring. In addition, depository institutions placed greater reliance on various nondeposit sources of funds, including the issuance of equity and subordinated debt. Through July, M3 had declined a little and was slightly below its annual range. Total domestic nonfinancial debt had expanded at a moderate rate in recent months, and for the year through June was estimated to have increased at a rate in the lower half of the Committee's monitoring range.\n\nThe staff projection prepared for this meeting suggested moderate growth in economic activity and modest reductions in margins of unemployed labor and capital through next year. The fiscal restraint stemming from the recent legislation and uncertainty about other government policies would act as a drag on the economy. On the other hand, lower interest rates were expected to contribute to further gains in spending on consumer durables, housing, and business fixed investment. Continued expansion also would be supported by further improvements in the availability of credit, a small boost to production over the next several quarters associated with rebuilding activity in areas of the Midwest affected by the recent floods, and a pickup in foreign demand resulting from some strengthening in economic activity abroad. The projected slack in labor and product markets, coupled with some tempering of inflation expectations, was expected to foster modest further reductions in wage and price inflation.\n\nIn the Committee's discussion of prospective economic conditions, members commented that recent developments had not materially altered the outlook for moderate and sustained growth in economic activity. Despite widespread indications of pessimistic consumer and business attitudes, overall consumer spending and business investment appeared to be reasonably well maintained. Likewise, the outlook for increased fiscal restraint associated with the recently enacted deficit-reduction legislation needed to be weighed against the favorable effects on spending of reduced interest rates in intermediate- and long-term debt markets, the improved balance sheets of consumers and businesses, and the indications of a somewhat better availability of loans from financial intermediaries. In an environment of moderate economic growth, the fundamentals bearing on the outlook for inflation were consistent with further disinflation, and the members drew some encouragement from consumer and producer price developments in recent months. Several cautioned, however, that recent price measures probably overstated the reduction in inflation, just as the surge in prices earlier in the year seemed to have overstated the underlying inflation trend. Members also referred to the persistence of inflationary expectations among business executives and consumers. Thus, while the rise in inflation appeared to have been arrested, any further progress toward price stability was likely to be limited over the quarters ahead.\n\nBusiness contacts and other sources of information suggested little change since the July meeting in the pace or composition of economic activity in different parts of the country. Descriptions of economic performance varied from slow to moderate growth in most regions, though business activity probably continued to weaken in some major areas such as California. Despite sustained, if not ebullient, growth in sales to consumers and the relative strength in business investment spending in the first half of this year, business sentiment was widely described as cautious or negative even in some regions whose economies were outperforming the nation as a whole. According to business contacts, the recent enactment of deficit-reduction legislation had tended to mitigate concerns about the size of future federal deficits, but business executives were now focusing on the implications of higher taxes and many were expressing apprehension about further though still unannounced tax increases that might be associated with health care reform. Business sentiment and sales also were being affected adversely in many areas by cutbacks in defense contracts and closings of military installations and by the weakness in foreign demand for some products.\n\nWith regard to developments and prospects in key sectors of the economy, members noted that despite further survey indications of eroding consumer confidence, consumer expenditures had strengthened in recent months after a pause earlier in the year. The pickup had featured rising sales of motor vehicles, and while the latter had slipped recently, a number of special factors such as shortages of popular models at the end of the model year and the effects of flooding in some parts of the Midwest suggested the need to withhold judgment on any downward shift in the underlying demand for motor vehicles. Tourism was reported to have strengthened considerably in many areas this summer, though there were major exceptions. As had been true for an extended period, consumer attitudes continued to be inhibited by concerns about employment opportunities, especially given further reductions in defense spending, the ongoing restructuring and related downsizing of many business operations, and the continuing efforts by business firms to limit the number of their permanent employees in order to hold down the rising costs of health care and other nonwage worker benefits. Members noted, however, that the growth in employment thus far this year, while tending to involve many low paying jobs, had greatly exceeded the rate of expansion in 1992. In the view of at least some members, appreciable further growth was likely as business firms found it increasingly difficult in an expanding economy to meet growing demands through outsourcing, temporary workers, and overtime work. Some members also noted that the newly legislated taxes on higher incomes would tend to curtail some consumer spending. The timing of that effect was uncertain; tax liabilities had already risen, but some payments on the added tax liabilities were not due until April of 1994 and 1995.\n\nMembers anticipated that building activity, notably housing construction, would provide some stimulus to the expansion. Although indicators of housing activity were somewhat mixed for the nation as a whole, sales of new and existing homes were brisk in many regions and even sales of second homes were reported to be improving in some areas. Prospective homebuyers continued to exercise considerable caution, but reductions in mortgage rates and generally improved affordability pointed to rising housing sales and construction over the quarters ahead. In the nonresidential sector, there was growing evidence of some strengthening in the construction of commercial and institutional structures, but overcapacity was likely to depress the construction of new office buildings for an extended period in most parts of the country. In some areas, infrastructure and other rebuilding associated with the recent floods was likely to stimulate some construction activity later this year.\n\nWith regard to the external sector of the economy, the members again noted a somewhat mixed picture. Exporters from some parts of the country continued to report relatively brisk sales abroad, but many domestic producers were expressing concerns about weak markets in key foreign nations. Against the background of more stimulative economic policies in a number of those countries, some overall strengthening in the major foreign economies was viewed as a reasonable expectation, but the overall growth in exports was likely to lag the anticipated expansion in imports over the projection horizon. The North American Free Trade Agreement now under consideration in the Congress was a topic of active discussion among business contacts, and the uncertain outcome of that treaty was a matter of concern in several parts of the country.\n\nMembers observed that the more favorable performance of key measures of prices in recent months had tended to relieve earlier concerns about a possible worsening of inflation. However, because the recent price indexes probably overstated the improvement in the trend rate of inflation, it was too early to determine whether they pointed to renewed disinflation. In any event, a number of fundamental factors appeared to have favorable implications for the inflation outlook, notably the prospect that some slack in labor and capital resources would persist in the context of projections that pointed to a relatively moderate rate of economic expansion. Members continued to cite reports from numerous business firms regarding their inability to raise prices because of the highly competitive markets in which those firms had to operate. Many business contacts also referred to the absence of significant increases--and indeed to occasional decreases--in the costs of their outside purchases. Oil price developments in world markets and the ongoing competition from foreign producers also were noted as favorable elements in the outlook for inflation. On the negative side, adverse weather conditions in recent months including severe floods in the Midwest appeared to have fostered some upward pressure on food prices, and higher taxes would raise gasoline prices in the fourth quarter. Perhaps of greater significance, business contacts and surveys of households indicated persisting expectations that inflation would rise at some point. In this connection, however, passage of the federal deficit-reduction legislation and the Committee's reaffirmation in its directive and in congressional testimony of its commitment to price stability seemed to have had a constructive effect on attitudes in financial markets and on long-term interest rates, and these developments could prove to be harbingers of more favorable inflation attitudes more generally.\n\nIn the Committee's discussion of policy for the intermeeting period ahead, the members agreed that recent developments pointed to the desirability of a steady policy course. While economic growth did not seem particularly robust, neither was it clear that a disinflationary trend had been reestablished. Many members observed that real short-term interest rates were at very low levels, indeed slightly negative by some calculations, and while real intermediate- and long-term interest rates were higher, it was apparent that monetary policy was in an accommodative posture. This conclusion was seen as reinforcing the view that monetary policy probably would have to move in the direction of restraint at some point to resist any incipient tendency for inflationary pressures to intensify. For now, the relatively slow economic expansion in the first half of the year, the fiscal restraint associated with the deficit-reduction legislation, other obstacles to economic growth, and the encouraging inflation statistics for recent months argued against any near-term policy adjustment.\n\nMoreover, there was no compelling evidence that current monetary policy was fostering credit flows usually associated with speculative excesses or impending increases in price pressures. Growth in the broad measures of money and in the debt of nonfinancial sectors remained fairly damped despite indications of greater willingness to supply credit by banks, other financial intermediaries, and investors in securities markets. With regard to the monetary aggregates, low short-term interest rates undoubtedly were contributing to large shifts of funds from depository institutions, notably from components of M2 and M3 to stock and bond mutual funds and to other financial instruments, and thus to the sluggish behavior of the broad measures of money. In this connection, a staff analysis pointed to continuing slow growth in M2 over the near term and, on the assumption of little or no change in the degree of pressure on reserve positions, to growth for the year at a rate around the lower end of the Committee's range. Growth in M3 was likely to fall marginally below the Committee's range for the year. On the other hand, growth in Ml and in various reserve measures was expected to remain relatively robust.\n\nTurning to possible adjustments to policy during the intermeeting period ahead, the members endorsed a proposal to return to an unbiased intermeeting instruction that did not incorporate any presumption with regard to the direction of possible intermeeting policy changes. The members agreed that the probability of an intermeeting policy adjustment was relatively remote. Incoming data on economic activity and prices had reduced concerns that inflation and inflationary expectations might be worsening. The Committee retained its fundamental objectives of fostering economic expansion at a sustainable pace that was consistent with further progress over time toward stable prices. However, it now appeared less likely than at the time of the May and July meetings that the Committee needed to bias its consideration of responses to incoming information in the intermeeting period toward possible tightening in order to achieve those objectives. One member, while agreeing that a tightening move would not be appropriate under current circumstances, nonetheless believed that monetary policy had been overly stimulative for some time and that the Committee should move toward restraint at the first favorable opportunity.\n\nAt the conclusion of the Committee's discussion, all the members expressed a preference for a directive that called for maintaining the existing degree of pressure on reserve positions. They also indicated their support of a directive that did not include a presumption about the likely direction of any adjustment to policy during the intermeeting period. Accordingly, in the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, the Committee decided that slightly greater or slightly lesser reserve restraint might be acceptable during the intermeeting period. The reserve conditions contemplated at this meeting were expected to be consistent with modest growth in M2 and little net change in M3 over the balance of the third quarter.\n\nAt the conclusion of the meeting, the Federal Reserve Bank of New York was authorized and directed, until instructed otherwise by the Committee, to execute transactions in the System account in accordance with the following domestic policy directive:\n\nThe information reviewed at this meeting suggests that economic activity is expanding at a moderate pace. Total nonfarm payroll employment increased in July at a rate close to its average advance in earlier months of the year, and the civilian unemployment rate declined to 6.8 percent. Industrial production turned up in July after posting small declines in May and June. Retail sales edged higher in July following a sizable rise in the second quarter. Housing starts were down somewhat in July, but permits moved up. Available indicators point to continued expansion in business capital spending. The nominal U.S. merchandise trade deficit declined in May, but for April and May combined it was larger than its average rate in the first quarter. After rising at a faster rate in the early part of the year, consumer prices have changed little and producer prices have fallen in recent months.\n\nShort-and intermediate-term interest rates have changed little since the Committee meeting on July 6-7, while yields on long-term Treasury and corporate bonds have declined somewhat. In foreign exchange markets, the trade-weighted value of the dollar in terms of the other G-10 currencies was about unchanged on balance over the intermeeting period.\n\nAfter expanding appreciably over the second quarter, M2 increased slightly further in July and M3 declined. For the year through July, M2 is estimated to have grown at a rate close to the lower end of the Committee's range for the year, and M3 at a rate slightly below its range. Total domestic nonfinancial debt has expanded at a moderate rate in recent months, and for the year through June it is estimated to have increased at a rate in the lower half of the Committee's monitoring range.\n\nThe Federal Open Market Committee seeks monetary and financial conditions that will foster price stability and promote sustainable growth in output. In furtherance of these objectives, the Committee at its meeting in July lowered the ranges it had established in February for growth of M2 and M3 to ranges of 1 to 5 percent and 0 to 4 percent respectively, measured from the fourth quarter of 1992 to the fourth quarter of 1993. The Committee anticipated that developments contributing to unusual velocity increases would persist over the balance of the year and that money growth within these lower ranges would be consistent with its broad policy objectives. The monitoring range for growth of total domestic nonfinancial debt also was lowered to 4 to 8 percent for the year. For 1994, the Committee agreed on tentative ranges for monetary growth, measured from the fourth quarter of 1993 to the fourth quarter of 1994, of 1 to 5 percent for M2 and 0 to 4 percent for M3. The Committee provisionally set the monitoring range for growth of total domestic nonfinancial debt at 4 to 8 percent for 1994. The behavior of the monetary aggregates will continue to be evaluated in the light of progress toward price level stability, movements in their velocities, and developments in the economy and financial markets.\n\nIn the implementation of policy for the immediate future, the Committee seeks to maintain the existing degree of pressure on reserve positions. In the context of the Committee's long-run objectives for price stability and sustainable economic growth, and giving careful consideration to economic, financial, and monetary developments, slightly greater reserve restraint or slightly lesser reserve restraint might be acceptable in the intermeeting period. The contemplated reserve conditions are expected to be consistent with modest growth in M2 and little net change in M3 over the balance of the third quarter.\n\nVotes for this action: Messrs. Greenspan, McDonough, Angell, Boehne, Keehn, Kelley, LaWare, Lindsey, McTeer, Mullins, Ms. Phillips, and Mr. Stern.\n\nVotes against this action: None.\n\nFOMC",
            "content_type": "fomc_minutes",
        },
    ],
}
