"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PostResolveToken = exports.ignoreEmpty = exports.capitalizePropertyNames = exports.findLastCommonElement = exports.pathToTopLevelStack = exports.filterUndefined = exports.isAllowedValue = exports.ifUndefined = void 0;
const intrinsic_1 = require("./private/intrinsic");
const stack_1 = require("./stack");
function ifUndefined(value, defaultValue) {
    return value !== undefined ? value : defaultValue;
}
exports.ifUndefined = ifUndefined;
function isAllowedValue(props, name, context, allowedValues) {
    const value = props[name];
    let res = false;
    // if the value is undefined, there is no need to check
    if (value === undefined)
        return;
    allowedValues.forEach(function checkValue(x) {
        if (x === value)
            res = true;
    });
    if (!res) {
        throw new Error(`The property ${name} (value: ${value}) of ${context.toString()} is not allowed.\nIts value should be in ${allowedValues}`);
    }
}
exports.isAllowedValue = isAllowedValue;
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter((x) => x != null).map((x) => filterUndefined(x));
    }
    if (typeof obj === "object") {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value == null) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
exports.filterUndefined = filterUndefined;
/**
 * @returns the list of stacks that lead from the top-level stack (non-nested) all the way to a nested stack.
 */
function pathToTopLevelStack(s) {
    if (s.nestedStackParent) {
        return [...pathToTopLevelStack(s.nestedStackParent), s];
    }
    else {
        return [s];
    }
}
exports.pathToTopLevelStack = pathToTopLevelStack;
/**
 * Given two arrays, returns the last common element or `undefined` if there
 * isn't (arrays are foriegn).
 */
function findLastCommonElement(path1, path2) {
    let i = 0;
    while (i < path1.length && i < path2.length) {
        if (path1[i] !== path2[i]) {
            break;
        }
        i++;
    }
    return path1[i - 1];
}
exports.findLastCommonElement = findLastCommonElement;
function capitalizePropertyNames(construct, obj) {
    const stack = stack_1.Stack.of(construct);
    obj = stack.resolve(obj);
    if (typeof obj !== "object") {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map((x) => capitalizePropertyNames(construct, x));
    }
    const newObj = {};
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        const first = key.charAt(0).toUpperCase();
        const newKey = first + key.slice(1);
        newObj[newKey] = capitalizePropertyNames(construct, value);
    }
    return newObj;
}
exports.capitalizePropertyNames = capitalizePropertyNames;
/**
 * Turns empty arrays/objects to undefined (after evaluating tokens).
 */
function ignoreEmpty(obj) {
    return new PostResolveToken(obj, (o) => {
        // undefined/null
        if (o == null) {
            return o;
        }
        if (Array.isArray(o) && o.length === 0) {
            return undefined;
        }
        if (typeof o === "object" && Object.keys(o).length === 0) {
            return undefined;
        }
        return o;
    });
}
exports.ignoreEmpty = ignoreEmpty;
class PostResolveToken extends intrinsic_1.Intrinsic {
    constructor(value, processor) {
        super(value);
        this.processor = processor;
    }
    resolve(context) {
        context.registerPostProcessor(this);
        return super.resolve(context);
    }
    postProcess(o, _context) {
        return this.processor(o);
    }
}
exports.PostResolveToken = PostResolveToken;
//# sourceMappingURL=data:application/json;base64,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