"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.withResolved = exports.isResolvableObject = exports.Tokenization = exports.Token = void 0;
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
const lazy_1 = require("./lazy");
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    /**
     * Returns true if obj represents an unresolved value
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     */
    static isUnresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    static asString(value, options = {}) {
        if (typeof value === "string") {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token
     */
    static asNumber(value) {
        if (typeof value === "number") {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every((x) => typeof x === "string")) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a resolvable representation of the given value
     */
    static asAny(value) {
        return isResolvableObject(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    constructor() { }
}
exports.Token = Token;
/**
 * Less oft-needed functions to manipulate Tokens
 */
class Tokenization {
    /**
     * Un-encode a string potentially containing encoded tokens
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a Tokenized value from a number
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     */
    static resolve(obj, options) {
        return resolve_1.resolve(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: options.preparing !== undefined ? options.preparing : false,
        });
    }
    /**
     * Return whether the given object is an IResolvable object
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     */
    static isResolvable(obj) {
        return isResolvableObject(obj);
    }
    /**
     * Stringify a number directly or lazily if it's a Token. If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.stringValue({
                produce: (context) => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== "number" ? resolved : `${resolved}`;
                },
            });
        }
        else {
            return typeof x !== "number" ? x : `${x}`;
        }
    }
    constructor() { }
}
exports.Tokenization = Tokenization;
function isResolvableObject(x) {
    return typeof x === "object" && x !== null && typeof x.resolve === "function";
}
exports.isResolvableObject = isResolvableObject;
function withResolved(...args) {
    if (args.length < 2) {
        return;
    }
    const argArray = args.slice(0, args.length - 1);
    if (argArray.some(Token.isUnresolved)) {
        return;
    }
    args[args.length - 1].apply(arguments, argArray);
}
exports.withResolved = withResolved;
//# sourceMappingURL=data:application/json;base64,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