"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagManager = exports.TagType = void 0;
var TagType;
(function (TagType) {
    TagType["STANDARD"] = "StandardTag";
    TagType["AUTOSCALING_GROUP"] = "AutoScalingGroupTag";
    TagType["MAP"] = "StringToStringMap";
    TagType["KEY_VALUE"] = "KeyValue";
    TagType["NOT_TAGGABLE"] = "NotTaggable";
})(TagType = exports.TagType || (exports.TagType = {}));
/**
 * Standard tags are a list of { key, value } objects
 */
class StandardFormatter {
    parseTags(rosPropertyTags, priority) {
        if (!Array.isArray(rosPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value} have ${JSON.stringify(rosPropertyTags)}`);
        }
        const tags = [];
        for (const tag of rosPropertyTags) {
            if (tag.key === undefined || tag.value === undefined) {
                throw new Error(`Invalid tag input expected {key, value} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const rosTags = [];
        for (const tag of tags) {
            rosTags.push({
                key: tag.key,
                value: tag.value,
            });
        }
        return rosTags.length === 0 ? undefined : rosTags;
    }
}
/**
 * ASG tags are a list of { key, value, propagateAtLaunch } objects
 */
class AsgFormatter {
    parseTags(rosPropertyTags, priority) {
        const tags = [];
        if (!Array.isArray(rosPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value, propagateAtLaunch} have ${JSON.stringify(rosPropertyTags)}`);
        }
        for (const tag of rosPropertyTags) {
            if (tag.key === undefined ||
                tag.value === undefined ||
                tag.propagateAtLaunch === undefined) {
                throw new Error(`Invalid tag input expected {key, value, propagateAtLaunch} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
                applyToLaunchedInstances: !!tag.propagateAtLaunch,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const rosTags = [];
        for (const tag of tags) {
            rosTags.push({
                key: tag.key,
                value: tag.value,
                propagateAtLaunch: tag.applyToLaunchedInstances !== false,
            });
        }
        return rosTags.length === 0 ? undefined : rosTags;
    }
}
/**
 * Some ROS constructs use a { key: value } map for tags
 */
class MapFormatter {
    parseTags(rosPropertyTags, priority) {
        const tags = [];
        if (Array.isArray(rosPropertyTags) || typeof rosPropertyTags !== "object") {
            throw new Error(`Invalid tag input expected map of {key: value} have ${JSON.stringify(rosPropertyTags)}`);
        }
        for (const [key, value] of Object.entries(rosPropertyTags)) {
            tags.push({
                key,
                value: `${value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const rosTags = {};
        for (const tag of tags) {
            rosTags[`${tag.key}`] = `${tag.value}`;
        }
        return Object.keys(rosTags).length === 0 ? undefined : rosTags;
    }
}
/**
 * StackTags are of the format { Key: key, Value: value }
 */
class KeyValueFormatter {
    parseTags(keyValueTags, priority) {
        const tags = [];
        for (const key in keyValueTags) {
            if (keyValueTags.hasOwnProperty(key)) {
                const value = keyValueTags[key];
                tags.push({
                    key,
                    value,
                    priority,
                });
            }
        }
        return tags;
    }
    formatTags(unformattedTags) {
        const tags = [];
        unformattedTags.forEach((tag) => {
            tags.push({
                Key: tag.key,
                Value: tag.value,
            });
        });
        return tags;
    }
}
class NoFormat {
    parseTags(_rosPropertyTags) {
        return [];
    }
    formatTags(_tags) {
        return undefined;
    }
}
const TAG_FORMATTERS = {
    [TagType.AUTOSCALING_GROUP]: new AsgFormatter(),
    [TagType.STANDARD]: new StandardFormatter(),
    [TagType.MAP]: new MapFormatter(),
    [TagType.KEY_VALUE]: new KeyValueFormatter(),
    [TagType.NOT_TAGGABLE]: new NoFormat(),
};
/**
 * TagManager facilitates a common implementation of tagging for Constructs.
 */
class TagManager {
    constructor(tagType, resourceTypeName, tagStructure, options = {}) {
        this.tags = new Map();
        this.priorities = new Map();
        this.initialTagPriority = 50;
        this.resourceTypeName = resourceTypeName;
        this.tagFormatter = TAG_FORMATTERS[tagType];
        if (tagStructure !== undefined) {
            this._setTag(...this.tagFormatter.parseTags(tagStructure, this.initialTagPriority));
        }
        this.tagPropertyName = options.tagPropertyName || "tags";
    }
    /**
     * Check whether the given construct is Taggable
     */
    static isTaggable(construct) {
        return construct.tags !== undefined;
    }
    /**
     * Adds the specified tag to the array of tags
     *
     */
    setTag(key, value, priority = 0, applyToLaunchedInstances = true) {
        // This method mostly exists because we don't want to expose the 'Tag' type used (it will be confusing
        // to users).
        this._setTag({ key, value, priority, applyToLaunchedInstances });
    }
    /**
     * Removes the specified tag from the array if it exists
     *
     * @param key The tag to remove
     * @param priority The priority of the remove operation
     */
    removeTag(key, priority) {
        if (priority >= (this.priorities.get(key) || 0)) {
            this.tags.delete(key);
            this.priorities.set(key, priority);
        }
    }
    /**
     * Renders tags into the proper format based on TagType
     */
    renderTags() {
        const sortedTags = Array.from(this.tags.values()).sort((a, b) => a.key.localeCompare(b.key));
        let temp = this.tagFormatter.formatTags(sortedTags);
        return temp;
    }
    /**
     * Determine if the aspect applies here
     *
     * Looks at the include and exclude resourceTypeName arrays to determine if
     * the aspect applies here
     */
    applyTagAspectHere(include, exclude) {
        if (exclude &&
            exclude.length > 0 &&
            exclude.indexOf(this.resourceTypeName) !== -1) {
            return false;
        }
        if (include &&
            include.length > 0 &&
            include.indexOf(this.resourceTypeName) === -1) {
            return false;
        }
        return true;
    }
    /**
     * Returns true if there are any tags defined
     */
    hasTags() {
        return this.tags.size > 0;
    }
    _setTag(...tags) {
        for (const tag of tags) {
            if (tag.priority >= (this.priorities.get(tag.key) || 0)) {
                this.tags.set(tag.key, tag);
                this.priorities.set(tag.key, tag.priority);
            }
        }
    }
}
exports.TagManager = TagManager;
//# sourceMappingURL=data:application/json;base64,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