"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenizedStringFragments = void 0;
const token_1 = require("./token");
/**
 * Fragments of a concatenated string containing stringified Tokens
 */
class TokenizedStringFragments {
    constructor() {
        this.fragments = new Array();
    }
    get firstToken() {
        const first = this.fragments[0];
        if (first.type === "token") {
            return first.token;
        }
        return undefined;
    }
    get firstValue() {
        return fragmentValue(this.fragments[0]);
    }
    get length() {
        return this.fragments.length;
    }
    addLiteral(lit) {
        this.fragments.push({ type: "literal", lit });
    }
    addToken(token) {
        this.fragments.push({ type: "token", token });
    }
    addIntrinsic(value) {
        this.fragments.push({ type: "intrinsic", value });
    }
    /**
     * Return all Tokens from this string
     */
    get tokens() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === "token") {
                ret.push(f.token);
            }
        }
        return ret;
    }
    /**
     * Apply a transformation function to all tokens in the string
     */
    mapTokens(mapper) {
        const ret = new TokenizedStringFragments();
        for (const f of this.fragments) {
            switch (f.type) {
                case "literal":
                    ret.addLiteral(f.lit);
                    break;
                case "token":
                    const mapped = mapper.mapToken(f.token);
                    if (token_1.isResolvableObject(mapped)) {
                        ret.addToken(mapped);
                    }
                    else {
                        ret.addIntrinsic(mapped);
                    }
                    break;
                case "intrinsic":
                    ret.addIntrinsic(f.value);
                    break;
            }
        }
        return ret;
    }
    /**
     * Combine the string fragments using the given joiner.
     *
     * If there are any
     */
    join(concat) {
        if (this.fragments.length === 0) {
            return concat.join(undefined, undefined);
        }
        if (this.fragments.length === 1) {
            return this.firstValue;
        }
        const values = this.fragments.map(fragmentValue);
        while (values.length > 1) {
            const prefix = values.splice(0, 2);
            values.splice(0, 0, concat.join(prefix[0], prefix[1]));
        }
        return values[0];
    }
}
exports.TokenizedStringFragments = TokenizedStringFragments;
/**
 * Resolve the value from a single fragment
 *
 * If the fragment is a Token, return the string encoding of the Token.
 */
function fragmentValue(fragment) {
    switch (fragment.type) {
        case "literal":
            return fragment.lit;
        case "token":
            return fragment.token.toString();
        case "intrinsic":
            return fragment.value;
    }
}
//# sourceMappingURL=data:application/json;base64,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