"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const cxapi = require("@alicloud/ros-cdk-cxapi");
const construct_compat_1 = require("./construct-compat");
const synthesis_1 = require("./private/synthesis");
/**
 * An abstract application modeling unit consisting of Stacks that should be
 * deployed together.
 *
 * Derive a subclass of `Stage` and use it to model a single instance of your
 * application.
 *
 * You can then instantiate your subclass multiple times to model multiple
 * copies of your application which should be be deployed to different
 * environments.
 */
class Stage extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        if (id !== "" && !/^[a-z][a-z0-9\-\_\.]+$/i.test(id)) {
            throw new Error(`invalid stage name "${id}". Stage name must start with a letter and contain only alphanumeric characters, hypens ('-'), underscores ('_') and periods ('.')`);
        }
        this.parentStage = Stage.of(this);
        // this.region = props.env?.region ?? this.parentStage?.region;
        // this.account = props.env?.account ?? this.parentStage?.account;
        this._assemblyBuilder = this.createBuilder(props.outdir);
        this.stageName = [(_a = this.parentStage) === null || _a === void 0 ? void 0 : _a.stageName, id]
            .filter((x) => x)
            .join("-");
    }
    /**
     * Return the stage this construct is contained with, if available. If called
     * on a nested stage, returns its parent.
     *
     * @experimental
     */
    static of(construct) {
        return construct.node.scopes.reverse().slice(1).find(Stage.isStage);
    }
    /**
     * Test whether the given construct is a stage.
     *
     * @experimental
     */
    static isStage(x) {
        return x !== null && x instanceof Stage;
    }
    /**
     * Artifact ID of the assembly if it is a nested stage. The root stage (app)
     * will return an empty string.
     *
     * Derived from the construct path.
     *
     * @experimental
     */
    get artifactId() {
        if (!this.node.path) {
            return "";
        }
        return `assembly-${this.node.path
            .replace(/\//g, "-")
            .replace(/^-+|-+$/g, "")}`;
    }
    /**
     * Synthesize this stage into a cloud assembly.
     *
     * Once an assembly has been synthesized, it cannot be modified. Subsequent
     * calls will return the same assembly.
     */
    synth(options = {}) {
        if (!this.assembly) {
            this.assembly = synthesis_1.synthesize(this, {
                skipValidation: options.skipValidation,
            });
        }
        return this.assembly;
    }
    createBuilder(outdir) {
        // cannot specify "outdir" if we are a nested stage
        if (this.parentStage && outdir) {
            throw new Error('"outdir" cannot be specified for nested stages');
        }
        // Need to determine fixed output directory already, because we must know where
        // to write sub-assemblies (which must happen before we actually get to this app's
        // synthesize() phase).
        return this.parentStage
            ? this.parentStage._assemblyBuilder.createNestedAssembly(this.artifactId, this.node.path)
            : new cxapi.CloudAssemblyBuilder(outdir);
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhZ2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGFnZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxpREFBaUQ7QUFDakQseURBQTJEO0FBQzNELG1EQUFpRDtBQW9CakQ7Ozs7Ozs7Ozs7R0FVRztBQUNILE1BQWEsS0FBTSxTQUFRLDRCQUFTO0lBOERsQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQW9CLEVBQUU7O1FBQzlELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMseUJBQXlCLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFO1lBQ3BELE1BQU0sSUFBSSxLQUFLLENBQ2IsdUJBQXVCLEVBQUUsb0lBQW9JLENBQzlKLENBQUM7U0FDSDtRQUVELElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVsQywrREFBK0Q7UUFDL0Qsa0VBQWtFO1FBRWxFLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6RCxJQUFJLENBQUMsU0FBUyxHQUFHLE9BQUMsSUFBSSxDQUFDLFdBQVcsMENBQUUsU0FBUyxFQUFFLEVBQUUsQ0FBQzthQUMvQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQzthQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDZixDQUFDO0lBL0VEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxTQUFxQjtRQUNwQyxPQUFPLFNBQVMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3RFLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFNO1FBQzFCLE9BQU8sQ0FBQyxLQUFLLElBQUksSUFBSSxDQUFDLFlBQVksS0FBSyxDQUFDO0lBQzFDLENBQUM7SUFnRUQ7Ozs7Ozs7T0FPRztJQUNILElBQVcsVUFBVTtRQUNuQixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDbkIsT0FBTyxFQUFFLENBQUM7U0FDWDtRQUNELE9BQU8sWUFBWSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUk7YUFDOUIsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUM7YUFDbkIsT0FBTyxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLEtBQUssQ0FBQyxVQUFpQyxFQUFFO1FBQzlDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2xCLElBQUksQ0FBQyxRQUFRLEdBQUcsc0JBQVUsQ0FBQyxJQUFJLEVBQUU7Z0JBQy9CLGNBQWMsRUFBRSxPQUFPLENBQUMsY0FBYzthQUN2QyxDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQztJQUN2QixDQUFDO0lBRU8sYUFBYSxDQUFDLE1BQWU7UUFDbkMsbURBQW1EO1FBQ25ELElBQUksSUFBSSxDQUFDLFdBQVcsSUFBSSxNQUFNLEVBQUU7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxnREFBZ0QsQ0FBQyxDQUFDO1NBQ25FO1FBRUQsK0VBQStFO1FBQy9FLGtGQUFrRjtRQUNsRix1QkFBdUI7UUFDdkIsT0FBTyxJQUFJLENBQUMsV0FBVztZQUNyQixDQUFDLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsQ0FBQyxvQkFBb0IsQ0FDcEQsSUFBSSxDQUFDLFVBQVUsRUFDZixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDZjtZQUNILENBQUMsQ0FBQyxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUM3QyxDQUFDO0NBQ0Y7QUFuSUQsc0JBbUlDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY3hhcGkgZnJvbSBcIkBhbGljbG91ZC9yb3MtY2RrLWN4YXBpXCI7XHJcbmltcG9ydCB7IENvbnN0cnVjdCwgSUNvbnN0cnVjdCB9IGZyb20gXCIuL2NvbnN0cnVjdC1jb21wYXRcIjtcclxuaW1wb3J0IHsgc3ludGhlc2l6ZSB9IGZyb20gXCIuL3ByaXZhdGUvc3ludGhlc2lzXCI7XHJcblxyXG4vKipcclxuICogSW5pdGlhbGl6YXRpb24gcHJvcHMgZm9yIGEgc3RhZ2UuXHJcbiAqL1xyXG5leHBvcnQgaW50ZXJmYWNlIFN0YWdlUHJvcHMge1xyXG4gIC8qKlxyXG4gICAqIFRoZSBvdXRwdXQgZGlyZWN0b3J5IGludG8gd2hpY2ggdG8gZW1pdCBzeW50aGVzaXplZCBhcnRpZmFjdHMuXHJcbiAgICpcclxuICAgKiBDYW4gb25seSBiZSBzcGVjaWZpZWQgaWYgdGhpcyBzdGFnZSBpcyB0aGUgcm9vdCBzdGFnZSAodGhlIGFwcCkuIElmIHRoaXMgaXNcclxuICAgKiBzcGVjaWZpZWQgYW5kIHRoaXMgc3RhZ2UgaXMgbmVzdGVkIHdpdGhpbiBhbm90aGVyIHN0YWdlLCBhbiBlcnJvciB3aWxsIGJlXHJcbiAgICogdGhyb3duLlxyXG4gICAqXHJcbiAgICogQGRlZmF1bHQgLSBmb3IgbmVzdGVkIHN0YWdlcywgb3V0ZGlyIHdpbGwgYmUgZGV0ZXJtaW5lZCBhcyBhIHJlbGF0aXZlXHJcbiAgICogZGlyZWN0b3J5IHRvIHRoZSBvdXRkaXIgb2YgdGhlIGFwcC4gRm9yIGFwcHMsIGlmIG91dGRpciBpcyBub3Qgc3BlY2lmaWVkLCBhXHJcbiAgICogdGVtcG9yYXJ5IGRpcmVjdG9yeSB3aWxsIGJlIGNyZWF0ZWQuXHJcbiAgICovXHJcbiAgcmVhZG9ubHkgb3V0ZGlyPzogc3RyaW5nO1xyXG59XHJcblxyXG4vKipcclxuICogQW4gYWJzdHJhY3QgYXBwbGljYXRpb24gbW9kZWxpbmcgdW5pdCBjb25zaXN0aW5nIG9mIFN0YWNrcyB0aGF0IHNob3VsZCBiZVxyXG4gKiBkZXBsb3llZCB0b2dldGhlci5cclxuICpcclxuICogRGVyaXZlIGEgc3ViY2xhc3Mgb2YgYFN0YWdlYCBhbmQgdXNlIGl0IHRvIG1vZGVsIGEgc2luZ2xlIGluc3RhbmNlIG9mIHlvdXJcclxuICogYXBwbGljYXRpb24uXHJcbiAqXHJcbiAqIFlvdSBjYW4gdGhlbiBpbnN0YW50aWF0ZSB5b3VyIHN1YmNsYXNzIG11bHRpcGxlIHRpbWVzIHRvIG1vZGVsIG11bHRpcGxlXHJcbiAqIGNvcGllcyBvZiB5b3VyIGFwcGxpY2F0aW9uIHdoaWNoIHNob3VsZCBiZSBiZSBkZXBsb3llZCB0byBkaWZmZXJlbnRcclxuICogZW52aXJvbm1lbnRzLlxyXG4gKi9cclxuZXhwb3J0IGNsYXNzIFN0YWdlIGV4dGVuZHMgQ29uc3RydWN0IHtcclxuICAvKipcclxuICAgKiBSZXR1cm4gdGhlIHN0YWdlIHRoaXMgY29uc3RydWN0IGlzIGNvbnRhaW5lZCB3aXRoLCBpZiBhdmFpbGFibGUuIElmIGNhbGxlZFxyXG4gICAqIG9uIGEgbmVzdGVkIHN0YWdlLCByZXR1cm5zIGl0cyBwYXJlbnQuXHJcbiAgICpcclxuICAgKiBAZXhwZXJpbWVudGFsXHJcbiAgICovXHJcbiAgcHVibGljIHN0YXRpYyBvZihjb25zdHJ1Y3Q6IElDb25zdHJ1Y3QpOiBTdGFnZSB8IHVuZGVmaW5lZCB7XHJcbiAgICByZXR1cm4gY29uc3RydWN0Lm5vZGUuc2NvcGVzLnJldmVyc2UoKS5zbGljZSgxKS5maW5kKFN0YWdlLmlzU3RhZ2UpO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogVGVzdCB3aGV0aGVyIHRoZSBnaXZlbiBjb25zdHJ1Y3QgaXMgYSBzdGFnZS5cclxuICAgKlxyXG4gICAqIEBleHBlcmltZW50YWxcclxuICAgKi9cclxuICBwdWJsaWMgc3RhdGljIGlzU3RhZ2UoeDogYW55KTogeCBpcyBTdGFnZSB7XHJcbiAgICByZXR1cm4geCAhPT0gbnVsbCAmJiB4IGluc3RhbmNlb2YgU3RhZ2U7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBUaGUgZGVmYXVsdCByZWdpb24gZm9yIGFsbCByZXNvdXJjZXMgZGVmaW5lZCB3aXRoaW4gdGhpcyBzdGFnZS5cclxuICAgKlxyXG4gICAqIEBleHBlcmltZW50YWxcclxuICAgKi9cclxuICAvLyBwdWJsaWMgcmVhZG9ubHkgcmVnaW9uPzogc3RyaW5nO1xyXG5cclxuICAvKipcclxuICAgKiBUaGUgZGVmYXVsdCBhY2NvdW50IGZvciBhbGwgcmVzb3VyY2VzIGRlZmluZWQgd2l0aGluIHRoaXMgc3RhZ2UuXHJcbiAgICpcclxuICAgKiBAZXhwZXJpbWVudGFsXHJcbiAgICovXHJcbiAgLy8gcHVibGljIHJlYWRvbmx5IGFjY291bnQ/OiBzdHJpbmc7XHJcblxyXG4gIC8qKlxyXG4gICAqIFRoZSBjbG91ZCBhc3NlbWJseSBidWlsZGVyIHRoYXQgaXMgYmVpbmcgdXNlZCBmb3IgdGhpcyBBcHBcclxuICAgKlxyXG4gICAqIEBleHBlcmltZW50YWxcclxuICAgKiBAaW50ZXJuYWxcclxuICAgKi9cclxuICBwdWJsaWMgcmVhZG9ubHkgX2Fzc2VtYmx5QnVpbGRlcjogY3hhcGkuQ2xvdWRBc3NlbWJseUJ1aWxkZXI7XHJcblxyXG4gIC8qKlxyXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzdGFnZS4gQmFzZWQgb24gbmFtZXMgb2YgdGhlIHBhcmVudCBzdGFnZXMgc2VwYXJhdGVkIGJ5XHJcbiAgICogaHlwZW5zLlxyXG4gICAqXHJcbiAgICogQGV4cGVyaW1lbnRhbFxyXG4gICAqL1xyXG4gIHB1YmxpYyByZWFkb25seSBzdGFnZU5hbWU6IHN0cmluZztcclxuXHJcbiAgLyoqXHJcbiAgICogVGhlIHBhcmVudCBzdGFnZSBvciBgdW5kZWZpbmVkYCBpZiB0aGlzIGlzIHRoZSBhcHAuXHJcbiAgICogKlxyXG4gICAqIEBleHBlcmltZW50YWxcclxuICAgKi9cclxuICBwdWJsaWMgcmVhZG9ubHkgcGFyZW50U3RhZ2U/OiBTdGFnZTtcclxuXHJcbiAgLyoqXHJcbiAgICogVGhlIGNhY2hlZCBhc3NlbWJseSBpZiBpdCB3YXMgYWxyZWFkeSBidWlsdFxyXG4gICAqL1xyXG4gIHByaXZhdGUgYXNzZW1ibHk/OiBjeGFwaS5DbG91ZEFzc2VtYmx5O1xyXG5cclxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU3RhZ2VQcm9wcyA9IHt9KSB7XHJcbiAgICBzdXBlcihzY29wZSwgaWQpO1xyXG5cclxuICAgIGlmIChpZCAhPT0gXCJcIiAmJiAhL15bYS16XVthLXowLTlcXC1cXF9cXC5dKyQvaS50ZXN0KGlkKSkge1xyXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXHJcbiAgICAgICAgYGludmFsaWQgc3RhZ2UgbmFtZSBcIiR7aWR9XCIuIFN0YWdlIG5hbWUgbXVzdCBzdGFydCB3aXRoIGEgbGV0dGVyIGFuZCBjb250YWluIG9ubHkgYWxwaGFudW1lcmljIGNoYXJhY3RlcnMsIGh5cGVucyAoJy0nKSwgdW5kZXJzY29yZXMgKCdfJykgYW5kIHBlcmlvZHMgKCcuJylgXHJcbiAgICAgICk7XHJcbiAgICB9XHJcblxyXG4gICAgdGhpcy5wYXJlbnRTdGFnZSA9IFN0YWdlLm9mKHRoaXMpO1xyXG5cclxuICAgIC8vIHRoaXMucmVnaW9uID0gcHJvcHMuZW52Py5yZWdpb24gPz8gdGhpcy5wYXJlbnRTdGFnZT8ucmVnaW9uO1xyXG4gICAgLy8gdGhpcy5hY2NvdW50ID0gcHJvcHMuZW52Py5hY2NvdW50ID8/IHRoaXMucGFyZW50U3RhZ2U/LmFjY291bnQ7XHJcblxyXG4gICAgdGhpcy5fYXNzZW1ibHlCdWlsZGVyID0gdGhpcy5jcmVhdGVCdWlsZGVyKHByb3BzLm91dGRpcik7XHJcbiAgICB0aGlzLnN0YWdlTmFtZSA9IFt0aGlzLnBhcmVudFN0YWdlPy5zdGFnZU5hbWUsIGlkXVxyXG4gICAgICAuZmlsdGVyKCh4KSA9PiB4KVxyXG4gICAgICAuam9pbihcIi1cIik7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBBcnRpZmFjdCBJRCBvZiB0aGUgYXNzZW1ibHkgaWYgaXQgaXMgYSBuZXN0ZWQgc3RhZ2UuIFRoZSByb290IHN0YWdlIChhcHApXHJcbiAgICogd2lsbCByZXR1cm4gYW4gZW1wdHkgc3RyaW5nLlxyXG4gICAqXHJcbiAgICogRGVyaXZlZCBmcm9tIHRoZSBjb25zdHJ1Y3QgcGF0aC5cclxuICAgKlxyXG4gICAqIEBleHBlcmltZW50YWxcclxuICAgKi9cclxuICBwdWJsaWMgZ2V0IGFydGlmYWN0SWQoKSB7XHJcbiAgICBpZiAoIXRoaXMubm9kZS5wYXRoKSB7XHJcbiAgICAgIHJldHVybiBcIlwiO1xyXG4gICAgfVxyXG4gICAgcmV0dXJuIGBhc3NlbWJseS0ke3RoaXMubm9kZS5wYXRoXHJcbiAgICAgIC5yZXBsYWNlKC9cXC8vZywgXCItXCIpXHJcbiAgICAgIC5yZXBsYWNlKC9eLSt8LSskL2csIFwiXCIpfWA7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBTeW50aGVzaXplIHRoaXMgc3RhZ2UgaW50byBhIGNsb3VkIGFzc2VtYmx5LlxyXG4gICAqXHJcbiAgICogT25jZSBhbiBhc3NlbWJseSBoYXMgYmVlbiBzeW50aGVzaXplZCwgaXQgY2Fubm90IGJlIG1vZGlmaWVkLiBTdWJzZXF1ZW50XHJcbiAgICogY2FsbHMgd2lsbCByZXR1cm4gdGhlIHNhbWUgYXNzZW1ibHkuXHJcbiAgICovXHJcbiAgcHVibGljIHN5bnRoKG9wdGlvbnM6IFN0YWdlU3ludGhlc2lzT3B0aW9ucyA9IHt9KTogY3hhcGkuQ2xvdWRBc3NlbWJseSB7XHJcbiAgICBpZiAoIXRoaXMuYXNzZW1ibHkpIHtcclxuICAgICAgdGhpcy5hc3NlbWJseSA9IHN5bnRoZXNpemUodGhpcywge1xyXG4gICAgICAgIHNraXBWYWxpZGF0aW9uOiBvcHRpb25zLnNraXBWYWxpZGF0aW9uLFxyXG4gICAgICB9KTtcclxuICAgIH1cclxuXHJcbiAgICByZXR1cm4gdGhpcy5hc3NlbWJseTtcclxuICB9XHJcblxyXG4gIHByaXZhdGUgY3JlYXRlQnVpbGRlcihvdXRkaXI/OiBzdHJpbmcpIHtcclxuICAgIC8vIGNhbm5vdCBzcGVjaWZ5IFwib3V0ZGlyXCIgaWYgd2UgYXJlIGEgbmVzdGVkIHN0YWdlXHJcbiAgICBpZiAodGhpcy5wYXJlbnRTdGFnZSAmJiBvdXRkaXIpIHtcclxuICAgICAgdGhyb3cgbmV3IEVycm9yKCdcIm91dGRpclwiIGNhbm5vdCBiZSBzcGVjaWZpZWQgZm9yIG5lc3RlZCBzdGFnZXMnKTtcclxuICAgIH1cclxuXHJcbiAgICAvLyBOZWVkIHRvIGRldGVybWluZSBmaXhlZCBvdXRwdXQgZGlyZWN0b3J5IGFscmVhZHksIGJlY2F1c2Ugd2UgbXVzdCBrbm93IHdoZXJlXHJcbiAgICAvLyB0byB3cml0ZSBzdWItYXNzZW1ibGllcyAod2hpY2ggbXVzdCBoYXBwZW4gYmVmb3JlIHdlIGFjdHVhbGx5IGdldCB0byB0aGlzIGFwcCdzXHJcbiAgICAvLyBzeW50aGVzaXplKCkgcGhhc2UpLlxyXG4gICAgcmV0dXJuIHRoaXMucGFyZW50U3RhZ2VcclxuICAgICAgPyB0aGlzLnBhcmVudFN0YWdlLl9hc3NlbWJseUJ1aWxkZXIuY3JlYXRlTmVzdGVkQXNzZW1ibHkoXHJcbiAgICAgICAgICB0aGlzLmFydGlmYWN0SWQsXHJcbiAgICAgICAgICB0aGlzLm5vZGUucGF0aFxyXG4gICAgICAgIClcclxuICAgICAgOiBuZXcgY3hhcGkuQ2xvdWRBc3NlbWJseUJ1aWxkZXIob3V0ZGlyKTtcclxuICB9XHJcbn1cclxuXHJcbi8qKlxyXG4gKiBPcHRpb25zIGZvciBhc3NlbWx5IHN5bnRoZXNpcy5cclxuICovXHJcbmV4cG9ydCBpbnRlcmZhY2UgU3RhZ2VTeW50aGVzaXNPcHRpb25zIHtcclxuICAvKipcclxuICAgKiBTaG91bGQgd2Ugc2tpcCBjb25zdHJ1Y3QgdmFsaWRhdGlvbi5cclxuICAgKiBAZGVmYXVsdCAtIGZhbHNlXHJcbiAgICovXHJcbiAgcmVhZG9ubHkgc2tpcFZhbGlkYXRpb24/OiBib29sZWFuO1xyXG59XHJcbiJdfQ==