"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.assertBound = exports.addStackArtifactToAssembly = void 0;
const cxschema = require("@alicloud/ros-cdk-assembly-schema");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
/**
 * Shared logic of writing stack artifact to the Cloud Assembly
 *
 * This logic is shared between StackSyntheses.
 *
 * It could have been a protected method on a base class, but it
 * uses `Partial<cxapi.AliyunRosStackProperties>` in the
 * parameters (which is convenient so I can remain typesafe without
 * copy/pasting), and jsii will choke on this type.
 */
function addStackArtifactToAssembly(session, stack, stackProps, additionalStackDependencies) {
    // nested stack tags are applied at the ALIYUN::ROS::Stack resource
    // level and are not needed in the cloud assembly.
    // TODO: move these to the cloud assembly artifact properties instead of metadata
    if (stack.tags.hasTags()) {
        stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.STACK_TAGS, stack.tags.renderTags());
    }
    const deps = [
        ...stack.dependencies.map((s) => s.artifactId),
        ...additionalStackDependencies,
    ];
    const meta = collectStackMetadata(stack);
    // backwards compatibility since originally artifact ID was always equal to
    // stack name the stackName attribute is optional and if it is not specified
    // the CLI will use the artifact ID as the stack name. we *could have*
    // always put the stack name here but wanted to minimize the risk around
    // changes to the assembly manifest. so this means that as long as stack
    // name and artifact ID are the same, the cloud assembly manifest will not
    // change.
    const stackNameProperty = stack.stackName === stack.artifactId ? {} : { stackName: stack.stackName };
    const properties = {
        templateFile: stack.templateFile,
        ...stackProps,
        ...stackNameProperty,
    };
    // add an artifact that represents this stack
    session.assembly.addArtifact(stack.artifactId, {
        type: cxschema.ArtifactType.ALIYUN_ROS_STACK,
        properties,
        dependencies: deps.length > 0 ? deps : undefined,
        metadata: Object.keys(meta).length > 0 ? meta : undefined,
    });
}
exports.addStackArtifactToAssembly = addStackArtifactToAssembly;
/**
 * Collect the metadata from a stack
 */
function collectStackMetadata(stack) {
    const output = {};
    visit(stack);
    return output;
    function visit(node) {
        // break off if we reached a node that is not a child of this stack
        const parent = findParentStack(node);
        if (parent !== stack) {
            return;
        }
        if (node.node.metadata.length > 0) {
            // Make the path absolute
            output[construct_compat_1.ConstructNode.PATH_SEP + node.node.path] = node.node.metadata.map((md) => stack.resolve(md));
        }
        for (const child of node.node.children) {
            visit(child);
        }
    }
    function findParentStack(node) {
        if (node instanceof stack_1.Stack && node.nestedStackParent === undefined) {
            return node;
        }
        if (!node.node.scope) {
            return undefined;
        }
        return findParentStack(node.node.scope);
    }
}
/**
 * Throw an error message about binding() if we don't have a value for x.
 *
 * This replaces the ! assertions we would need everywhere otherwise.
 */
function assertBound(x) {
    if (x === null && x === undefined) {
        throw new Error("You must call bindStack() first");
    }
}
exports.assertBound = assertBound;
//# sourceMappingURL=data:application/json;base64,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