"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.unionValidator = exports.requireProperty = exports.requiredValidator = exports.propertyValidator = exports.hashValidator = exports.listValidator = exports.validateRosTag = exports.validateObject = exports.validateDate = exports.validateAny = exports.validateBoolean = exports.validateNumber = exports.validateString = exports.validateAllowedValues = exports.validateAllowedPattern = exports.validateRange = exports.validateLength = exports.canInspect = exports.VALIDATION_SUCCESS = exports.ValidationResults = exports.ValidationResult = exports.unionMapper = exports.hashMapper = exports.listMapper = exports.rosTagToRosTemplate = exports.dateToRosTemplate = exports.numberToRosTemplate = exports.objectToRosTemplate = exports.booleanToRosTemplate = exports.stringToRosTemplate = void 0;
function identity(x) {
    return x;
}
exports.stringToRosTemplate = identity;
exports.booleanToRosTemplate = identity;
exports.objectToRosTemplate = identity;
exports.numberToRosTemplate = identity;
/**
 * The date needs to be formatted as an ISO date in UTC
 *
 * Some usage sites require a date, some require a timestamp. We'll
 * always output a timestamp and hope the parser on the other end
 * is smart enough to ignore the time part... (?)
 */
function dateToRosTemplate(x) {
    if (!x) {
        return undefined;
    }
    // tslint:disable-next-line:max-line-length
    return `${x.getUTCFullYear()}-${pad(x.getUTCMonth() + 1)}-${pad(x.getUTCDate())}T${pad(x.getUTCHours())}:${pad(x.getUTCMinutes())}:${pad(x.getUTCSeconds())}`;
}
exports.dateToRosTemplate = dateToRosTemplate;
/**
 * Pad a number to 2 decimal places
 */
function pad(x) {
    if (x < 10) {
        return "0" + x.toString();
    }
    return x.toString();
}
/**
 * Turn a tag object into the proper ROS representation
 */
function rosTagToRosTemplate(x) {
    return {
        Key: x.key,
        Value: x.value,
    };
}
exports.rosTagToRosTemplate = rosTagToRosTemplate;
function listMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        return x.map(elementMapper);
    };
}
exports.listMapper = listMapper;
function hashMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        const ret = {};
        Object.keys(x).forEach((key) => {
            ret[key] = elementMapper(x[key]);
        });
        return ret;
    };
}
exports.hashMapper = hashMapper;
/**
 * Return a union mapper
 *
 * Takes a list of validators and a list of mappers, which should correspond pairwise.
 *
 * The mapper of the first successful validator will be called.
 */
function unionMapper(validators, mappers) {
    if (validators.length !== mappers.length) {
        throw Error("Not the same amount of validators and mappers passed to unionMapper()");
    }
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        for (let i = 0; i < validators.length; i++) {
            if (validators[i](x).isSuccess) {
                return mappers[i](x);
            }
        }
        // Should not be possible because the union must have passed validation before this function
        // will be called, but catch it anyway.
        throw new TypeError("No validators matched in the union()");
    };
}
exports.unionMapper = unionMapper;
// ----------------------------------------------------------------------
// VALIDATORS
//
// These are used while checking that supplied property bags match the expected schema
//
// We have a couple of datatypes that model validation errors and collections of validation
// errors (together forming a tree of errors so that we can trace validation errors through
// an object graph), and validators.
//
// Validators are simply functions that take a value and return a validation results. Then
// we have some combinators to turn primitive validators into more complex validators.
//
/**
 * Representation of validation results
 *
 * Models a tree of validation errors so that we have as much information as possible
 * about the failure that occurred.
 */
class ValidationResult {
    constructor(errorMessage = "", results = new ValidationResults()) {
        this.errorMessage = errorMessage;
        this.results = results;
    }
    get isSuccess() {
        return !this.errorMessage && this.results.isSuccess;
    }
    /**
     * Turn a failed validation into an exception
     */
    assertSuccess() {
        if (!this.isSuccess) {
            let message = this.errorTree();
            // The first letter will be lowercase, so uppercase it for a nicer error message
            message = message.substr(0, 1).toUpperCase() + message.substr(1);
            throw new RosSynthesisError(message);
        }
    }
    /**
     * Return a string rendering of the tree of validation failures
     */
    errorTree() {
        const childMessages = this.results.errorTreeList();
        return (this.errorMessage +
            (childMessages.length
                ? `\n  ${childMessages.replace(/\n/g, "\n  ")}`
                : ""));
    }
    /**
     * Wrap this result with an error message, if it concerns an error
     */
    prefix(message) {
        if (this.isSuccess) {
            return this;
        }
        return new ValidationResult(`${message}: ${this.errorMessage}`, this.results);
    }
}
exports.ValidationResult = ValidationResult;
/**
 * A collection of validation results
 */
class ValidationResults {
    constructor(results = []) {
        this.results = results;
    }
    collect(result) {
        // Only collect failures
        if (!result.isSuccess) {
            this.results.push(result);
        }
    }
    get isSuccess() {
        return this.results.every((x) => x.isSuccess);
    }
    errorTreeList() {
        return this.results.map((child) => child.errorTree()).join("\n");
    }
    /**
     * Wrap up all validation results into a single tree node
     *
     * If there are failures in the collection, add a message, otherwise
     * return a success.
     */
    wrap(message) {
        if (this.isSuccess) {
            return exports.VALIDATION_SUCCESS;
        }
        return new ValidationResult(message, this);
    }
}
exports.ValidationResults = ValidationResults;
// Singleton object to save on allocations
exports.VALIDATION_SUCCESS = new ValidationResult();
/**
 * Return whether this object can be validated at all
 *
 * True unless it's undefined or a ROS intrinsic
 */
function canInspect(x) {
    // Note: using weak equality on purpose, we also want to catch undefined
    return x != null && !isRosIntrinsic(x);
}
exports.canInspect = canInspect;
function validateLength(prop) {
    if (prop.min && prop.data < prop.min) {
        return new ValidationResult(`${JSON.stringify(prop.data)} is less than min value(${prop.min})`);
    }
    if (prop.max && prop.data > prop.max) {
        return new ValidationResult(`${JSON.stringify(prop.data)} is larger than max value(${prop.max})`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateLength = validateLength;
function validateRange(prop) {
    if (prop.min && prop.data < prop.min) {
        return new ValidationResult(`${JSON.stringify(prop.data)} is less than min value(${prop.min})`);
    }
    if (prop.max && prop.data > prop.max) {
        return new ValidationResult(`${JSON.stringify(prop.data)} is larger than min value(${prop.min})`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateRange = validateRange;
function validateAllowedPattern(prop) {
    const regExp = new RegExp(prop.reg);
    if (regExp.test(prop.data)) {
        return exports.VALIDATION_SUCCESS;
    }
    else {
        return new ValidationResult(`The string ${JSON.stringify(prop.data)} does not match the regular expression "${prop.reg}"`);
    }
}
exports.validateAllowedPattern = validateAllowedPattern;
function validateAllowedValues(prop) {
    for (let value of prop.allowedValues) {
        if (value === prop.data)
            return exports.VALIDATION_SUCCESS;
    }
    return new ValidationResult(`${JSON.stringify(prop.data)} doesn't exist in [${prop.allowedValues}]`);
}
exports.validateAllowedValues = validateAllowedValues;
// ROS validators for primitive types
function validateString(x) {
    if (canInspect(x) && typeof x !== "string") {
        return new ValidationResult(`${JSON.stringify(x)} should be a string`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateString = validateString;
function validateNumber(x) {
    if (canInspect(x) && typeof x !== "number") {
        return new ValidationResult(`${JSON.stringify(x)} should be a number`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateNumber = validateNumber;
function validateBoolean(x) {
    if (canInspect(x) && typeof x !== "boolean") {
        return new ValidationResult(`${JSON.stringify(x)} should be a boolean`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateBoolean = validateBoolean;
function validateAny(x) {
    // avoid tsc error -> 'x' is declared but its value is never read
    if (canInspect(x) || true) {
        return exports.VALIDATION_SUCCESS;
    }
}
exports.validateAny = validateAny;
function validateDate(x) {
    if (canInspect(x) && !(x instanceof Date)) {
        return new ValidationResult(`${JSON.stringify(x)} should be a Date`);
    }
    if (x !== undefined && isNaN(x.getTime())) {
        return new ValidationResult("got an unparseable Date");
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateDate = validateDate;
function validateObject(x) {
    if (canInspect(x) && typeof x !== "object") {
        return new ValidationResult(`${JSON.stringify(x)} should be an 'object'`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateObject = validateObject;
function validateRosTag(x) {
    if (!canInspect(x)) {
        return exports.VALIDATION_SUCCESS;
    }
    if (x.key == null || x.value == null) {
        return new ValidationResult(`${JSON.stringify(x)} should have a 'key' and a 'value' property`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateRosTag = validateRosTag;
/**
 * Return a list validator based on the given element validator
 */
function listValidator(elementValidator) {
    return (x) => {
        if (!canInspect(x)) {
            return exports.VALIDATION_SUCCESS;
        }
        if (!x.forEach) {
            return new ValidationResult(`${JSON.stringify(x)} should be a list`);
        }
        for (let i = 0; i < x.length; i++) {
            const element = x[i];
            const result = elementValidator(element);
            if (!result.isSuccess) {
                return result.prefix(`element ${i}`);
            }
        }
        return exports.VALIDATION_SUCCESS;
    };
}
exports.listValidator = listValidator;
/**
 * Return a hash validator based on the given element validator
 */
function hashValidator(elementValidator) {
    return (x) => {
        if (!canInspect(x)) {
            return exports.VALIDATION_SUCCESS;
        }
        for (const key of Object.keys(x)) {
            const result = elementValidator(x[key]);
            if (!result.isSuccess) {
                return result.prefix(`element '${key}'`);
            }
        }
        return exports.VALIDATION_SUCCESS;
    };
}
exports.hashValidator = hashValidator;
/**
 * Decorate a validator with a message clarifying the property the failure is for.
 */
function propertyValidator(propName, validator) {
    return (x) => {
        return validator(x).prefix(propName);
    };
}
exports.propertyValidator = propertyValidator;
/**
 * Return a validator that will fail if the passed property is not present
 *
 * Does not distinguish between the property actually not being present, vs being present but 'null'
 * or 'undefined' (courtesy of JavaScript), which is generally the behavior that we want.
 *
 * Empty strings are considered "present"--don't know if this agrees with how ROS looks at the world.
 */
function requiredValidator(x) {
    if (x == null) {
        return new ValidationResult("required but missing");
    }
    return exports.VALIDATION_SUCCESS;
}
exports.requiredValidator = requiredValidator;
/**
 * Require a property from a property bag.
 *
 * @param props  the property bag from which a property is required.
 * @param name   the name of the required property.
 * @param typeName the name of the construct type that requires the property
 *
 * @returns the value of ``props[name]``
 *
 * @throws if the property ``name`` is not present in ``props``.
 */
function requireProperty(props, name, context) {
    const value = props[name];
    if (value == null) {
        throw new Error(`${context.toString()} is missing required property: ${name}`);
    }
    // Possibly add type-checking here...
    return value;
}
exports.requireProperty = requireProperty;
/**
 * Validates if any of the given validators matches
 *
 * We add either/or words to the front of the error mesages so that they read
 * more nicely. Example:
 *
 *   Properties not correct for 'FunctionProps'
 *     codeUri: not one of the possible types
 *       either: properties not correct for 'S3LocationProperty'
 *         bucket: required but missing
 *         key: required but missing
 *         version: required but missing
 *       or: '3' should be a 'string'
 *
 */
function unionValidator(...validators) {
    return (x) => {
        const results = new ValidationResults();
        let eitherOr = "either";
        for (const validator of validators) {
            const result = validator(x);
            if (result.isSuccess) {
                return result;
            }
            results.collect(result.prefix(eitherOr));
            eitherOr = "or";
        }
        return results.wrap("not one of the possible types");
    };
}
exports.unionValidator = unionValidator;
/**
 * Return whether the indicated value represents a ROS intrinsic.
 *
 * ROS intrinsics are modeled as objects with a single key, which
 * look like: { "Fn::GetAtt": [...] } or similar.
 */
function isRosIntrinsic(x) {
    if (!(typeof x === "object")) {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === "Ref" || keys[0].substr(0, 4) === "Fn::";
}
// Cannot be public because JSII gets confused about es5.d.ts
class RosSynthesisError extends Error {
    constructor() {
        super(...arguments);
        this.type = "RosSynthesisError";
    }
}
//# sourceMappingURL=data:application/json;base64,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