"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RosResource = void 0;
const ros_resource_policy_1 = require("./ros-resource-policy");
const util_1 = require("./util");
const tag_manager_1 = require("./tag-manager");
const deps_1 = require("./deps");
const ros_reference_1 = require("./private/ros-reference");
const ros_removal_policy_1 = require("./ros-removal-policy");
const ros_element_1 = require("./ros-element");
/**
 * Represents a ROS resource.
 */
class RosResource extends ros_element_1.RosRefElement {
    /**
     * Creates a resource construct.
     * @param rosResourceType The ROS type of this resource (e.g. ALIYUN::ECS::Instance)
     */
    constructor(scope, id, props) {
        super(scope, id);
        // MAINTAINERS NOTE: this class serves as the base class for the generated L1
        // ("ALIYUN") resources (such as `ecs.Instance`). These resources will have a
        // property for each ROS property of the resource. This means that
        // if at some point in the future a property is introduced with a name similar
        // to one of the properties here, it will be "masked" by the derived class. To
        // that end, we prefix all properties in this class with `rosXxx` with the
        // hope to avoid those conflicts in the future.
        /**
         * Options for this resource, such as condition, update policy etc.
         */
        this.rosOptions = {};
        /**
         * An object to be merged on top of the entire resource definition.
         */
        this.rawOverrides = {};
        /**
         * Logical IDs of dependencies.
         *
         * Is filled during prepare().
         */
        this.dependsOn = new Set();
        this.rosDependsOn = new Set();
        this.id = id;
        if (!props.type) {
            throw new Error("The `type` property is required");
        }
        this.rosResourceType = props.type;
        this._rosProperties = props.properties || {};
    }
    /**
     * Check whether the given construct is a RosResource
     */
    static isRosResource(construct) {
        return construct.rosResourceType !== undefined;
    }
    /**
     * Sets the deletion policy of the resource based on the removal policy specified.
     */
    applyRemovalPolicy(policy, options = {}) {
        policy = policy || options.defaultPolicy || ros_removal_policy_1.RemovalPolicy.RETAIN;
        let deletionPolicy;
        switch (policy) {
            case ros_removal_policy_1.RemovalPolicy.DESTROY:
                deletionPolicy = ros_resource_policy_1.RosDeletionPolicy.DELETE;
                break;
            case ros_removal_policy_1.RemovalPolicy.RETAIN:
                deletionPolicy = ros_resource_policy_1.RosDeletionPolicy.RETAIN;
                break;
            default:
                throw new Error(`Invalid removal policy: ${policy}`);
        }
        this.rosOptions.deletionPolicy = deletionPolicy;
        // if (options.applyToUpdateReplacePolicy !== false) {
        //   this.rosOptions.updateReplacePolicy = deletionPolicy;
        // }
    }
    /**
     * Returns a token for an runtime attribute of this resource.
     * Ideally, use generated attribute accessors (e.g. `resource.arn`), but this can be used for future compatibility
     * in case there is no generated attribute.
     * @param attributeName The name of the attribute.
     */
    getAtt(attributeName) {
        return ros_reference_1.RosReference.for(this, attributeName);
    }
    /**
     * Adds an override to the synthesized ROS resource. To add a
     * property override, either use `addPropertyOverride` or prefix `path` with
     * "Properties." (i.e. `Properties.TopicName`).
     *
     * If the override is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * For example,
     * ```typescript
     * addOverride('Properties.GlobalSecondaryIndexes.0.Projection.NonKeyAttributes', ['myattribute'])
     * addOverride('Properties.GlobalSecondaryIndexes.1.ProjectionType', 'INCLUDE')
     * ```
     * would add the overrides
     * ```json
     * "Properties": {
     *   "GlobalSecondaryIndexes": [
     *     {
     *       "Projection": {
     *         "NonKeyAttributes": [ "myattribute" ]
     *         ...
     *       }
     *       ...
     *     },
     *     {
     *       "ProjectionType": "INCLUDE"
     *       ...
     *     },
     *   ]
     *   ...
     * }
     * ```
     *
     * @param path - The path of the property, you can use dot notation to
     *        override values in complex types. Any intermdediate keys
     *        will be created as needed.
     * @param value - The value. Could be primitive or complex.
     */
    addOverride(path, value) {
        const parts = path.split(".");
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null &&
                typeof curr[key] === "object" &&
                !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    /**
     * Syntactic sugar for `addOverride(path, undefined)`.
     * @param path The path of the value to delete
     */
    addDeletionOverride(path) {
        this.addOverride(path, undefined);
    }
    addMetaData(key, value) {
        if (!this.rosOptions.metadata) {
            this.rosOptions.metadata = {};
        }
        this.rosOptions.metadata[key] = value;
    }
    addDesc(desc) {
        this.rosOptions.description = desc;
    }
    addCondition(con) {
        this.rosOptions.condition = con;
    }
    /**
     * Adds an override to a resource property.
     *
     * Syntactic sugar for `addOverride("Properties.<...>", value)`.
     *
     * @param propertyPath The path of the property
     * @param value The value
     */
    addPropertyOverride(propertyPath, value) {
        this.addOverride(`Properties.${propertyPath}`, value);
    }
    /**
     * Adds an override that deletes the value of a property from the resource definition.
     * @param propertyPath The path to the property.
     */
    addPropertyDeletionOverride(propertyPath) {
        this.addPropertyOverride(propertyPath, undefined);
    }
    /**
     * Indicates that this resource depends on another resource and cannot be
     * provisioned unless the other resource has been successfully provisioned.
     *
     * This can be used for resources across stacks (or nested stack) boundaries
     * and the dependency will automatically be transferred to the relevant scope.
     */
    addDependsOn(target) {
        deps_1.addDependency(this, target, `"${this.node.path}" depends on "${target.node.path}"`);
    }
    /**
     * @returns a string representation of this resource
     */
    toString() {
        return `${super.toString()} [${this.rosResourceType}]`;
    }
    /**
     * Called by the `addDependency` helper function in order to realize a direct
     * dependency between two resources that are directly defined in the same
     * stacks.
     *
     * Use `resource.addDependsOn` to define the dependency between two resources,
     * which also takes stack boundaries into account.
     *
     * @internal
     */
    _addResourceDependency(target) {
        this.dependsOn.add(target);
    }
    addRosDependency(target) {
        // target.forEach(x => this.rosDependsOn.add(x));
        this.rosDependsOn.add(target);
    }
    /**
     * Emits ROS template for this resource.
     * @internal
     */
    _toRosTemplate() {
        try {
            const ret = {
                Resources: {
                    // Post-Resolve operation since otherwise deepMerge is going to mix values into
                    // the Token objects returned by ignoreEmpty.
                    [this.id]: new util_1.PostResolveToken({
                        Type: this.rosResourceType,
                        Properties: util_1.ignoreEmpty(this.rosProperties),
                        DependsOn: util_1.ignoreEmpty(renderRosDependsOn(this.rosDependsOn)),
                        // UpdatePolicy: capitalizePropertyNames(this, this.rosOptions.updatePolicy),
                        // UpdateReplacePolicy: capitalizePropertyNames(this, this.rosOptions.updateReplacePolicy),
                        DeletionPolicy: util_1.capitalizePropertyNames(this, this.rosOptions.deletionPolicy),
                        Metadata: util_1.ignoreEmpty(this.rosOptions.metadata),
                        Description: util_1.ignoreEmpty(this.rosOptions.description),
                        Condition: this.rosOptions.condition &&
                            this.rosOptions.condition.logicalId,
                    }, (props) => {
                        const renderedProps = this.renderProperties(props.Properties || {});
                        props.Properties =
                            renderedProps &&
                                (Object.values(renderedProps).find((v) => !!v)
                                    ? renderedProps
                                    : undefined);
                        return deepMerge(props, this.rawOverrides);
                    }),
                },
            };
            return ret;
        }
        catch (e) {
            // Change message
            e.message = `While synthesizing ${this.node.path}: ${e.message}`;
            // Adjust stack trace (make it look like node built it, too...)
            const trace = this.creationStack;
            if (trace) {
                const creationStack = ["--- resource created at ---", ...trace].join("\n  at ");
                const problemTrace = e.stack.substr(e.stack.indexOf(e.message) + e.message.length);
                e.stack = `${e.message}\n  ${creationStack}\n  --- problem discovered at ---${problemTrace}`;
            }
            // Re-throw
            throw e;
        }
        function renderRosDependsOn(rosDependsOn) {
            return Array.from(rosDependsOn);
        }
    }
    get rosProperties() {
        const props = this._rosProperties || {};
        if (tag_manager_1.TagManager.isTaggable(this)) {
            const tagsProp = {};
            tagsProp[this.tags.tagPropertyName] = this.tags.renderTags();
            return deepMerge(props, tagsProp);
        }
        return props;
    }
    renderProperties(props) {
        return props;
    }
    /**
     * Return properties modified after initiation
     *
     * Resources that expose mutable properties should override this function to
     * collect and return the properties object for this resource.
     */
    get updatedProperites() {
        return this._rosProperties;
    }
    validateProperties(_properties) {
        // Nothing
    }
}
exports.RosResource = RosResource;
/**
 * Merges `source` into `target`, overriding any existing values.
 * `null`s will cause a value to be deleted.
 */
function deepMerge(target, ...sources) {
    for (const source of sources) {
        if (typeof source !== "object" || typeof target !== "object") {
            throw new Error(`Invalid usage. Both source (${JSON.stringify(source)}) and target (${JSON.stringify(target)}) must be objects`);
        }
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (typeof value === "object" && value != null && !Array.isArray(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = {};
                }
                deepMerge(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof output === "object" && Object.keys(output).length === 0) {
                    delete target[key];
                }
            }
            else if (value === undefined) {
                delete target[key];
            }
            else {
                target[key] = value;
            }
        }
    }
    return target;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicm9zLXJlc291cmNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicm9zLXJlc291cmNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtEQUEwRDtBQUMxRCxpQ0FBZ0Y7QUFFaEYsK0NBQTJDO0FBRTNDLGlDQUF1QztBQUN2QywyREFBdUQ7QUFDdkQsNkRBQTJFO0FBRTNFLCtDQUE4QztBQWU5Qzs7R0FFRztBQUNILE1BQWEsV0FBWSxTQUFRLDJCQUFhO0lBOEM1Qzs7O09BR0c7SUFDSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCO1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUEzQ25CLDZFQUE2RTtRQUM3RSw2RUFBNkU7UUFDN0Usa0VBQWtFO1FBQ2xFLDhFQUE4RTtRQUM5RSw4RUFBOEU7UUFDOUUsMEVBQTBFO1FBQzFFLCtDQUErQztRQUUvQzs7V0FFRztRQUNhLGVBQVUsR0FBd0IsRUFBRSxDQUFDO1FBYXJEOztXQUVHO1FBQ2MsaUJBQVksR0FBUSxFQUFFLENBQUM7UUFFeEM7Ozs7V0FJRztRQUNjLGNBQVMsR0FBRyxJQUFJLEdBQUcsRUFBZSxDQUFDO1FBRW5DLGlCQUFZLEdBQUcsSUFBSSxHQUFHLEVBQVUsQ0FBQztRQVFoRCxJQUFJLENBQUMsRUFBRSxHQUFHLEVBQUUsQ0FBQztRQUNiLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFO1lBQ2YsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1NBQ3BEO1FBRUQsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDO1FBQ2xDLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUM7SUFDL0MsQ0FBQztJQTFERDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxhQUFhLENBQUMsU0FBcUI7UUFDL0MsT0FBUSxTQUFpQixDQUFDLGVBQWUsS0FBSyxTQUFTLENBQUM7SUFDMUQsQ0FBQztJQXVERDs7T0FFRztJQUNJLGtCQUFrQixDQUN2QixNQUFpQyxFQUNqQyxVQUFnQyxFQUFFO1FBRWxDLE1BQU0sR0FBRyxNQUFNLElBQUksT0FBTyxDQUFDLGFBQWEsSUFBSSxrQ0FBYSxDQUFDLE1BQU0sQ0FBQztRQUVqRSxJQUFJLGNBQWMsQ0FBQztRQUVuQixRQUFRLE1BQU0sRUFBRTtZQUNkLEtBQUssa0NBQWEsQ0FBQyxPQUFPO2dCQUN4QixjQUFjLEdBQUcsdUNBQWlCLENBQUMsTUFBTSxDQUFDO2dCQUMxQyxNQUFNO1lBRVIsS0FBSyxrQ0FBYSxDQUFDLE1BQU07Z0JBQ3ZCLGNBQWMsR0FBRyx1Q0FBaUIsQ0FBQyxNQUFNLENBQUM7Z0JBQzFDLE1BQU07WUFFUjtnQkFDRSxNQUFNLElBQUksS0FBSyxDQUFDLDJCQUEyQixNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQ3hEO1FBRUQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxjQUFjLEdBQUcsY0FBYyxDQUFDO1FBQ2hELHNEQUFzRDtRQUN0RCwwREFBMEQ7UUFDMUQsSUFBSTtJQUNOLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxhQUFxQjtRQUNqQyxPQUFPLDRCQUFZLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxhQUFhLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FxQ0c7SUFDSSxXQUFXLENBQUMsSUFBWSxFQUFFLEtBQVU7UUFDekMsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUM5QixJQUFJLElBQUksR0FBUSxJQUFJLENBQUMsWUFBWSxDQUFDO1FBRWxDLE9BQU8sS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDdkIsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLEtBQUssRUFBRyxDQUFDO1lBRTNCLDhEQUE4RDtZQUM5RCxzQ0FBc0M7WUFDdEMsTUFBTSxRQUFRLEdBQ1osSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLElBQUk7Z0JBQ2pCLE9BQU8sSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLFFBQVE7Z0JBQzdCLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUM1QixJQUFJLENBQUMsUUFBUSxFQUFFO2dCQUNiLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUM7YUFDaEI7WUFFRCxJQUFJLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ2xCO1FBRUQsTUFBTSxPQUFPLEdBQUcsS0FBSyxDQUFDLEtBQUssRUFBRyxDQUFDO1FBQy9CLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxLQUFLLENBQUM7SUFDeEIsQ0FBQztJQUVEOzs7T0FHRztJQUNJLG1CQUFtQixDQUFDLElBQVk7UUFDckMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVNLFdBQVcsQ0FBQyxHQUFXLEVBQUUsS0FBVTtRQUN4QyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUU7WUFDN0IsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLEdBQUcsRUFBRSxDQUFDO1NBQy9CO1FBQ0QsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO0lBQ3hDLENBQUM7SUFFTSxPQUFPLENBQUMsSUFBWTtRQUN6QixJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUM7SUFDckMsQ0FBQztJQUVNLFlBQVksQ0FBQyxHQUFpQjtRQUNuQyxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsR0FBRyxHQUFHLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxtQkFBbUIsQ0FBQyxZQUFvQixFQUFFLEtBQVU7UUFDekQsSUFBSSxDQUFDLFdBQVcsQ0FBQyxjQUFjLFlBQVksRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7O09BR0c7SUFDSSwyQkFBMkIsQ0FBQyxZQUFvQjtRQUNyRCxJQUFJLENBQUMsbUJBQW1CLENBQUMsWUFBWSxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZLENBQUMsTUFBbUI7UUFDckMsb0JBQWEsQ0FDWCxJQUFJLEVBQ0osTUFBTSxFQUNOLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLGlCQUFpQixNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUN2RCxDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0ksUUFBUTtRQUNiLE9BQU8sR0FBRyxLQUFLLENBQUMsUUFBUSxFQUFFLEtBQUssSUFBSSxDQUFDLGVBQWUsR0FBRyxDQUFDO0lBQ3pELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQkFBc0IsQ0FBQyxNQUFtQjtRQUMvQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUM3QixDQUFDO0lBRU0sZ0JBQWdCLENBQUMsTUFBYztRQUNwQyxpREFBaUQ7UUFDakQsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGNBQWM7UUFDbkIsSUFBSTtZQUNGLE1BQU0sR0FBRyxHQUFHO2dCQUNWLFNBQVMsRUFBRTtvQkFDVCwrRUFBK0U7b0JBQy9FLDZDQUE2QztvQkFDN0MsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLEVBQUUsSUFBSSx1QkFBZ0IsQ0FDN0I7d0JBQ0UsSUFBSSxFQUFFLElBQUksQ0FBQyxlQUFlO3dCQUMxQixVQUFVLEVBQUUsa0JBQVcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO3dCQUMzQyxTQUFTLEVBQUUsa0JBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7d0JBQzdELDZFQUE2RTt3QkFDN0UsMkZBQTJGO3dCQUMzRixjQUFjLEVBQUUsOEJBQXVCLENBQ3JDLElBQUksRUFDSixJQUFJLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FDL0I7d0JBQ0QsUUFBUSxFQUFFLGtCQUFXLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUM7d0JBQy9DLFdBQVcsRUFBRSxrQkFBVyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDO3dCQUNyRCxTQUFTLEVBQ1AsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTOzRCQUN6QixJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxTQUFTO3FCQUN0QyxFQUNELENBQUMsS0FBSyxFQUFFLEVBQUU7d0JBQ1IsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUN6QyxLQUFLLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FDdkIsQ0FBQzt3QkFDRixLQUFLLENBQUMsVUFBVTs0QkFDZCxhQUFhO2dDQUNiLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7b0NBQzVDLENBQUMsQ0FBQyxhQUFhO29DQUNmLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQzt3QkFDakIsT0FBTyxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztvQkFDN0MsQ0FBQyxDQUNGO2lCQUNGO2FBQ0YsQ0FBQztZQUNGLE9BQU8sR0FBRyxDQUFDO1NBQ1o7UUFBQyxPQUFPLENBQUMsRUFBRTtZQUNWLGlCQUFpQjtZQUNqQixDQUFDLENBQUMsT0FBTyxHQUFHLHNCQUFzQixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDakUsK0RBQStEO1lBQy9ELE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUM7WUFDakMsSUFBSSxLQUFLLEVBQUU7Z0JBQ1QsTUFBTSxhQUFhLEdBQUcsQ0FBQyw2QkFBNkIsRUFBRSxHQUFHLEtBQUssQ0FBQyxDQUFDLElBQUksQ0FDbEUsU0FBUyxDQUNWLENBQUM7Z0JBQ0YsTUFBTSxZQUFZLEdBQUcsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQ2pDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FDOUMsQ0FBQztnQkFDRixDQUFDLENBQUMsS0FBSyxHQUFHLEdBQUcsQ0FBQyxDQUFDLE9BQU8sT0FBTyxhQUFhLG9DQUFvQyxZQUFZLEVBQUUsQ0FBQzthQUM5RjtZQUVELFdBQVc7WUFDWCxNQUFNLENBQUMsQ0FBQztTQUNUO1FBRUQsU0FBUyxrQkFBa0IsQ0FBQyxZQUF5QjtZQUNuRCxPQUFPLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDbEMsQ0FBQztJQUNILENBQUM7SUFFRCxJQUFjLGFBQWE7UUFDekIsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLGNBQWMsSUFBSSxFQUFFLENBQUM7UUFDeEMsSUFBSSx3QkFBVSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUMvQixNQUFNLFFBQVEsR0FBMkIsRUFBRSxDQUFDO1lBQzVDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDN0QsT0FBTyxTQUFTLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1NBQ25DO1FBQ0QsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRVMsZ0JBQWdCLENBQUMsS0FFMUI7UUFDQyxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNILElBQWMsaUJBQWlCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQztJQUM3QixDQUFDO0lBRVMsa0JBQWtCLENBQUMsV0FBZ0I7UUFDM0MsVUFBVTtJQUNaLENBQUM7Q0FDRjtBQXRWRCxrQ0FzVkM7QUEyQkQ7OztHQUdHO0FBQ0gsU0FBUyxTQUFTLENBQUMsTUFBVyxFQUFFLEdBQUcsT0FBYztJQUMvQyxLQUFLLE1BQU0sTUFBTSxJQUFJLE9BQU8sRUFBRTtRQUM1QixJQUFJLE9BQU8sTUFBTSxLQUFLLFFBQVEsSUFBSSxPQUFPLE1BQU0sS0FBSyxRQUFRLEVBQUU7WUFDNUQsTUFBTSxJQUFJLEtBQUssQ0FDYiwrQkFBK0IsSUFBSSxDQUFDLFNBQVMsQ0FDM0MsTUFBTSxDQUNQLGlCQUFpQixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxtQkFBbUIsQ0FDNUQsQ0FBQztTQUNIO1FBRUQsS0FBSyxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3JDLE1BQU0sS0FBSyxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUMxQixJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsSUFBSSxLQUFLLElBQUksSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDdkUsbUVBQW1FO2dCQUNuRSwwQ0FBMEM7Z0JBQzFDLElBQUksT0FBTyxNQUFNLENBQUMsR0FBRyxDQUFDLEtBQUssUUFBUSxFQUFFO29CQUNuQyxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsRUFBRSxDQUFDO2lCQUNsQjtnQkFFRCxTQUFTLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO2dCQUU5QixrRUFBa0U7Z0JBQ2xFLDhEQUE4RDtnQkFDOUQsaUVBQWlFO2dCQUNqRSxNQUFNLE1BQU0sR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQzNCLElBQUksT0FBTyxNQUFNLEtBQUssUUFBUSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtvQkFDbEUsT0FBTyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7aUJBQ3BCO2FBQ0Y7aUJBQU0sSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO2dCQUM5QixPQUFPLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUNwQjtpQkFBTTtnQkFDTCxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO2FBQ3JCO1NBQ0Y7S0FDRjtJQUVELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBSb3NEZWxldGlvblBvbGljeSB9IGZyb20gXCIuL3Jvcy1yZXNvdXJjZS1wb2xpY3lcIjtcclxuaW1wb3J0IHsgY2FwaXRhbGl6ZVByb3BlcnR5TmFtZXMsIGlnbm9yZUVtcHR5LCBQb3N0UmVzb2x2ZVRva2VuIH0gZnJvbSBcIi4vdXRpbFwiO1xyXG5pbXBvcnQgeyBDb25zdHJ1Y3QsIElDb25zdHJ1Y3QgfSBmcm9tIFwiLi9jb25zdHJ1Y3QtY29tcGF0XCI7XHJcbmltcG9ydCB7IFRhZ01hbmFnZXIgfSBmcm9tIFwiLi90YWctbWFuYWdlclwiO1xyXG5pbXBvcnQgeyBSZWZlcmVuY2UgfSBmcm9tIFwiLi9yZWZlcmVuY2VcIjtcclxuaW1wb3J0IHsgYWRkRGVwZW5kZW5jeSB9IGZyb20gXCIuL2RlcHNcIjtcclxuaW1wb3J0IHsgUm9zUmVmZXJlbmNlIH0gZnJvbSBcIi4vcHJpdmF0ZS9yb3MtcmVmZXJlbmNlXCI7XHJcbmltcG9ydCB7IFJlbW92YWxQb2xpY3ksIFJlbW92YWxQb2xpY3lPcHRpb25zIH0gZnJvbSBcIi4vcm9zLXJlbW92YWwtcG9saWN5XCI7XHJcbmltcG9ydCB7IFJvc0NvbmRpdGlvbiB9IGZyb20gXCIuL3Jvcy1jb25kaXRpb25cIjtcclxuaW1wb3J0IHsgUm9zUmVmRWxlbWVudCB9IGZyb20gXCIuL3Jvcy1lbGVtZW50XCI7XHJcbmV4cG9ydCBpbnRlcmZhY2UgUm9zUmVzb3VyY2VQcm9wcyB7XHJcbiAgLyoqXHJcbiAgICogUk9TIHRlbXBsYXRlIHJlc291cmNlIHR5cGUgKGUuZy4gYEFMSVlVTjo6RUNTOjpJbnN0YW5jZWApLlxyXG4gICAqL1xyXG4gIHJlYWRvbmx5IHR5cGU6IHN0cmluZztcclxuXHJcbiAgLyoqXHJcbiAgICogUmVzb3VyY2UgcHJvcGVydGllcy5cclxuICAgKlxyXG4gICAqIEBkZWZhdWx0IC0gTm8gcmVzb3VyY2UgcHJvcGVydGllcy5cclxuICAgKi9cclxuICByZWFkb25seSBwcm9wZXJ0aWVzPzogeyBbbmFtZTogc3RyaW5nXTogYW55IH07XHJcbn1cclxuXHJcbi8qKlxyXG4gKiBSZXByZXNlbnRzIGEgUk9TIHJlc291cmNlLlxyXG4gKi9cclxuZXhwb3J0IGNsYXNzIFJvc1Jlc291cmNlIGV4dGVuZHMgUm9zUmVmRWxlbWVudCB7XHJcbiAgLyoqXHJcbiAgICogQ2hlY2sgd2hldGhlciB0aGUgZ2l2ZW4gY29uc3RydWN0IGlzIGEgUm9zUmVzb3VyY2VcclxuICAgKi9cclxuICBwdWJsaWMgc3RhdGljIGlzUm9zUmVzb3VyY2UoY29uc3RydWN0OiBJQ29uc3RydWN0KTogY29uc3RydWN0IGlzIFJvc1Jlc291cmNlIHtcclxuICAgIHJldHVybiAoY29uc3RydWN0IGFzIGFueSkucm9zUmVzb3VyY2VUeXBlICE9PSB1bmRlZmluZWQ7XHJcbiAgfVxyXG5cclxuICAvLyBNQUlOVEFJTkVSUyBOT1RFOiB0aGlzIGNsYXNzIHNlcnZlcyBhcyB0aGUgYmFzZSBjbGFzcyBmb3IgdGhlIGdlbmVyYXRlZCBMMVxyXG4gIC8vIChcIkFMSVlVTlwiKSByZXNvdXJjZXMgKHN1Y2ggYXMgYGVjcy5JbnN0YW5jZWApLiBUaGVzZSByZXNvdXJjZXMgd2lsbCBoYXZlIGFcclxuICAvLyBwcm9wZXJ0eSBmb3IgZWFjaCBST1MgcHJvcGVydHkgb2YgdGhlIHJlc291cmNlLiBUaGlzIG1lYW5zIHRoYXRcclxuICAvLyBpZiBhdCBzb21lIHBvaW50IGluIHRoZSBmdXR1cmUgYSBwcm9wZXJ0eSBpcyBpbnRyb2R1Y2VkIHdpdGggYSBuYW1lIHNpbWlsYXJcclxuICAvLyB0byBvbmUgb2YgdGhlIHByb3BlcnRpZXMgaGVyZSwgaXQgd2lsbCBiZSBcIm1hc2tlZFwiIGJ5IHRoZSBkZXJpdmVkIGNsYXNzLiBUb1xyXG4gIC8vIHRoYXQgZW5kLCB3ZSBwcmVmaXggYWxsIHByb3BlcnRpZXMgaW4gdGhpcyBjbGFzcyB3aXRoIGByb3NYeHhgIHdpdGggdGhlXHJcbiAgLy8gaG9wZSB0byBhdm9pZCB0aG9zZSBjb25mbGljdHMgaW4gdGhlIGZ1dHVyZS5cclxuXHJcbiAgLyoqXHJcbiAgICogT3B0aW9ucyBmb3IgdGhpcyByZXNvdXJjZSwgc3VjaCBhcyBjb25kaXRpb24sIHVwZGF0ZSBwb2xpY3kgZXRjLlxyXG4gICAqL1xyXG4gIHB1YmxpYyByZWFkb25seSByb3NPcHRpb25zOiBJUm9zUmVzb3VyY2VPcHRpb25zID0ge307XHJcblxyXG4gIC8qKlxyXG4gICAqIFJPUyByZXNvdXJjZSB0eXBlLlxyXG4gICAqL1xyXG4gIHB1YmxpYyByZWFkb25seSByb3NSZXNvdXJjZVR5cGU6IHN0cmluZztcclxuXHJcbiAgLyoqXHJcbiAgICogQWxpeXVuIHJlc291cmNlIHByb3BlcnRpZXMuXHJcbiAgICogQGludGVybmFsXHJcbiAgICovXHJcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IF9yb3NQcm9wZXJ0aWVzOiBhbnk7XHJcblxyXG4gIC8qKlxyXG4gICAqIEFuIG9iamVjdCB0byBiZSBtZXJnZWQgb24gdG9wIG9mIHRoZSBlbnRpcmUgcmVzb3VyY2UgZGVmaW5pdGlvbi5cclxuICAgKi9cclxuICBwcml2YXRlIHJlYWRvbmx5IHJhd092ZXJyaWRlczogYW55ID0ge307XHJcblxyXG4gIC8qKlxyXG4gICAqIExvZ2ljYWwgSURzIG9mIGRlcGVuZGVuY2llcy5cclxuICAgKlxyXG4gICAqIElzIGZpbGxlZCBkdXJpbmcgcHJlcGFyZSgpLlxyXG4gICAqL1xyXG4gIHByaXZhdGUgcmVhZG9ubHkgZGVwZW5kc09uID0gbmV3IFNldDxSb3NSZXNvdXJjZT4oKTtcclxuICBwcml2YXRlIHJlYWRvbmx5IGlkOiBzdHJpbmc7XHJcbiAgcHJpdmF0ZSByZWFkb25seSByb3NEZXBlbmRzT24gPSBuZXcgU2V0PHN0cmluZz4oKTtcclxuXHJcbiAgLyoqXHJcbiAgICogQ3JlYXRlcyBhIHJlc291cmNlIGNvbnN0cnVjdC5cclxuICAgKiBAcGFyYW0gcm9zUmVzb3VyY2VUeXBlIFRoZSBST1MgdHlwZSBvZiB0aGlzIHJlc291cmNlIChlLmcuIEFMSVlVTjo6RUNTOjpJbnN0YW5jZSlcclxuICAgKi9cclxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUm9zUmVzb3VyY2VQcm9wcykge1xyXG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcclxuICAgIHRoaXMuaWQgPSBpZDtcclxuICAgIGlmICghcHJvcHMudHlwZSkge1xyXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJUaGUgYHR5cGVgIHByb3BlcnR5IGlzIHJlcXVpcmVkXCIpO1xyXG4gICAgfVxyXG5cclxuICAgIHRoaXMucm9zUmVzb3VyY2VUeXBlID0gcHJvcHMudHlwZTtcclxuICAgIHRoaXMuX3Jvc1Byb3BlcnRpZXMgPSBwcm9wcy5wcm9wZXJ0aWVzIHx8IHt9O1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogU2V0cyB0aGUgZGVsZXRpb24gcG9saWN5IG9mIHRoZSByZXNvdXJjZSBiYXNlZCBvbiB0aGUgcmVtb3ZhbCBwb2xpY3kgc3BlY2lmaWVkLlxyXG4gICAqL1xyXG4gIHB1YmxpYyBhcHBseVJlbW92YWxQb2xpY3koXHJcbiAgICBwb2xpY3k6IFJlbW92YWxQb2xpY3kgfCB1bmRlZmluZWQsXHJcbiAgICBvcHRpb25zOiBSZW1vdmFsUG9saWN5T3B0aW9ucyA9IHt9XHJcbiAgKSB7XHJcbiAgICBwb2xpY3kgPSBwb2xpY3kgfHwgb3B0aW9ucy5kZWZhdWx0UG9saWN5IHx8IFJlbW92YWxQb2xpY3kuUkVUQUlOO1xyXG5cclxuICAgIGxldCBkZWxldGlvblBvbGljeTtcclxuXHJcbiAgICBzd2l0Y2ggKHBvbGljeSkge1xyXG4gICAgICBjYXNlIFJlbW92YWxQb2xpY3kuREVTVFJPWTpcclxuICAgICAgICBkZWxldGlvblBvbGljeSA9IFJvc0RlbGV0aW9uUG9saWN5LkRFTEVURTtcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgUmVtb3ZhbFBvbGljeS5SRVRBSU46XHJcbiAgICAgICAgZGVsZXRpb25Qb2xpY3kgPSBSb3NEZWxldGlvblBvbGljeS5SRVRBSU47XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBkZWZhdWx0OlxyXG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCByZW1vdmFsIHBvbGljeTogJHtwb2xpY3l9YCk7XHJcbiAgICB9XHJcblxyXG4gICAgdGhpcy5yb3NPcHRpb25zLmRlbGV0aW9uUG9saWN5ID0gZGVsZXRpb25Qb2xpY3k7XHJcbiAgICAvLyBpZiAob3B0aW9ucy5hcHBseVRvVXBkYXRlUmVwbGFjZVBvbGljeSAhPT0gZmFsc2UpIHtcclxuICAgIC8vICAgdGhpcy5yb3NPcHRpb25zLnVwZGF0ZVJlcGxhY2VQb2xpY3kgPSBkZWxldGlvblBvbGljeTtcclxuICAgIC8vIH1cclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIFJldHVybnMgYSB0b2tlbiBmb3IgYW4gcnVudGltZSBhdHRyaWJ1dGUgb2YgdGhpcyByZXNvdXJjZS5cclxuICAgKiBJZGVhbGx5LCB1c2UgZ2VuZXJhdGVkIGF0dHJpYnV0ZSBhY2Nlc3NvcnMgKGUuZy4gYHJlc291cmNlLmFybmApLCBidXQgdGhpcyBjYW4gYmUgdXNlZCBmb3IgZnV0dXJlIGNvbXBhdGliaWxpdHlcclxuICAgKiBpbiBjYXNlIHRoZXJlIGlzIG5vIGdlbmVyYXRlZCBhdHRyaWJ1dGUuXHJcbiAgICogQHBhcmFtIGF0dHJpYnV0ZU5hbWUgVGhlIG5hbWUgb2YgdGhlIGF0dHJpYnV0ZS5cclxuICAgKi9cclxuICBwdWJsaWMgZ2V0QXR0KGF0dHJpYnV0ZU5hbWU6IHN0cmluZyk6IFJlZmVyZW5jZSB7XHJcbiAgICByZXR1cm4gUm9zUmVmZXJlbmNlLmZvcih0aGlzLCBhdHRyaWJ1dGVOYW1lKTtcclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIEFkZHMgYW4gb3ZlcnJpZGUgdG8gdGhlIHN5bnRoZXNpemVkIFJPUyByZXNvdXJjZS4gVG8gYWRkIGFcclxuICAgKiBwcm9wZXJ0eSBvdmVycmlkZSwgZWl0aGVyIHVzZSBgYWRkUHJvcGVydHlPdmVycmlkZWAgb3IgcHJlZml4IGBwYXRoYCB3aXRoXHJcbiAgICogXCJQcm9wZXJ0aWVzLlwiIChpLmUuIGBQcm9wZXJ0aWVzLlRvcGljTmFtZWApLlxyXG4gICAqXHJcbiAgICogSWYgdGhlIG92ZXJyaWRlIGlzIG5lc3RlZCwgc2VwYXJhdGUgZWFjaCBuZXN0ZWQgbGV2ZWwgdXNpbmcgYSBkb3QgKC4pIGluIHRoZSBwYXRoIHBhcmFtZXRlci5cclxuICAgKiBJZiB0aGVyZSBpcyBhbiBhcnJheSBhcyBwYXJ0IG9mIHRoZSBuZXN0aW5nLCBzcGVjaWZ5IHRoZSBpbmRleCBpbiB0aGUgcGF0aC5cclxuICAgKlxyXG4gICAqIEZvciBleGFtcGxlLFxyXG4gICAqIGBgYHR5cGVzY3JpcHRcclxuICAgKiBhZGRPdmVycmlkZSgnUHJvcGVydGllcy5HbG9iYWxTZWNvbmRhcnlJbmRleGVzLjAuUHJvamVjdGlvbi5Ob25LZXlBdHRyaWJ1dGVzJywgWydteWF0dHJpYnV0ZSddKVxyXG4gICAqIGFkZE92ZXJyaWRlKCdQcm9wZXJ0aWVzLkdsb2JhbFNlY29uZGFyeUluZGV4ZXMuMS5Qcm9qZWN0aW9uVHlwZScsICdJTkNMVURFJylcclxuICAgKiBgYGBcclxuICAgKiB3b3VsZCBhZGQgdGhlIG92ZXJyaWRlc1xyXG4gICAqIGBgYGpzb25cclxuICAgKiBcIlByb3BlcnRpZXNcIjoge1xyXG4gICAqICAgXCJHbG9iYWxTZWNvbmRhcnlJbmRleGVzXCI6IFtcclxuICAgKiAgICAge1xyXG4gICAqICAgICAgIFwiUHJvamVjdGlvblwiOiB7XHJcbiAgICogICAgICAgICBcIk5vbktleUF0dHJpYnV0ZXNcIjogWyBcIm15YXR0cmlidXRlXCIgXVxyXG4gICAqICAgICAgICAgLi4uXHJcbiAgICogICAgICAgfVxyXG4gICAqICAgICAgIC4uLlxyXG4gICAqICAgICB9LFxyXG4gICAqICAgICB7XHJcbiAgICogICAgICAgXCJQcm9qZWN0aW9uVHlwZVwiOiBcIklOQ0xVREVcIlxyXG4gICAqICAgICAgIC4uLlxyXG4gICAqICAgICB9LFxyXG4gICAqICAgXVxyXG4gICAqICAgLi4uXHJcbiAgICogfVxyXG4gICAqIGBgYFxyXG4gICAqXHJcbiAgICogQHBhcmFtIHBhdGggLSBUaGUgcGF0aCBvZiB0aGUgcHJvcGVydHksIHlvdSBjYW4gdXNlIGRvdCBub3RhdGlvbiB0b1xyXG4gICAqICAgICAgICBvdmVycmlkZSB2YWx1ZXMgaW4gY29tcGxleCB0eXBlcy4gQW55IGludGVybWRlZGlhdGUga2V5c1xyXG4gICAqICAgICAgICB3aWxsIGJlIGNyZWF0ZWQgYXMgbmVlZGVkLlxyXG4gICAqIEBwYXJhbSB2YWx1ZSAtIFRoZSB2YWx1ZS4gQ291bGQgYmUgcHJpbWl0aXZlIG9yIGNvbXBsZXguXHJcbiAgICovXHJcbiAgcHVibGljIGFkZE92ZXJyaWRlKHBhdGg6IHN0cmluZywgdmFsdWU6IGFueSkge1xyXG4gICAgY29uc3QgcGFydHMgPSBwYXRoLnNwbGl0KFwiLlwiKTtcclxuICAgIGxldCBjdXJyOiBhbnkgPSB0aGlzLnJhd092ZXJyaWRlcztcclxuXHJcbiAgICB3aGlsZSAocGFydHMubGVuZ3RoID4gMSkge1xyXG4gICAgICBjb25zdCBrZXkgPSBwYXJ0cy5zaGlmdCgpITtcclxuXHJcbiAgICAgIC8vIGlmIHdlIGNhbid0IHJlY3Vyc2UgZnVydGhlciBvciB0aGUgcHJldmlvdXMgdmFsdWUgaXMgbm90IGFuXHJcbiAgICAgIC8vIG9iamVjdCBvdmVyd3JpdGUgaXQgd2l0aCBhbiBvYmplY3QuXHJcbiAgICAgIGNvbnN0IGlzT2JqZWN0ID1cclxuICAgICAgICBjdXJyW2tleV0gIT0gbnVsbCAmJlxyXG4gICAgICAgIHR5cGVvZiBjdXJyW2tleV0gPT09IFwib2JqZWN0XCIgJiZcclxuICAgICAgICAhQXJyYXkuaXNBcnJheShjdXJyW2tleV0pO1xyXG4gICAgICBpZiAoIWlzT2JqZWN0KSB7XHJcbiAgICAgICAgY3VycltrZXldID0ge307XHJcbiAgICAgIH1cclxuXHJcbiAgICAgIGN1cnIgPSBjdXJyW2tleV07XHJcbiAgICB9XHJcblxyXG4gICAgY29uc3QgbGFzdEtleSA9IHBhcnRzLnNoaWZ0KCkhO1xyXG4gICAgY3VycltsYXN0S2V5XSA9IHZhbHVlO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogU3ludGFjdGljIHN1Z2FyIGZvciBgYWRkT3ZlcnJpZGUocGF0aCwgdW5kZWZpbmVkKWAuXHJcbiAgICogQHBhcmFtIHBhdGggVGhlIHBhdGggb2YgdGhlIHZhbHVlIHRvIGRlbGV0ZVxyXG4gICAqL1xyXG4gIHB1YmxpYyBhZGREZWxldGlvbk92ZXJyaWRlKHBhdGg6IHN0cmluZykge1xyXG4gICAgdGhpcy5hZGRPdmVycmlkZShwYXRoLCB1bmRlZmluZWQpO1xyXG4gIH1cclxuXHJcbiAgcHVibGljIGFkZE1ldGFEYXRhKGtleTogc3RyaW5nLCB2YWx1ZTogYW55KSB7XHJcbiAgICBpZiAoIXRoaXMucm9zT3B0aW9ucy5tZXRhZGF0YSkge1xyXG4gICAgICB0aGlzLnJvc09wdGlvbnMubWV0YWRhdGEgPSB7fTtcclxuICAgIH1cclxuICAgIHRoaXMucm9zT3B0aW9ucy5tZXRhZGF0YVtrZXldID0gdmFsdWU7XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgYWRkRGVzYyhkZXNjOiBzdHJpbmcpIHtcclxuICAgIHRoaXMucm9zT3B0aW9ucy5kZXNjcmlwdGlvbiA9IGRlc2M7XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgYWRkQ29uZGl0aW9uKGNvbjogUm9zQ29uZGl0aW9uKSB7XHJcbiAgICB0aGlzLnJvc09wdGlvbnMuY29uZGl0aW9uID0gY29uO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogQWRkcyBhbiBvdmVycmlkZSB0byBhIHJlc291cmNlIHByb3BlcnR5LlxyXG4gICAqXHJcbiAgICogU3ludGFjdGljIHN1Z2FyIGZvciBgYWRkT3ZlcnJpZGUoXCJQcm9wZXJ0aWVzLjwuLi4+XCIsIHZhbHVlKWAuXHJcbiAgICpcclxuICAgKiBAcGFyYW0gcHJvcGVydHlQYXRoIFRoZSBwYXRoIG9mIHRoZSBwcm9wZXJ0eVxyXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWVcclxuICAgKi9cclxuICBwdWJsaWMgYWRkUHJvcGVydHlPdmVycmlkZShwcm9wZXJ0eVBhdGg6IHN0cmluZywgdmFsdWU6IGFueSkge1xyXG4gICAgdGhpcy5hZGRPdmVycmlkZShgUHJvcGVydGllcy4ke3Byb3BlcnR5UGF0aH1gLCB2YWx1ZSk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBBZGRzIGFuIG92ZXJyaWRlIHRoYXQgZGVsZXRlcyB0aGUgdmFsdWUgb2YgYSBwcm9wZXJ0eSBmcm9tIHRoZSByZXNvdXJjZSBkZWZpbml0aW9uLlxyXG4gICAqIEBwYXJhbSBwcm9wZXJ0eVBhdGggVGhlIHBhdGggdG8gdGhlIHByb3BlcnR5LlxyXG4gICAqL1xyXG4gIHB1YmxpYyBhZGRQcm9wZXJ0eURlbGV0aW9uT3ZlcnJpZGUocHJvcGVydHlQYXRoOiBzdHJpbmcpIHtcclxuICAgIHRoaXMuYWRkUHJvcGVydHlPdmVycmlkZShwcm9wZXJ0eVBhdGgsIHVuZGVmaW5lZCk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBJbmRpY2F0ZXMgdGhhdCB0aGlzIHJlc291cmNlIGRlcGVuZHMgb24gYW5vdGhlciByZXNvdXJjZSBhbmQgY2Fubm90IGJlXHJcbiAgICogcHJvdmlzaW9uZWQgdW5sZXNzIHRoZSBvdGhlciByZXNvdXJjZSBoYXMgYmVlbiBzdWNjZXNzZnVsbHkgcHJvdmlzaW9uZWQuXHJcbiAgICpcclxuICAgKiBUaGlzIGNhbiBiZSB1c2VkIGZvciByZXNvdXJjZXMgYWNyb3NzIHN0YWNrcyAob3IgbmVzdGVkIHN0YWNrKSBib3VuZGFyaWVzXHJcbiAgICogYW5kIHRoZSBkZXBlbmRlbmN5IHdpbGwgYXV0b21hdGljYWxseSBiZSB0cmFuc2ZlcnJlZCB0byB0aGUgcmVsZXZhbnQgc2NvcGUuXHJcbiAgICovXHJcbiAgcHVibGljIGFkZERlcGVuZHNPbih0YXJnZXQ6IFJvc1Jlc291cmNlKSB7XHJcbiAgICBhZGREZXBlbmRlbmN5KFxyXG4gICAgICB0aGlzLFxyXG4gICAgICB0YXJnZXQsXHJcbiAgICAgIGBcIiR7dGhpcy5ub2RlLnBhdGh9XCIgZGVwZW5kcyBvbiBcIiR7dGFyZ2V0Lm5vZGUucGF0aH1cImBcclxuICAgICk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBAcmV0dXJucyBhIHN0cmluZyByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHJlc291cmNlXHJcbiAgICovXHJcbiAgcHVibGljIHRvU3RyaW5nKCkge1xyXG4gICAgcmV0dXJuIGAke3N1cGVyLnRvU3RyaW5nKCl9IFske3RoaXMucm9zUmVzb3VyY2VUeXBlfV1gO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogQ2FsbGVkIGJ5IHRoZSBgYWRkRGVwZW5kZW5jeWAgaGVscGVyIGZ1bmN0aW9uIGluIG9yZGVyIHRvIHJlYWxpemUgYSBkaXJlY3RcclxuICAgKiBkZXBlbmRlbmN5IGJldHdlZW4gdHdvIHJlc291cmNlcyB0aGF0IGFyZSBkaXJlY3RseSBkZWZpbmVkIGluIHRoZSBzYW1lXHJcbiAgICogc3RhY2tzLlxyXG4gICAqXHJcbiAgICogVXNlIGByZXNvdXJjZS5hZGREZXBlbmRzT25gIHRvIGRlZmluZSB0aGUgZGVwZW5kZW5jeSBiZXR3ZWVuIHR3byByZXNvdXJjZXMsXHJcbiAgICogd2hpY2ggYWxzbyB0YWtlcyBzdGFjayBib3VuZGFyaWVzIGludG8gYWNjb3VudC5cclxuICAgKlxyXG4gICAqIEBpbnRlcm5hbFxyXG4gICAqL1xyXG4gIHB1YmxpYyBfYWRkUmVzb3VyY2VEZXBlbmRlbmN5KHRhcmdldDogUm9zUmVzb3VyY2UpIHtcclxuICAgIHRoaXMuZGVwZW5kc09uLmFkZCh0YXJnZXQpO1xyXG4gIH1cclxuXHJcbiAgcHVibGljIGFkZFJvc0RlcGVuZGVuY3kodGFyZ2V0OiBzdHJpbmcpIHtcclxuICAgIC8vIHRhcmdldC5mb3JFYWNoKHggPT4gdGhpcy5yb3NEZXBlbmRzT24uYWRkKHgpKTtcclxuICAgIHRoaXMucm9zRGVwZW5kc09uLmFkZCh0YXJnZXQpO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogRW1pdHMgUk9TIHRlbXBsYXRlIGZvciB0aGlzIHJlc291cmNlLlxyXG4gICAqIEBpbnRlcm5hbFxyXG4gICAqL1xyXG4gIHB1YmxpYyBfdG9Sb3NUZW1wbGF0ZSgpOiBvYmplY3Qge1xyXG4gICAgdHJ5IHtcclxuICAgICAgY29uc3QgcmV0ID0ge1xyXG4gICAgICAgIFJlc291cmNlczoge1xyXG4gICAgICAgICAgLy8gUG9zdC1SZXNvbHZlIG9wZXJhdGlvbiBzaW5jZSBvdGhlcndpc2UgZGVlcE1lcmdlIGlzIGdvaW5nIHRvIG1peCB2YWx1ZXMgaW50b1xyXG4gICAgICAgICAgLy8gdGhlIFRva2VuIG9iamVjdHMgcmV0dXJuZWQgYnkgaWdub3JlRW1wdHkuXHJcbiAgICAgICAgICBbdGhpcy5pZF06IG5ldyBQb3N0UmVzb2x2ZVRva2VuKFxyXG4gICAgICAgICAgICB7XHJcbiAgICAgICAgICAgICAgVHlwZTogdGhpcy5yb3NSZXNvdXJjZVR5cGUsXHJcbiAgICAgICAgICAgICAgUHJvcGVydGllczogaWdub3JlRW1wdHkodGhpcy5yb3NQcm9wZXJ0aWVzKSxcclxuICAgICAgICAgICAgICBEZXBlbmRzT246IGlnbm9yZUVtcHR5KHJlbmRlclJvc0RlcGVuZHNPbih0aGlzLnJvc0RlcGVuZHNPbikpLFxyXG4gICAgICAgICAgICAgIC8vIFVwZGF0ZVBvbGljeTogY2FwaXRhbGl6ZVByb3BlcnR5TmFtZXModGhpcywgdGhpcy5yb3NPcHRpb25zLnVwZGF0ZVBvbGljeSksXHJcbiAgICAgICAgICAgICAgLy8gVXBkYXRlUmVwbGFjZVBvbGljeTogY2FwaXRhbGl6ZVByb3BlcnR5TmFtZXModGhpcywgdGhpcy5yb3NPcHRpb25zLnVwZGF0ZVJlcGxhY2VQb2xpY3kpLFxyXG4gICAgICAgICAgICAgIERlbGV0aW9uUG9saWN5OiBjYXBpdGFsaXplUHJvcGVydHlOYW1lcyhcclxuICAgICAgICAgICAgICAgIHRoaXMsXHJcbiAgICAgICAgICAgICAgICB0aGlzLnJvc09wdGlvbnMuZGVsZXRpb25Qb2xpY3lcclxuICAgICAgICAgICAgICApLFxyXG4gICAgICAgICAgICAgIE1ldGFkYXRhOiBpZ25vcmVFbXB0eSh0aGlzLnJvc09wdGlvbnMubWV0YWRhdGEpLFxyXG4gICAgICAgICAgICAgIERlc2NyaXB0aW9uOiBpZ25vcmVFbXB0eSh0aGlzLnJvc09wdGlvbnMuZGVzY3JpcHRpb24pLFxyXG4gICAgICAgICAgICAgIENvbmRpdGlvbjpcclxuICAgICAgICAgICAgICAgIHRoaXMucm9zT3B0aW9ucy5jb25kaXRpb24gJiZcclxuICAgICAgICAgICAgICAgIHRoaXMucm9zT3B0aW9ucy5jb25kaXRpb24ubG9naWNhbElkLFxyXG4gICAgICAgICAgICB9LFxyXG4gICAgICAgICAgICAocHJvcHMpID0+IHtcclxuICAgICAgICAgICAgICBjb25zdCByZW5kZXJlZFByb3BzID0gdGhpcy5yZW5kZXJQcm9wZXJ0aWVzKFxyXG4gICAgICAgICAgICAgICAgcHJvcHMuUHJvcGVydGllcyB8fCB7fVxyXG4gICAgICAgICAgICAgICk7XHJcbiAgICAgICAgICAgICAgcHJvcHMuUHJvcGVydGllcyA9XHJcbiAgICAgICAgICAgICAgICByZW5kZXJlZFByb3BzICYmXHJcbiAgICAgICAgICAgICAgICAoT2JqZWN0LnZhbHVlcyhyZW5kZXJlZFByb3BzKS5maW5kKCh2KSA9PiAhIXYpXHJcbiAgICAgICAgICAgICAgICAgID8gcmVuZGVyZWRQcm9wc1xyXG4gICAgICAgICAgICAgICAgICA6IHVuZGVmaW5lZCk7XHJcbiAgICAgICAgICAgICAgcmV0dXJuIGRlZXBNZXJnZShwcm9wcywgdGhpcy5yYXdPdmVycmlkZXMpO1xyXG4gICAgICAgICAgICB9XHJcbiAgICAgICAgICApLFxyXG4gICAgICAgIH0sXHJcbiAgICAgIH07XHJcbiAgICAgIHJldHVybiByZXQ7XHJcbiAgICB9IGNhdGNoIChlKSB7XHJcbiAgICAgIC8vIENoYW5nZSBtZXNzYWdlXHJcbiAgICAgIGUubWVzc2FnZSA9IGBXaGlsZSBzeW50aGVzaXppbmcgJHt0aGlzLm5vZGUucGF0aH06ICR7ZS5tZXNzYWdlfWA7XHJcbiAgICAgIC8vIEFkanVzdCBzdGFjayB0cmFjZSAobWFrZSBpdCBsb29rIGxpa2Ugbm9kZSBidWlsdCBpdCwgdG9vLi4uKVxyXG4gICAgICBjb25zdCB0cmFjZSA9IHRoaXMuY3JlYXRpb25TdGFjaztcclxuICAgICAgaWYgKHRyYWNlKSB7XHJcbiAgICAgICAgY29uc3QgY3JlYXRpb25TdGFjayA9IFtcIi0tLSByZXNvdXJjZSBjcmVhdGVkIGF0IC0tLVwiLCAuLi50cmFjZV0uam9pbihcclxuICAgICAgICAgIFwiXFxuICBhdCBcIlxyXG4gICAgICAgICk7XHJcbiAgICAgICAgY29uc3QgcHJvYmxlbVRyYWNlID0gZS5zdGFjay5zdWJzdHIoXHJcbiAgICAgICAgICBlLnN0YWNrLmluZGV4T2YoZS5tZXNzYWdlKSArIGUubWVzc2FnZS5sZW5ndGhcclxuICAgICAgICApO1xyXG4gICAgICAgIGUuc3RhY2sgPSBgJHtlLm1lc3NhZ2V9XFxuICAke2NyZWF0aW9uU3RhY2t9XFxuICAtLS0gcHJvYmxlbSBkaXNjb3ZlcmVkIGF0IC0tLSR7cHJvYmxlbVRyYWNlfWA7XHJcbiAgICAgIH1cclxuXHJcbiAgICAgIC8vIFJlLXRocm93XHJcbiAgICAgIHRocm93IGU7XHJcbiAgICB9XHJcblxyXG4gICAgZnVuY3Rpb24gcmVuZGVyUm9zRGVwZW5kc09uKHJvc0RlcGVuZHNPbjogU2V0PHN0cmluZz4pIHtcclxuICAgICAgcmV0dXJuIEFycmF5LmZyb20ocm9zRGVwZW5kc09uKTtcclxuICAgIH1cclxuICB9XHJcblxyXG4gIHByb3RlY3RlZCBnZXQgcm9zUHJvcGVydGllcygpOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IHtcclxuICAgIGNvbnN0IHByb3BzID0gdGhpcy5fcm9zUHJvcGVydGllcyB8fCB7fTtcclxuICAgIGlmIChUYWdNYW5hZ2VyLmlzVGFnZ2FibGUodGhpcykpIHtcclxuICAgICAgY29uc3QgdGFnc1Byb3A6IHsgW2tleTogc3RyaW5nXTogYW55IH0gPSB7fTtcclxuICAgICAgdGFnc1Byb3BbdGhpcy50YWdzLnRhZ1Byb3BlcnR5TmFtZV0gPSB0aGlzLnRhZ3MucmVuZGVyVGFncygpO1xyXG4gICAgICByZXR1cm4gZGVlcE1lcmdlKHByb3BzLCB0YWdzUHJvcCk7XHJcbiAgICB9XHJcbiAgICByZXR1cm4gcHJvcHM7XHJcbiAgfVxyXG5cclxuICBwcm90ZWN0ZWQgcmVuZGVyUHJvcGVydGllcyhwcm9wczoge1xyXG4gICAgW2tleTogc3RyaW5nXTogYW55O1xyXG4gIH0pOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IHtcclxuICAgIHJldHVybiBwcm9wcztcclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIFJldHVybiBwcm9wZXJ0aWVzIG1vZGlmaWVkIGFmdGVyIGluaXRpYXRpb25cclxuICAgKlxyXG4gICAqIFJlc291cmNlcyB0aGF0IGV4cG9zZSBtdXRhYmxlIHByb3BlcnRpZXMgc2hvdWxkIG92ZXJyaWRlIHRoaXMgZnVuY3Rpb24gdG9cclxuICAgKiBjb2xsZWN0IGFuZCByZXR1cm4gdGhlIHByb3BlcnRpZXMgb2JqZWN0IGZvciB0aGlzIHJlc291cmNlLlxyXG4gICAqL1xyXG4gIHByb3RlY3RlZCBnZXQgdXBkYXRlZFByb3Blcml0ZXMoKTogeyBba2V5OiBzdHJpbmddOiBhbnkgfSB7XHJcbiAgICByZXR1cm4gdGhpcy5fcm9zUHJvcGVydGllcztcclxuICB9XHJcblxyXG4gIHByb3RlY3RlZCB2YWxpZGF0ZVByb3BlcnRpZXMoX3Byb3BlcnRpZXM6IGFueSkge1xyXG4gICAgLy8gTm90aGluZ1xyXG4gIH1cclxufVxyXG5cclxuZXhwb3J0IGludGVyZmFjZSBJUm9zUmVzb3VyY2VPcHRpb25zIHtcclxuICAvKipcclxuICAgKiBBIGNvbmRpdGlvbiB0byBhc3NvY2lhdGUgd2l0aCB0aGlzIHJlc291cmNlLiBUaGlzIG1lYW5zIHRoYXQgb25seSBpZiB0aGUgY29uZGl0aW9uIGV2YWx1YXRlcyB0byAndHJ1ZScgd2hlbiB0aGUgc3RhY2tcclxuICAgKiBpcyBkZXBsb3llZCwgdGhlIHJlc291cmNlIHdpbGwgYmUgaW5jbHVkZWQuIFRoaXMgaXMgcHJvdmlkZWQgdG8gYWxsb3cgQ0RLIHByb2plY3RzIHRvIHByb2R1Y2UgbGVnYWN5IHRlbXBsYXRlcywgYnV0IG5vcmFtbGx5XHJcbiAgICogdGhlcmUgaXMgbm8gbmVlZCB0byB1c2UgaXQgaW4gQ0RLIHByb2plY3RzLlxyXG4gICAqL1xyXG4gIGNvbmRpdGlvbj86IFJvc0NvbmRpdGlvbjtcclxuXHJcbiAgLyoqXHJcbiAgICogV2l0aCB0aGUgRGVsZXRpb25Qb2xpY3kgYXR0cmlidXRlIHlvdSBjYW4gcHJlc2VydmUgb3IgKGluIHNvbWUgY2FzZXMpIGJhY2t1cCBhIHJlc291cmNlIHdoZW4gaXRzIHN0YWNrIGlzIGRlbGV0ZWQuXHJcbiAgICogWW91IHNwZWNpZnkgYSBEZWxldGlvblBvbGljeSBhdHRyaWJ1dGUgZm9yIGVhY2ggcmVzb3VyY2UgdGhhdCB5b3Ugd2FudCB0byBjb250cm9sLiBJZiBhIHJlc291cmNlIGhhcyBubyBEZWxldGlvblBvbGljeVxyXG4gICAqIGF0dHJpYnV0ZSwgUk9TIGRlbGV0ZXMgdGhlIHJlc291cmNlIGJ5IGRlZmF1bHQuIE5vdGUgdGhhdCB0aGlzIGNhcGFiaWxpdHkgYWxzbyBhcHBsaWVzIHRvIHVwZGF0ZSBvcGVyYXRpb25zXHJcbiAgICogdGhhdCBsZWFkIHRvIHJlc291cmNlcyBiZWluZyByZW1vdmVkLlxyXG4gICAqL1xyXG4gIGRlbGV0aW9uUG9saWN5PzogUm9zRGVsZXRpb25Qb2xpY3k7XHJcblxyXG4gIC8qKlxyXG4gICAqIE1ldGFkYXRhIGFzc29jaWF0ZWQgd2l0aCB0aGUgUk9TIHJlc291cmNlLiBUaGlzIGlzIG5vdCB0aGUgc2FtZSBhcyB0aGUgY29uc3RydWN0IG1ldGFkYXRhIHdoaWNoIGNhbiBiZSBhZGRlZFxyXG4gICAqIHVzaW5nIGNvbnN0cnVjdC5hZGRNZXRhZGF0YSgpLCBidXQgd291bGQgbm90IGFwcGVhciBpbiB0aGUgUk9TIHRlbXBsYXRlIGF1dG9tYXRpY2FsbHkuXHJcbiAgICovXHJcbiAgbWV0YWRhdGE/OiB7IFtrZXk6IHN0cmluZ106IGFueSB9O1xyXG5cclxuICBkZXNjcmlwdGlvbj86IHN0cmluZztcclxufVxyXG5cclxuLyoqXHJcbiAqIE1lcmdlcyBgc291cmNlYCBpbnRvIGB0YXJnZXRgLCBvdmVycmlkaW5nIGFueSBleGlzdGluZyB2YWx1ZXMuXHJcbiAqIGBudWxsYHMgd2lsbCBjYXVzZSBhIHZhbHVlIHRvIGJlIGRlbGV0ZWQuXHJcbiAqL1xyXG5mdW5jdGlvbiBkZWVwTWVyZ2UodGFyZ2V0OiBhbnksIC4uLnNvdXJjZXM6IGFueVtdKSB7XHJcbiAgZm9yIChjb25zdCBzb3VyY2Ugb2Ygc291cmNlcykge1xyXG4gICAgaWYgKHR5cGVvZiBzb3VyY2UgIT09IFwib2JqZWN0XCIgfHwgdHlwZW9mIHRhcmdldCAhPT0gXCJvYmplY3RcIikge1xyXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXHJcbiAgICAgICAgYEludmFsaWQgdXNhZ2UuIEJvdGggc291cmNlICgke0pTT04uc3RyaW5naWZ5KFxyXG4gICAgICAgICAgc291cmNlXHJcbiAgICAgICAgKX0pIGFuZCB0YXJnZXQgKCR7SlNPTi5zdHJpbmdpZnkodGFyZ2V0KX0pIG11c3QgYmUgb2JqZWN0c2BcclxuICAgICAgKTtcclxuICAgIH1cclxuXHJcbiAgICBmb3IgKGNvbnN0IGtleSBvZiBPYmplY3Qua2V5cyhzb3VyY2UpKSB7XHJcbiAgICAgIGNvbnN0IHZhbHVlID0gc291cmNlW2tleV07XHJcbiAgICAgIGlmICh0eXBlb2YgdmFsdWUgPT09IFwib2JqZWN0XCIgJiYgdmFsdWUgIT0gbnVsbCAmJiAhQXJyYXkuaXNBcnJheSh2YWx1ZSkpIHtcclxuICAgICAgICAvLyBpZiB0aGUgdmFsdWUgYXQgdGhlIHRhcmdldCBpcyBub3QgYW4gb2JqZWN0LCBvdmVycmlkZSBpdCB3aXRoIGFuXHJcbiAgICAgICAgLy8gb2JqZWN0IHNvIHdlIGNhbiBjb250aW51ZSB0aGUgcmVjdXJzaW9uXHJcbiAgICAgICAgaWYgKHR5cGVvZiB0YXJnZXRba2V5XSAhPT0gXCJvYmplY3RcIikge1xyXG4gICAgICAgICAgdGFyZ2V0W2tleV0gPSB7fTtcclxuICAgICAgICB9XHJcblxyXG4gICAgICAgIGRlZXBNZXJnZSh0YXJnZXRba2V5XSwgdmFsdWUpO1xyXG5cclxuICAgICAgICAvLyBpZiB0aGUgcmVzdWx0IG9mIHRoZSBtZXJnZSBpcyBhbiBlbXB0eSBvYmplY3QsIGl0J3MgYmVjYXVzZSB0aGVcclxuICAgICAgICAvLyBldmVudHVhbCB2YWx1ZSB3ZSBhc3NpZ25lZCBpcyBgdW5kZWZpbmVkYCwgYW5kIHRoZXJlIGFyZSBub1xyXG4gICAgICAgIC8vIHNpYmxpbmcgY29uY3JldGUgdmFsdWVzIGFsb25nc2lkZSwgc28gd2UgY2FuIGRlbGV0ZSB0aGlzIHRyZWUuXHJcbiAgICAgICAgY29uc3Qgb3V0cHV0ID0gdGFyZ2V0W2tleV07XHJcbiAgICAgICAgaWYgKHR5cGVvZiBvdXRwdXQgPT09IFwib2JqZWN0XCIgJiYgT2JqZWN0LmtleXMob3V0cHV0KS5sZW5ndGggPT09IDApIHtcclxuICAgICAgICAgIGRlbGV0ZSB0YXJnZXRba2V5XTtcclxuICAgICAgICB9XHJcbiAgICAgIH0gZWxzZSBpZiAodmFsdWUgPT09IHVuZGVmaW5lZCkge1xyXG4gICAgICAgIGRlbGV0ZSB0YXJnZXRba2V5XTtcclxuICAgICAgfSBlbHNlIHtcclxuICAgICAgICB0YXJnZXRba2V5XSA9IHZhbHVlO1xyXG4gICAgICB9XHJcbiAgICB9XHJcbiAgfVxyXG5cclxuICByZXR1cm4gdGFyZ2V0O1xyXG59XHJcbiJdfQ==