"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RosParameter = exports.RosParameterType = void 0;
const ros_element_1 = require("./ros-element");
const ros_reference_1 = require("./private/ros-reference");
const token_1 = require("./token");
var RosParameterType;
(function (RosParameterType) {
    RosParameterType["STRING"] = "String";
    RosParameterType["NUMBER"] = "Number";
    RosParameterType["JSON"] = "Json";
    RosParameterType["BOOLEAN"] = "Boolean";
    RosParameterType["COMMAD_ELIMITED_LIST"] = "CommaDelimitedList";
})(RosParameterType = exports.RosParameterType || (exports.RosParameterType = {}));
/**
 * A ROS parameter.
 *
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 */
class RosParameter extends ros_element_1.RosElement {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.props = props;
        this.id = id;
        this.type = props.type || RosParameterType.STRING;
        this.validateProperties(props);
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     */
    get noEcho() {
        return !!this.props.noEcho;
    }
    /**
     * The parameter value as a Token
     */
    get value() {
        return ros_reference_1.RosReference.for(this, 'Ref');
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsString() {
        if (!isStringType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsBoolean() {
        if (!isStringType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string type`);
        }
        return token_1.Token.asAny(this.value);
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsAny() {
        return token_1.Token.asAny(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * The parameter value, if it represents a number.
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toRosTemplate() {
        return {
            Parameters: {
                [this.id]: {
                    Type: this.type,
                    Default: this.props.defaultValue,
                    AllowedPattern: this.props.allowedPattern,
                    AllowedValues: this.props.allowedValues,
                    ConstraintDescription: this.props.constraintDescription,
                    Description: this.props.description,
                    MaxLength: this.props.maxLength,
                    MaxValue: this.props.maxValue,
                    MinLength: this.props.minLength,
                    MinValue: this.props.minValue,
                    NoEcho: this.props.noEcho,
                    Label: this.props.label,
                    AssociationProperty: this.props.associationProperty,
                    Confirm: this.props.confirm,
                },
            },
        };
    }
    resolve(_context) {
        return this.value;
    }
    validateProperties(props = {}) {
        // check whether the required properties is in parameter(props)
        runtime_1.requireProperty(props, 'type', this);
        // check whether the associationProperty is allowed value
        util_1.isAllowedValue(props, 'associationProperty', this, RosParameter.ASSOCIATION_PROPERTY_ALLOWED_VALUES);
        // check the type of this parameter is allowed
        util_1.isAllowedValue(props, 'type', this, RosParameter.TYPE_ALLOWED_VALUES);
        // check whether the max length of string is larger than the min length
        if (props.maxLength && props.minLength && props.maxLength < props.minLength) {
            throw new Error(`The max length of string of the parameter ${this.toString()} is lower than its min length.`);
        }
        // check whether the max value of this parameter is larger than the min value
        if (props.maxValue && props.minValue && props.maxValue < props.minValue) {
            throw new Error(`The max value of the parameter ${this.toString()} is lower than its min value.`);
        }
        // check whether the "confirm" property is allowed to be set "true"
        if (props.confirm && (props.type !== 'String' || !props.noEcho)) {
            throw new Error(`The property "confirm" of the parameter ${this.toString()} could be true only if type is "String" and "noEcho" is true.`);
        }
    }
}
exports.RosParameter = RosParameter;
RosParameter.ASSOCIATION_PROPERTY_ALLOWED_VALUES = [
    'ALIYUN::ECS::Instance::ImageId',
    'ALIYUN::ECS::Instance::ZoneId',
    'ALIYUN::ECS::VPC::VPCId',
    'ALIYUN::ECS::VSwitch::VSwitchId',
];
RosParameter.TYPE_ALLOWED_VALUES = ['String', 'Number', 'CommaDelimitedList', 'Json', 'Boolean'];
(function (RosParameter) {
    class AssociationProperty {
    }
    AssociationProperty.ZONE_ID = 'ALIYUN::ECS::Instance::ZoneId';
    AssociationProperty.VPC_ID = 'ALIYUN::ECS::VPC::VPCId';
    AssociationProperty.IMAGE_ID = 'ALIYUN::ECS::Instance::ImageId';
    AssociationProperty.VSWITCH_ID = 'ALIYUN::ECS::VSwitch::VSwitchId';
    RosParameter.AssociationProperty = AssociationProperty;
})(RosParameter = exports.RosParameter || (exports.RosParameter = {}));
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
const runtime_1 = require("./runtime");
const util_1 = require("./util");
//# sourceMappingURL=data:application/json;base64,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