"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fn = void 0;
const template_lang_1 = require("./private/template-lang");
const intrinsic_1 = require("./private/intrinsic");
const reference_1 = require("./reference");
const token_1 = require("./token");
const stack_trace_1 = require("./stack-trace");
// tslint:disable:max-line-length
/**
 * Resource Orchestration Service intrinsic functions.
 * https://help.aliyun.com/document_detail/28852.html
 */
class Fn {
    static str(value) {
        if (!token_1.Token.isUnresolved(value)) {
            return value.toString();
        }
        return new FnStr(value).toString();
    }
    static base64Decode(data) {
        return new FnBase64Decode(data).toString();
    }
    static replace(replaceData, content) {
        return new FnReplace([replaceData, content]).toString();
    }
    static listMerge(valueList) {
        return new FnListMerge(valueList).toString();
    }
    static getJsonValue(value) {
        return new FnGetJsonValue(value).toString();
    }
    static avg(ndigits, values) {
        return new FnAvg(ndigits, values).toString();
    }
    static add(values) {
        return new FnAdd(values).toString();
    }
    static calculate(values, ndigits, para) {
        return new FnCalculate(values, ndigits, para).toString();
    }
    static max(values) {
        return new FnMax(values).toString();
    }
    static min(values) {
        return new FnMin(values).toString();
    }
    static jq(method, script, inputString) {
        return new FnJq(method, script, inputString).toString();
    }
    static mergeMapToList(mapList) {
        return new FnMergeMapToList(mapList).toString();
    }
    static selectMapList(key, mapList) {
        return new FnSelectMapList(key, mapList).toString();
    }
    /**
     * The ``Ref`` intrinsic function returns the value of the specified parameter or resource.
     * Note that it doesn't validate the logicalName, it mainly serves paremeter/resource reference defined in a ``RosInclude`` template.
     * @param logicalName The logical name of a parameter/resource for which you want to retrieve its value.
     */
    static ref(logicalName) {
        return new FnRef(logicalName).toString();
    }
    /** @internal */
    static _ref(logicalId) {
        return new FnRef(logicalId);
    }
    /**
     * The ``Fn::GetAtt`` intrinsic function returns the value of an attribute
     * from a resource in the template.
     * @param logicalNameOfResource The logical name (also called logical ID) of
     * the resource that contains the attribute that you want.
     * @param attributeName The name of the resource-specific attribute whose
     * value you want. See the resource's reference page for details about the
     * attributes available for that resource type.
     * @returns an IResolvable object
     */
    static getAtt(logicalNameOfResource, attributeName) {
        return new FnGetAtt(logicalNameOfResource, attributeName);
    }
    /**
     * The intrinsic function ``Fn::Join`` appends a set of values into a single
     * value, separated by the specified delimiter. If a delimiter is the empty
     * string, the set of values are concatenated with no delimiter.
     * @param delimiter The value you want to occur between fragments. The
     * delimiter will occur between fragments only. It will not terminate the
     * final value.
     * @param listOfValues The list of values you want combined.
     * @returns a token represented as a string
     */
    static join(delimiter, listOfValues) {
        if (listOfValues.length === 0) {
            throw new Error("FnJoin requires at least one value to be provided");
        }
        return new FnJoin(delimiter, listOfValues).toString();
    }
    /**
     * To split a string into a list of string values so that you can select an element from the
     * resulting string list, use the ``Fn::Split`` intrinsic function. Specify the location of splits
     * with a delimiter, such as , (a comma). After you split a string, use the ``Fn::Select`` function
     * to pick a specific element.
     * @param delimiter A string value that determines where the source string is divided.
     * @param source The string value that you want to split.
     * @returns a token represented as a string array
     */
    static split(delimiter, source) {
        // short-circut if source is not a token
        if (!token_1.Token.isUnresolved(source)) {
            return source.split(delimiter);
        }
        return token_1.Token.asList(new FnSplit(delimiter, source));
    }
    /**
     * The intrinsic function ``Fn::Select`` returns a single object from a list of objects by index.
     * @param index The index of the object to retrieve. This must be a value from zero to N-1, where N represents the number of elements in the array.
     * @param array The list of objects to select from. This list must not be null, nor can it have null entries.
     * @returns a token represented as a string
     */
    static select(index, array) {
        return new FnSelect(index, array).toString();
    }
    /**
     * The intrinsic function ``Fn::Sub`` substitutes variables in an input string
     * with values that you specify. In your templates, you can use this function
     * to construct commands or outputs that include values that aren't available
     * until you create or update a stack.
     * @param body A string with variables that Ros Template substitutes
     * with their associated values at runtime. Write variables as ${MyVarName}.
     * Variables can be template parameter names, resource logical IDs, resource
     * attributes, or a variable in a key-value map. If you specify only template
     * parameter names, resource logical IDs, and resource attributes, don't
     * specify a key-value map.
     * @param variables The name of a variable that you included in the String
     * parameter. The value that Ros Template substitutes for the associated
     * variable name at runtime.
     * @returns a token represented as a string
     */
    static sub(body, variables) {
        return new FnSub(body, variables).toString();
    }
    /**
     * The intrinsic function ``Fn::Base64`` returns the Base64 representation of
     * the input string.
     * @param data The string value you want to convert to Base64.
     * @returns a token represented as a string
     */
    static base64Encode(data) {
        return new FnBase64Encode(data).toString();
    }
    /**
     * The intrinsic function ``Fn::GetAZs`` returns an array that lists
     * Availability Zones for a specified region. Because customers have access to
     * different Availability Zones, the intrinsic function ``Fn::GetAZs`` enables
     * template authors to write templates that adapt to the calling user's
     * access. That way you don't have to hard-code a full list of Availability
     * Zones for a specified region.
     * @param region The name of the region for which you want to get the
     * Availability Zones. You can use the ROS::Region pseudo parameter to specify
     * the region in which the stack is created. Specifying an empty string is
     * equivalent to specifying ROS::Region.
     * @returns a token represented as a string array
     */
    static getAzs(region) {
        return token_1.Token.asList(new FnGetAZs(region));
    }
    /**
     * The intrinsic function ``Fn::GetStackOutput`` returns the value of an output
     * exported by another stack.
     * @returns a token represented as a string
     */
    static getStackOutput(stackID, outputKey, stackRegion) {
        return new FnGetStackOutput(stackID, outputKey, stackRegion).toString();
    }
    /**
     * The intrinsic function ``Fn::FindInMap`` returns the value corresponding to
     * keys in a two-level map that is declared in the Mappings section.
     * @returns a token represented as a string
     */
    static findInMap(mapName, topLevelKey, secondLevelKey) {
        return new FnFindInMap(mapName, topLevelKey, secondLevelKey).toString();
    }
    /**
     * Returns true if all the specified conditions evaluate to true, or returns
     * false if any one of the conditions evaluates to false. ``Fn::And`` acts as
     * an AND operator. The minimum number of conditions that you can include is
     * 2, and the maximum is 10.
     * @param conditions conditions to AND
     * @returns an FnCondition token
     */
    static conditionAnd(...conditions) {
        return new FnAnd(...conditions);
    }
    /**
     * Compares if two values are equal. Returns true if the two values are equal
     * or false if they aren't.
     * @param lhs A value of any type that you want to compare.
     * @param rhs A value of any type that you want to compare.
     * @returns an FnCondition token
     */
    static conditionEquals(lhs, rhs) {
        return new FnEquals(lhs, rhs);
    }
    /**
     * Returns one value if the specified condition evaluates to true and another
     * value if the specified condition evaluates to false.
     * @param conditionId A reference to a condition in the Conditions section. Use
     * the condition's name to reference it.
     * @param valueIfTrue A value to be returned if the specified condition
     * evaluates to true.
     * @param valueIfFalse A value to be returned if the specified condition
     * evaluates to false.
     * @returns an FnCondition token
     */
    static conditionIf(conditionId, valueIfTrue, valueIfFalse) {
        return new FnIf(conditionId, valueIfTrue, valueIfFalse);
    }
    /**
     * Returns true for a condition that evaluates to false or returns false for a
     * condition that evaluates to true. ``Fn::Not`` acts as a NOT operator.
     * @param condition A condition such as ``Fn::Equals`` that evaluates to true
     * or false.
     * @returns an FnCondition token
     */
    static conditionNot(condition) {
        return new FnNot(condition);
    }
    /**
     * Returns true if any one of the specified conditions evaluate to true, or
     * returns false if all of the conditions evaluates to false. ``Fn::Or`` acts
     * as an OR operator. The minimum number of conditions that you can include is
     * 2, and the maximum is 10.
     * @param conditions conditions that evaluates to true or false.
     * @returns an FnCondition token
     */
    static conditionOr(...conditions) {
        return new FnOr(...conditions);
    }
    /**
     * Returns true if a specified string matches all values in a list.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list
     * of strings.
     * @returns an FnCondition token
     */
    static conditionEachMemberEquals(listOfStrings, value) {
        return new FnEachMemberEquals(listOfStrings, value);
    }
    constructor() { }
}
exports.Fn = Fn;
/**
 * Base class for tokens that represent ROS intrinsic functions.
 */
class FnBase extends intrinsic_1.Intrinsic {
    constructor(name, value) {
        super({ [name]: value });
    }
}
// new function
class FnStr extends FnBase {
    /**
     * Creates an ``Str`` function.
     */
    constructor(value) {
        super("Fn::Str", value);
    }
}
class FnBase64Decode extends FnBase {
    constructor(data) {
        super("Fn::Base64Decode", data);
    }
}
class FnReplace extends FnBase {
    /**
     * Creates an ``Replace`` function.
     */
    constructor(value) {
        super("Fn::Replace", value);
    }
}
class FnListMerge extends FnBase {
    /**
     * Creates an ``ListMerge`` function.
     */
    constructor(value) {
        super("Fn::ListMerge", value);
    }
}
class FnGetJsonValue extends FnBase {
    /**
     * Creates an ``GetJsonValue`` function.
     */
    constructor(value) {
        super("Fn::GetJsonValue", value);
    }
}
class FnAvg extends FnBase {
    /**
     * Creates an ``Avg`` function.
     */
    constructor(ndigits, values) {
        super("Fn::Avg", [ndigits, values]);
    }
}
class FnAdd extends FnBase {
    /**
     * Creates an ``Add`` function.
     */
    constructor(values) {
        super("Fn::Add", values);
    }
}
class FnCalculate extends FnBase {
    /**
     * Creates an ``Calculate`` function.
     */
    constructor(values, ndigits, para) {
        super("Fn::Calculate", [values, ndigits, para]);
    }
}
class FnMax extends FnBase {
    /**
     * Creates an ``Max`` function.
     */
    constructor(values) {
        super("Fn::Max", values);
    }
}
class FnMin extends FnBase {
    /**
     * Creates an ``Min`` function.
     */
    constructor(values) {
        super("Fn::Min", values);
    }
}
class FnGetStackOutput extends FnBase {
    /**
     * Creates an ``GetStackOutput`` function.
     */
    constructor(stackID, outputKey, stackRegion) {
        super("Fn::GetStackOutput", [stackID, outputKey, stackRegion]);
    }
}
class FnJq extends FnBase {
    /**
     * Creates an ``Jq`` function.
     */
    constructor(method, script, inputString) {
        super("Fn::Jq", [method, script, inputString]);
    }
}
class FnMergeMapToList extends FnBase {
    /**
     * Creates an ``FnMergeMapToList`` function.
     */
    constructor(mapList) {
        super("Fn::MergeMapToList", mapList);
    }
}
class FnSelectMapList extends FnBase {
    /**
     * Creates an ``FnMergeMapToList`` function.
     */
    constructor(key, mapList) {
        super("Fn::SelectMapList", [key, mapList]);
    }
}
/**
 * The intrinsic function ``Ref`` returns the value of the specified parameter or resource.
 * When you specify a parameter's logical name, it returns the value of the parameter.
 * When you specify a resource's logical name, it returns a value that you can typically use to refer to that resource, such as a physical ID.
 */
class FnRef extends FnBase {
    /**
     * Creates an ``Ref`` function.
     * @param logicalName The logical name of a parameter/resource for which you want to retrieve its value.
     */
    constructor(logicalName) {
        super("Ref", logicalName);
    }
}
/**
 * The intrinsic function ``Fn::FindInMap`` returns the value corresponding to keys in a two-level
 * map that is declared in the Mappings section.
 */
class FnFindInMap extends FnBase {
    /**
     * Creates an ``Fn::FindInMap`` function.
     * @param mapName The logical name of a mapping declared in the Mappings section that contains the keys and values.
     * @param topLevelKey The top-level key name. Its value is a list of key-value pairs.
     * @param secondLevelKey The second-level key name, which is set to one of the keys from the list assigned to TopLevelKey.
     */
    constructor(mapName, topLevelKey, secondLevelKey) {
        super("Fn::FindInMap", [mapName, topLevelKey, secondLevelKey]);
    }
}
/**
 * The ``Fn::GetAtt`` intrinsic function returns the value of an attribute from a resource in the template.
 */
class FnGetAtt extends FnBase {
    /**
     * Creates a ``Fn::GetAtt`` function.
     * @param logicalNameOfResource The logical name (also called logical ID) of the resource that contains the attribute that you want.
     * @param attributeName The name of the resource-specific attribute whose value you want. See the resource's reference page for details about the attributes available for that resource type.
     */
    constructor(logicalNameOfResource, attributeName) {
        super("Fn::GetAtt", [logicalNameOfResource, attributeName]);
    }
}
/**
 * The intrinsic function ``Fn::GetAZs`` returns an array that lists Availability Zones for a
 * specified region. Because customers have access to different Availability Zones, the intrinsic
 * function ``Fn::GetAZs`` enables template authors to write templates that adapt to the calling
 * user's access. That way you don't have to hard-code a full list of Availability Zones for a
 * specified region.
 */
class FnGetAZs extends FnBase {
    /**
     * Creates an ``Fn::GetAZs`` function.
     * @param region The name of the region for which you want to get the Availability Zones.
     */
    constructor(region) {
        super("Fn::GetAZs", region || "");
    }
}
/**
 * The intrinsic function ``Fn::Select`` returns a single object from a list of objects by index.
 */
class FnSelect extends FnBase {
    /**
     * Creates an ``Fn::Select`` function.
     * @param index The index of the object to retrieve. This must be a value from zero to N-1, where N represents the number of elements in the array.
     * @param array The list of objects to select from. This list must not be null, nor can it have null entries.
     */
    constructor(index, array) {
        super("Fn::Select", [index, array]);
    }
}
/**
 * To split a string into a list of string values so that you can select an element from the
 * resulting string list, use the ``Fn::Split`` intrinsic function. Specify the location of splits
 * with a delimiter, such as , (a comma). After you split a string, use the ``Fn::Select`` function
 * to pick a specific element.
 */
class FnSplit extends FnBase {
    /**
     * Create an ``Fn::Split`` function.
     * @param delimiter A string value that determines where the source string is divided.
     * @param source The string value that you want to split.
     */
    constructor(delimiter, source) {
        super("Fn::Split", [delimiter, source]);
    }
}
/**
 * The intrinsic function ``Fn::Sub`` substitutes variables in an input string with values that
 * you specify. In your templates, you can use this function to construct commands or outputs
 * that include values that aren't available until you create or update a stack.
 */
class FnSub extends FnBase {
    /**
     * Creates an ``Fn::Sub`` function.
     * @param body A string with variables that Ros Template substitutes with their
     *       associated values at runtime. Write variables as ${MyVarName}. Variables
     *       can be template parameter names, resource logical IDs, resource attributes,
     *       or a variable in a key-value map. If you specify only template parameter names,
     *       resource logical IDs, and resource attributes, don't specify a key-value map.
     * @param variables The name of a variable that you included in the String parameter.
     *          The value that Ros Template substitutes for the associated variable name at runtime.
     */
    constructor(body, variables) {
        super("Fn::Sub", variables ? [body, variables] : body);
    }
}
/**
 * The intrinsic function ``Fn::Base64`` returns the Base64 representation of the input string.
 */
class FnBase64Encode extends FnBase {
    /**
     * Creates an ``Fn::Base64`` function.
     * @param data The string value you want to convert to Base64.
     */
    constructor(data) {
        super("Fn::Base64Encode", data);
    }
}
class FnConditionBase extends intrinsic_1.Intrinsic {
    constructor(type, value) {
        super({ [type]: value });
    }
}
/**
 * Returns true if all the specified conditions evaluate to true, or returns false if any one
 *  of the conditions evaluates to false. ``Fn::And`` acts as an AND operator. The minimum number of
 * conditions that you can include is 2, and the maximum is 10.
 */
class FnAnd extends FnConditionBase {
    constructor(...condition) {
        super("Fn::And", condition);
    }
}
/**
 * Compares if two values are equal. Returns true if the two values are equal or false
 * if they aren't.
 */
class FnEquals extends FnConditionBase {
    /**
     * Creates an ``Fn::Equals`` condition function.
     * @param lhs A value of any type that you want to compare.
     * @param rhs A value of any type that you want to compare.
     */
    constructor(lhs, rhs) {
        super("Fn::Equals", [lhs, rhs]);
    }
}
/**
 * Returns one value if the specified condition evaluates to true and another value if the
 * specified condition evaluates to false.
 */
class FnIf extends FnConditionBase {
    /**
     * Creates an ``Fn::If`` condition function.
     * @param condition A reference to a condition in the Conditions section. Use the condition's name to reference it.
     * @param valueIfTrue A value to be returned if the specified condition evaluates to true.
     * @param valueIfFalse A value to be returned if the specified condition evaluates to false.
     */
    constructor(condition, valueIfTrue, valueIfFalse) {
        super("Fn::If", [condition, valueIfTrue, valueIfFalse]);
    }
}
/**
 * Returns true for a condition that evaluates to false or returns false for a condition that evaluates to true.
 * ``Fn::Not`` acts as a NOT operator.
 */
class FnNot extends FnConditionBase {
    /**
     * Creates an ``Fn::Not`` condition function.
     * @param condition A condition such as ``Fn::Equals`` that evaluates to true or false.
     */
    constructor(condition) {
        super("Fn::Not", [condition]);
    }
}
/**
 * Returns true if any one of the specified conditions evaluate to true, or returns false if
 * all of the conditions evaluates to false. ``Fn::Or`` acts as an OR operator. The minimum number
 * of conditions that you can include is 2, and the maximum is 10.
 */
class FnOr extends FnConditionBase {
    /**
     * Creates an ``Fn::Or`` condition function.
     * @param condition A condition that evaluates to true or false.
     */
    constructor(...condition) {
        super("Fn::Or", condition);
    }
}
/**
 * Returns true if a specified string matches all values in a list.
 */
class FnEachMemberEquals extends FnConditionBase {
    /**
     * Creates an ``Fn::EachMemberEquals`` function.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list of strings.
     */
    constructor(listOfStrings, value) {
        super("Fn::EachMemberEquals", [listOfStrings, value]);
    }
}
/**
 * The intrinsic function ``Fn::Join`` appends a set of values into a single value, separated by
 * the specified delimiter. If a delimiter is the empty string, the set of values are concatenated
 * with no delimiter.
 */
class FnJoin {
    /**
     * Creates an ``Fn::Join`` function.
     * @param delimiter The value you want to occur between fragments. The delimiter will occur between fragments only.
     *          It will not terminate the final value.
     * @param listOfValues The list of values you want combined.
     */
    constructor(delimiter, listOfValues) {
        if (listOfValues.length === 0) {
            throw new Error("FnJoin requires at least one value to be provided");
        }
        this.delimiter = delimiter;
        this.listOfValues = listOfValues;
        this.creationStack = stack_trace_1.captureStackTrace();
    }
    resolve(context) {
        if (token_1.Token.isUnresolved(this.listOfValues)) {
            // This is a list token, don't try to do smart things with it.
            return { "Fn::Join": [this.delimiter, this.listOfValues] };
        }
        const resolved = this.resolveValues(context);
        if (resolved.length === 1) {
            return resolved[0];
        }
        return { "Fn::Join": [this.delimiter, resolved] };
    }
    toString() {
        return token_1.Token.asString(this, { displayHint: "Fn::Join" });
    }
    toJSON() {
        return "<Fn::Join>";
    }
    /**
     * Optimization: if an Fn::Join is nested in another one and they share the same delimiter, then flatten it up. Also,
     * if two concatenated elements are literal strings (not tokens), then pre-concatenate them with the delimiter, to
     * generate shorter output.
     */
    resolveValues(context) {
        const resolvedValues = this.listOfValues.map((x) => reference_1.Reference.isReference(x) ? x : context.resolve(x));
        return template_lang_1.minimalRosTemplateJoin(this.delimiter, resolvedValues);
    }
}
//# sourceMappingURL=data:application/json;base64,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