"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RosRefElement = exports.RosElement = void 0;
const cxschema = require("@alicloud/ros-cdk-assembly-schema");
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const token_1 = require("./token");
const ROS_ELEMENT_SYMBOL = Symbol.for("ros-cdk-core.RosElement");
/**
 * An element of a stack.
 */
class RosElement extends construct_compat_1.Construct {
    /**
     * Creates an entity and binds it to a tree.
     * Note that the root of the tree must be a Stack object (not just any Root).
     *
     * @param scope The parent construct
     * @param props Construct properties
     */
    constructor(scope, id) {
        super(scope, id);
        Object.defineProperty(this, ROS_ELEMENT_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        this.logicalId = lazy_1.Lazy.stringValue({ produce: () => this.synthesizeLogicalId() }, {
            displayHint: `${notTooLong(this.node.path)}.LogicalID`,
        });
        this.logicalId = id;
        this.node.addMetadata(cxschema.ArtifactMetadataEntryType.LOGICAL_ID, this.logicalId, this.constructor);
    }
    /**
     * Returns `true` if a construct is a stack element (i.e. part of the
     * synthesized template).
     *
     * Uses duck-typing instead of `instanceof` to allow stack elements from different
     * versions of this library to be included in the same stack.
     *
     * @returns The construct as a stack element or undefined if it is not a stack element.
     */
    static isRosElement(x) {
        return ROS_ELEMENT_SYMBOL in x;
    }
    /**
     * Overrides the auto-generated logical ID with a specific ID.
     * @param newLogicalId The new logical ID to use for this stack element.
     */
    overrideLogicalId(newLogicalId) {
        this._logicalIdOverride = newLogicalId;
    }
    /**
     * @returns the stack trace of the point where this Resource was created from, sourced
     *      from the +metadata+ entry typed +aliyun:ros:logicalId+, and with the bottom-most
     *      node +internal+ entries filtered.
     */
    get creationStack() {
        const trace = this.node.metadata.find((md) => md.type === cxschema.ArtifactMetadataEntryType.LOGICAL_ID).trace;
        if (!trace) {
            return [];
        }
        return filterStackTrace(trace);
        function filterStackTrace(stack) {
            const result = Array.of(...stack);
            while (result.length > 0 && shouldFilter(result[result.length - 1])) {
                result.pop();
            }
            // It's weird if we filtered everything, so return the whole stack...
            return result.length === 0 ? stack : result;
        }
        function shouldFilter(str) {
            return str.match(/[^(]+\(internal\/.*/) !== null;
        }
    }
    /**
     * Called during synthesize to render the logical ID of this element. If
     * `overrideLogicalId` was it will be used, otherwise, we will allocate the
     * logical ID through the stack.
     */
    synthesizeLogicalId() {
        if (this._logicalIdOverride) {
            return this._logicalIdOverride;
        }
        else {
            return this.stack.getLogicalId(this);
        }
    }
}
exports.RosElement = RosElement;
/**
 * Base class for referenceable template constructs which are not Resources
 *
 * These constructs are things like Conditions and Parameters, can be
 * referenced by taking the `.ref` attribute.
 *
 * Resource constructs do not inherit from RosRefElement because they have their
 * own, more specific types returned from the .ref attribute. Also, some
 * resources aren't referenceable at all (such as BucketPolicies or GatewayAttachments).
 */
class RosRefElement extends RosElement {
    /**
     * Return a string that will be resolved to a RosTemplate `{ Ref }` for this element.
     *
     * If, by any chance, the intrinsic reference of a resource is not a string, you could
     * coerce it to an IResolvable through `Lazy.any({ produce: resource.ref })`.
     */
    get ref() {
        return token_1.Token.asString(ros_reference_1.RosReference.for(this, "Ref"));
    }
}
exports.RosRefElement = RosRefElement;
function notTooLong(x) {
    if (x.length < 100) {
        return x;
    }
    return x.substr(0, 47) + "..." + x.substr(x.length - 47);
}
const ros_reference_1 = require("./private/ros-reference");
const stack_1 = require("./stack");
//# sourceMappingURL=data:application/json;base64,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