"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
const construct_compat_1 = require("./construct-compat");
const physical_name_generator_1 = require("./private/physical-name-generator");
const token_1 = require("./token");
const lazy_1 = require("./lazy");
const stack_1 = require("./stack");
/**
 * A construct which represents a resource.
 */
class Resource extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.stringValue({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    addDependency(resource) {
        var _a;
        (_a = this.resource) === null || _a === void 0 ? void 0 : _a.addRosDependency(resource.node.id);
    }
    applyRemovalPolicy(policy) {
        var _a;
        (_a = this.resource) === null || _a === void 0 ? void 0 : _a.applyRemovalPolicy(policy);
    }
    addCondition(condition) {
        var _a;
        (_a = this.resource) === null || _a === void 0 ? void 0 : _a.addCondition(condition);
    }
    addResourceDesc(desc) {
        var _a;
        (_a = this.resource) === null || _a === void 0 ? void 0 : _a.addDesc(desc);
    }
    setMetadata(key, value) {
        if (this.resource) {
            this.resource.addMetaData(key, value);
        }
    }
    getAtt(name) {
        var _a;
        return token_1.Token.asString((_a = this.resource) === null || _a === void 0 ? void 0 : _a.getAtt(name));
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,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