"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultTokenResolver = exports.StringConcat = void 0;
const encoding_1 = require("./private/encoding");
const token_map_1 = require("./private/token-map");
/**
 * Converts all fragments to strings and concats those
 *
 * Drops 'undefined's.
 */
class StringConcat {
    join(left, right) {
        if (left === undefined) {
            return right !== undefined ? `${right}` : undefined;
        }
        if (right === undefined) {
            return `${left}`;
        }
        return `${left}${right}`;
    }
}
exports.StringConcat = StringConcat;
/**
 * Default resolver implementation
 *
 * @experimental
 */
class DefaultTokenResolver {
    constructor(concat) {
        this.concat = concat;
    }
    /**
     * Default Token resolution
     *
     * Resolve the Token, recurse into whatever it returns,
     * then finally post-process it.
     */
    resolveToken(t, context, postProcessor) {
        try {
            let resolved = t.resolve(context);
            // The token might have returned more values that need resolving, recurse
            resolved = context.resolve(resolved);
            resolved = postProcessor.postProcess(resolved, context);
            return resolved;
        }
        catch (e) {
            let message = `Resolution error: ${e.message}.`;
            if (t.creationStack && t.creationStack.length > 0) {
                message += `\nObject creation stack:\n  at ${t.creationStack.join("\n  at ")}`;
            }
            e.message = message;
            throw e;
        }
    }
    /**
     * Resolve string fragments to Tokens
     */
    resolveString(fragments, context) {
        return fragments.mapTokens({ mapToken: context.resolve }).join(this.concat);
    }
    resolveList(xs, context) {
        // Must be a singleton list token, because concatenation is not allowed.
        if (xs.length !== 1) {
            throw new Error(`Cannot add elements to list token, got: ${xs}`);
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const tokenMap = token_map_1.TokenMap.instance();
        const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
        if (fragments.length !== 1) {
            throw new Error(`Cannot concatenate strings in a tokenized string array, got: ${xs[0]}`);
        }
        return fragments.mapTokens({ mapToken: context.resolve }).firstValue;
    }
}
exports.DefaultTokenResolver = DefaultTokenResolver;
//# sourceMappingURL=data:application/json;base64,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