"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeUniqueId = void 0;
// tslint:disable-next-line:no-var-requires
const crypto = require("crypto");
const encoding_1 = require("./encoding");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = "Resource";
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = "Default";
const PATH_SEP = "/";
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
const MAX_ID_LEN = 255;
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components) {
    components = components.filter((x) => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error("Unable to calculate a unique id for an empty set of components");
    }
    // Lazy require in order to break a module dependency cycle
    const unresolvedTokens = components.filter((c) => encoding_1.unresolved(c));
    if (unresolvedTokens.length > 0) {
        throw new Error(`ID components may not include unresolved tokens: ${unresolvedTokens.join(",")}`);
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of ROS templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        // we filter out non-alpha characters but that is actually a bad idea
        // because it could create conflicts ("A-B" and "AB" will render the same
        // logical ID). sadly, changing it in the 1.x version line is impossible
        // because it will be a breaking change. we should consider for v2.0.
        const candidate = removeNonAlphanumeric(components[0]);
        // if our candidate is short enough, use it as is. otherwise, fall back to
        // the normal mode.
        if (candidate.length <= MAX_ID_LEN) {
            return candidate;
        }
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter((x) => x !== HIDDEN_FROM_HUMAN_ID)
        .map(removeNonAlphanumeric)
        .join("")
        .slice(0, MAX_HUMAN_LEN);
    return human + hash;
}
exports.makeUniqueId = makeUniqueId;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto
        .createHash("md5")
        .update(path.join(PATH_SEP))
        .digest("hex");
    return md5.slice(0, HASH_LEN).toUpperCase();
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, "");
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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