"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeMetadata = void 0;
const fs = require("fs");
const path = require("path");
const ros_cdk_assembly_schema_1 = require("@alicloud/ros-cdk-assembly-schema");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
const tree_1 = require("../tree");
const FILE_PATH = "tree.json";
/**
 * Construct that is automatically attached to the top-level `App`.
 * This generates, as part of synthesis, a file containing the construct tree and the metadata for each node in the tree.
 * The output is in a tree format so as to preserve the construct hierarchy.
 *
 * @experimental
 */
class TreeMetadata extends construct_compat_1.Construct {
    constructor(scope) {
        super(scope, "Tree");
    }
    synthesize(session) {
        const lookup = {};
        const visit = (construct) => {
            const children = construct.node.children.map((c) => {
                try {
                    return visit(c);
                }
                catch (e) {
                    this.node.addWarning(`Failed to render tree metadata for node [${c.node.id}]. Reason: ${e}`);
                    return undefined;
                }
            });
            const childrenMap = children
                .filter((child) => child !== undefined)
                .reduce((map, child) => Object.assign(map, { [child.id]: child }), {});
            const node = {
                id: construct.node.id || "App",
                path: construct.node.path,
                children: Object.keys(childrenMap).length === 0 ? undefined : childrenMap,
                attributes: this.synthAttributes(construct),
            };
            lookup[node.path] = node;
            return node;
        };
        const tree = {
            version: "tree-0.1",
            tree: visit(this.node.root),
        };
        const builder = session.assembly;
        fs.writeFileSync(path.join(builder.outdir, FILE_PATH), JSON.stringify(tree, undefined, 2), { encoding: "utf-8" });
        builder.addArtifact("Tree", {
            type: ros_cdk_assembly_schema_1.ArtifactType.CDK_TREE,
            properties: {
                file: FILE_PATH,
            },
        });
    }
    synthAttributes(construct) {
        // check if a construct implements IInspectable
        function canInspect(inspectable) {
            return inspectable.inspect !== undefined;
        }
        const inspector = new tree_1.TreeInspector();
        // get attributes from the inspector
        if (canInspect(construct)) {
            construct.inspect(inspector);
            return stack_1.Stack.of(construct).resolve(inspector.attributes);
        }
        return undefined;
    }
}
exports.TreeMetadata = TreeMetadata;
//# sourceMappingURL=data:application/json;base64,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