"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isNameOfRosIntrinsic = exports.minimalRosTemplateJoin = exports.ROS_TOKEN_RESOLVER = exports.RosTemplateLang = void 0;
const lazy_1 = require("../lazy");
const reference_1 = require("../reference");
const resolvable_1 = require("../resolvable");
const token_1 = require("../token");
const intrinsic_1 = require("./intrinsic");
const resolve_1 = require("./resolve");
/**
 * Routines that know how to do operations at the ROS document language level
 */
class RosTemplateLang {
    /**
     * Turn an arbitrary structure potentially containing Tokens into a JSON string.
     *
     * Returns a Token which will evaluate to ROS expression that
     * will be evaluated by ROS to the JSON representation of the
     * input structure.
     *
     * All Tokens substituted in this way must return strings, or the evaluation
     * in ROS will fail.
     *
     * @param obj The object to stringify
     * @param space Indentation to use (default: no pretty-printing)
     */
    static toJSON(obj, space) {
        // This works in two stages:
        //
        // First, resolve everything. This gets rid of the lazy evaluations, evaluation
        // to the real types of things (for example, would a function return a string, an
        // intrinsic, or a number? We have to resolve to know).
        //
        // We then to through the returned result, identify things that evaluated to
        // ROS intrinsics, and re-wrap those in Tokens that have a
        // toJSON() method returning their string representation. If we then call
        // JSON.stringify() on that result, that gives us essentially the same
        // string that we started with, except with the non-token characters quoted.
        //
        //    {"field": "${TOKEN}"} --> {\"field\": \"${TOKEN}\"}
        //
        // A final resolve() on that string (done by the framework) will yield the string
        // we're after.
        //
        // Resolving and wrapping are done in go using the resolver framework.
        class IntrinsincWrapper extends resolvable_1.DefaultTokenResolver {
            constructor() {
                super(ROS_CONCAT);
            }
            resolveToken(t, context, postProcess) {
                // Return References directly, so their type is maintained and the references will
                // continue to work. Only while preparing, because we do need the final value of the
                // token while resolving.
                if (reference_1.Reference.isReference(t) && context.preparing) {
                    return wrap(t);
                }
                // Deep-resolve and wrap. This is necessary for Lazy tokens so we can see "inside" them.
                return wrap(super.resolveToken(t, context, postProcess));
            }
            resolveString(fragments, context) {
                return wrap(super.resolveString(fragments, context));
            }
            resolveList(l, context) {
                return wrap(super.resolveList(l, context));
            }
        }
        // We need a ResolveContext to get started so return a Token
        return lazy_1.Lazy.stringValue({
            produce: (ctx) => JSON.stringify(resolve_1.resolve(obj, {
                preparing: ctx.preparing,
                scope: ctx.scope,
                resolver: new IntrinsincWrapper(),
            }), undefined, space),
        });
        function wrap(value) {
            return isIntrinsic(value)
                ? new JsonToken(deepQuoteStringsForJSON(value))
                : value;
        }
    }
    /**
     * Produce a ROS expression to concat two arbitrary expressions when resolving
     */
    static concat(left, right) {
        if (left === undefined && right === undefined) {
            return "";
        }
        const parts = new Array();
        if (left !== undefined) {
            parts.push(left);
        }
        if (right !== undefined) {
            parts.push(right);
        }
        // Some case analysis to produce minimal expressions
        if (parts.length === 1) {
            return parts[0];
        }
        if (parts.length === 2 &&
            typeof parts[0] === "string" &&
            typeof parts[1] === "string") {
            return parts[0] + parts[1];
        }
        // Otherwise return a Join intrinsic (already in the target document language to avoid taking
        // circular dependencies on FnJoin & friends)
        return { "Fn::Join": ["", minimalRosTemplateJoin("", parts)] };
    }
}
exports.RosTemplateLang = RosTemplateLang;
/**
 * Token that also stringifies in the toJSON() operation.
 */
class JsonToken extends intrinsic_1.Intrinsic {
    /**
     * Special handler that gets called when JSON.stringify() is used.
     */
    toJSON() {
        return this.toString();
    }
}
/**
 * Deep escape strings for use in a JSON context
 */
function deepQuoteStringsForJSON(x) {
    if (typeof x === "string") {
        // Whenever we escape a string we strip off the outermost quotes
        // since we're already in a quoted context.
        const stringified = JSON.stringify(x);
        return stringified.substring(1, stringified.length - 1);
    }
    if (Array.isArray(x)) {
        return x.map(deepQuoteStringsForJSON);
    }
    if (typeof x === "object") {
        for (const key of Object.keys(x)) {
            x[key] = deepQuoteStringsForJSON(x[key]);
        }
    }
    return x;
}
const ROS_CONCAT = {
    join(left, right) {
        return RosTemplateLang.concat(left, right);
    },
};
/**
 * Default Token resolver for ROS templates
 */
exports.ROS_TOKEN_RESOLVER = new resolvable_1.DefaultTokenResolver(ROS_CONCAT);
/**
 * Do an intelligent ROS join on the given values, producing a minimal expression
 */
function minimalRosTemplateJoin(delimiter, values) {
    let i = 0;
    while (i < values.length) {
        const el = values[i];
        if (isSplicableFnJoinIntrinsic(el)) {
            values.splice(i, 1, ...el["Fn::Join"][1]);
        }
        else if (i > 0 &&
            isPlainString(values[i - 1]) &&
            isPlainString(values[i])) {
            values[i - 1] += delimiter + values[i];
            values.splice(i, 1);
        }
        else {
            i += 1;
        }
    }
    return values;
    function isPlainString(obj) {
        return typeof obj === "string" && !token_1.Token.isUnresolved(obj);
    }
    function isSplicableFnJoinIntrinsic(obj) {
        if (!isIntrinsic(obj)) {
            return false;
        }
        if (Object.keys(obj)[0] !== "Fn::Join") {
            return false;
        }
        const [delim, list] = obj["Fn::Join"];
        if (delim !== delimiter) {
            return false;
        }
        if (token_1.Token.isUnresolved(list)) {
            return false;
        }
        if (!Array.isArray(list)) {
            return false;
        }
        return true;
    }
}
exports.minimalRosTemplateJoin = minimalRosTemplateJoin;
/**
 * Return whether the given value represents a ROS intrinsic
 */
function isIntrinsic(x) {
    if (Array.isArray(x) || x === null || typeof x !== "object") {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === "Ref" || isNameOfRosIntrinsic(keys[0]);
}
function isNameOfRosIntrinsic(name) {
    if (!name.startsWith("Fn::")) {
        return false;
    }
    // these are 'fake' intrinsics, only usable inside the parameter overrides of a ROS CodePipeline Action
    return name !== "Fn::GetArtifactAtt" && name !== "Fn::GetParam";
}
exports.isNameOfRosIntrinsic = isNameOfRosIntrinsic;
//# sourceMappingURL=data:application/json;base64,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