"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synthesize = void 0;
const cxapi = require("@alicloud/ros-cdk-cxapi");
const prepare_app_1 = require("./prepare-app");
const stage_1 = require("../stage");
function synthesize(root, options = {}) {
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    // This is mostly here for legacy purposes as the framework itself does not use prepare anymore.
    prepareTree(root);
    // resolve references
    prepare_app_1.prepareApp(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder);
    return builder.buildAssembly({
        runtimeInfo: options.runtimeInfo,
    });
}
exports.synthesize = synthesize;
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        // hackery to be able to access some private members with strong types (yack!)
        const node = construct.node
            ._actualNode;
        const allAspectsHere = [...(inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : []), ...node._aspects];
        for (const aspect of allAspectsHere) {
            if (node.invokedAspects.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            node.invokedAspects.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Prepare all constructs in the given construct tree in post-order.
 *
 * Stop at Assembly boundaries.
 */
function prepareTree(root) {
    visit(root, "post", (construct) => construct.onPrepare());
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder) {
    visit(root, "post", (construct) => construct.onSynthesize({
        outdir: builder.outdir,
        assembly: builder,
    }));
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    visit(root, "pre", (construct) => {
        for (const message of construct.onValidate()) {
            errors.push({ message, source: construct });
        }
    });
    if (errors.length > 0) {
        const errorList = errors
            .map((e) => `[${e.source.node.path}] ${e.message}`)
            .join("\n  ");
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === "pre") {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === "post") {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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