"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RosReference = void 0;
const stack_1 = require("../stack");
const reference_1 = require("../reference");
const token_1 = require("../token");
const ROS_REFERENCE_SYMBOL = Symbol.for("ros-cdk-core.RosReference");
/**
 * A Token that represents a ROS reference to another resource
 *
 * Additionally, the dependency between the stacks will be recorded, and the toolkit
 * will make sure to deploy producing stack before the consuming stack.
 *
 * This magic happens in the prepare() phase, where consuming stacks will call
 * `consumeFromStack` on these Tokens and if they happen to be exported by a different
 * Stack, we'll register the dependency.
 */
class RosReference extends reference_1.Reference {
    constructor(value, displayName, target) {
        // prepend scope path to display name
        super(value, target, displayName);
        this.replacementTokens = new Map();
        this.targetStack = stack_1.Stack.of(target);
        Object.defineProperty(this, ROS_REFERENCE_SYMBOL, { value: true });
    }
    /**
     * Check whether this is actually a Reference
     */
    static isRosReference(x) {
        return ROS_REFERENCE_SYMBOL in x;
    }
    /**
     * Return the RosReference for the indicated target
     *
     * Will make sure that multiple invocations for the same target and intrinsic
     * return the same RosReference. Because RosReferences accumulate state in
     * the prepare() phase (for the purpose of cross-stack references), it's
     * important that the state isn't lost if it's lazily created, like so:
     *
     *     Lazy.stringValue({ produce: () => new RosReference(...) })
     */
    static for(target, attribute) {
        return RosReference.singletonReference(target, attribute, () => {
            const rosIntrinsic = attribute === "Ref"
                ? { Ref: target.logicalId }
                : { "Fn::GetAtt": [target.logicalId, attribute] };
            return new RosReference(rosIntrinsic, attribute, target);
        });
    }
    /**
     * Return a RosReference that references a pseudo referencd
     */
    static forPseudo(pseudoName, scope) {
        return RosReference.singletonReference(scope, `Pseudo:${pseudoName}`, () => {
            const rosIntrinsic = { Ref: pseudoName };
            return new RosReference(rosIntrinsic, pseudoName, scope);
        });
    }
    /**
     * Get or create the table
     */
    static singletonReference(target, attribKey, fresh) {
        let attribs = RosReference.referenceTable.get(target);
        if (!attribs) {
            attribs = new Map();
            RosReference.referenceTable.set(target, attribs);
        }
        let ref = attribs.get(attribKey);
        if (!ref) {
            ref = fresh();
            attribs.set(attribKey, ref);
        }
        return ref;
    }
    resolve(context) {
        // If we have a special token for this consuming stack, resolve that. Otherwise resolve as if
        // we are in the same stack.
        const consumingStack = stack_1.Stack.of(context.scope);
        const token = this.replacementTokens.get(consumingStack);
        // if (!token && this.isCrossStackReference(consumingStack) && !context.preparing) {
        // tslint:disable-next-line:max-line-length
        //   throw new Error(`Cross-stack reference (${context.scope.node.path} -> ${this.target.node.path}) has not been assigned a value--call prepare() first`);
        // }
        if (token) {
            return token.resolve(context);
        }
        else {
            return super.resolve(context);
        }
    }
    hasValueForStack(stack) {
        if (stack === this.targetStack) {
            return true;
        }
        return this.replacementTokens.has(stack);
    }
    assignValueForStack(stack, value) {
        if (stack === this.targetStack) {
            throw new Error("cannot assign a value for the same stack");
        }
        if (this.hasValueForStack(stack)) {
            throw new Error("Cannot assign a reference value twice to the same stack. Use hasValueForStack to check first");
        }
        this.replacementTokens.set(stack, value);
    }
    /**
     * Implementation of toString() that will use the display name
     */
    toString() {
        return token_1.Token.asString(this, {
            displayHint: `${this.target.node.id}.${this.displayName}`,
        });
    }
}
exports.RosReference = RosReference;
/**
 * Static table where we keep singleton RosReference instances
 */
RosReference.referenceTable = new Map();
//# sourceMappingURL=data:application/json;base64,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