"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApp = void 0;
const constructs_1 = require("constructs");
const ros_resource_1 = require("../ros-resource");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    if (root.node.scope && !stage_1.Stage.isStage(root)) {
        throw new Error("prepareApp can only be called on a stage or a root construct");
    }
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetRosResources = findRosResources(dependency.target);
        const sourceRosResources = findRosResources(dependency.source);
        for (const target of targetRosResources) {
            for (const source of sourceRosResources) {
                source.addDependsOn(target);
            }
        }
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare its template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@alicloud/ros-cdk-core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    const includeStack = (stack) => {
        if (!stack_1.Stack.isStack(stack)) {
            return false;
        }
        if (!stack.nested) {
            return false;
        }
        // test: if we are not within a stage, then include it.
        if (!stage_1.Stage.of(stack)) {
            return true;
        }
        return stage_1.Stage.of(stack) === root;
    };
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (includeStack(stack)) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findRosResources(root) {
    return root.node.findAll().filter(ros_resource_1.RosResource.isRosResource);
}
//# sourceMappingURL=data:application/json;base64,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