"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isGeneratedWhenNeededMarker = exports.GeneratedWhenNeededMarker = exports.generatePhysicalName = void 0;
const stack_1 = require("../stack");
const crypto = require("crypto");
const token_map_1 = require("./token-map");
function generatePhysicalName(resource) {
    const stack = stack_1.Stack.of(resource);
    const stackPart = new PrefixNamePart(stack.stackName, 25);
    const idPart = new SuffixNamePart(resource.node.uniqueId, 24);
    const parts = [stackPart, idPart].map((part) => part.generate());
    const hashLength = 12;
    const sha256 = crypto
        .createHash("sha256")
        .update(stackPart.bareStr)
        .update(idPart.bareStr);
    const hash = sha256.digest("hex").slice(0, hashLength);
    const ret = [...parts, hash].join("");
    return ret.toLowerCase();
}
exports.generatePhysicalName = generatePhysicalName;
class NamePart {
    constructor(bareStr) {
        this.bareStr = bareStr;
    }
}
class PrefixNamePart extends NamePart {
    constructor(bareStr, prefixLength) {
        super(bareStr);
        this.prefixLength = prefixLength;
    }
    generate() {
        return this.bareStr.slice(0, this.prefixLength);
    }
}
class SuffixNamePart extends NamePart {
    constructor(str, suffixLength) {
        super(str);
        this.suffixLength = suffixLength;
    }
    generate() {
        const strLen = this.bareStr.length;
        const startIndex = Math.max(strLen - this.suffixLength, 0);
        return this.bareStr.slice(startIndex, strLen);
    }
}
const GENERATE_IF_NEEDED_SYMBOL = Symbol.for("ros-cdk-core.<private>.GenerateIfNeeded");
/**
 * This marker token is used by PhysicalName.GENERATE_IF_NEEDED. When that token is passed to the
 * physicalName property of a Resource, it triggers different behavior in the Resource constructor
 * that will allow emission of a generated physical name (when the resource is used across
 * environments) or undefined (when the resource is not shared).
 *
 * This token throws an Error when it is resolved, as a way to prevent inadvertent mis-uses of it.
 */
class GeneratedWhenNeededMarker {
    constructor() {
        this.creationStack = [];
        Object.defineProperty(this, GENERATE_IF_NEEDED_SYMBOL, { value: true });
    }
    resolve(_ctx) {
        throw new Error('Invalid physical name passed to ROS. Use "this.physicalName" instead');
    }
    toString() {
        return "PhysicalName.GENERATE_IF_NEEDED";
    }
}
exports.GeneratedWhenNeededMarker = GeneratedWhenNeededMarker;
/**
 * Checks whether a stringified token resolves to a `GeneratedWhenNeededMarker`.
 */
function isGeneratedWhenNeededMarker(val) {
    const token = token_map_1.TokenMap.instance().lookupString(val);
    return !!token && GENERATE_IF_NEEDED_SYMBOL in token;
}
exports.isGeneratedWhenNeededMarker = isGeneratedWhenNeededMarker;
//# sourceMappingURL=data:application/json;base64,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