"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogicalIDs = void 0;
/**
 * Class that keeps track of the logical IDs that are assigned to resources
 *
 * Supports renaming the generated IDs.
 */
class LogicalIDs {
    constructor() {
        /**
         * The rename table (old to new)
         */
        this.renames = {};
        /**
         * All assigned names (new to old, may be identical)
         *
         * This is used to ensure that:
         *
         * - No 2 resources end up with the same final logical ID, unless they were the same to begin with.
         * - All renames have been used at the end of renaming.
         */
        this.reverse = {};
    }
    /**
     * Rename a logical ID from an old ID to a new ID
     */
    addRename(oldId, newId) {
        if (oldId in this.renames) {
            throw new Error(`A rename has already been registered for '${oldId}'`);
        }
        this.renames[oldId] = newId;
    }
    /**
     * Return the renamed version of an ID or the original ID.
     */
    applyRename(oldId) {
        let newId = oldId;
        if (oldId in this.renames) {
            newId = this.renames[oldId];
        }
        // If this newId has already been used, it must have been with the same oldId
        if (newId in this.reverse && this.reverse[newId] !== oldId) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Two objects have been assigned the same Logical ID: '${this.reverse[newId]}' and '${oldId}' are now both named '${newId}'.`);
        }
        this.reverse[newId] = oldId;
        validateLogicalId(newId);
        return newId;
    }
    /**
     * Throw an error if not all renames have been used
     *
     * This is to assure that users didn't make typoes when registering renames.
     */
    assertAllRenamesApplied() {
        const keys = new Set();
        Object.keys(this.renames).forEach(keys.add.bind(keys));
        Object.keys(this.reverse).map((newId) => {
            keys.delete(this.reverse[newId]);
        });
        if (keys.size !== 0) {
            const unusedRenames = Array.from(keys.values());
            throw new Error(`The following Logical IDs were attempted to be renamed, but not found: ${unusedRenames.join(", ")}`);
        }
    }
}
exports.LogicalIDs = LogicalIDs;
const VALID_LOGICALID_REGEX = /^[A-Za-z][A-Za-z0-9]{1,254}$/;
/**
 * Validate logical ID is valid for ROS
 */
function validateLogicalId(logicalId) {
    if (!VALID_LOGICALID_REGEX.test(logicalId)) {
        throw new Error(`Logical ID must adhere to the regular expression: ${VALID_LOGICALID_REGEX.toString()}, got '${logicalId}'`);
    }
}
//# sourceMappingURL=data:application/json;base64,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