"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractTokenDouble = exports.createTokenDouble = exports.unresolved = exports.containsListTokenElement = exports.NullConcat = exports.regexQuote = exports.TokenString = exports.VALID_KEY_CHARS = exports.END_TOKEN_MARKER = exports.BEGIN_LIST_TOKEN_MARKER = exports.BEGIN_STRING_TOKEN_MARKER = void 0;
const string_fragments_1 = require("../string-fragments");
const token_1 = require("../token");
// Details for encoding and decoding Tokens into native types; should not be exported
exports.BEGIN_STRING_TOKEN_MARKER = "${Token[";
exports.BEGIN_LIST_TOKEN_MARKER = "#{Token[";
exports.END_TOKEN_MARKER = "]}";
exports.VALID_KEY_CHARS = "a-zA-Z0-9:._-";
const QUOTED_BEGIN_STRING_TOKEN_MARKER = regexQuote(exports.BEGIN_STRING_TOKEN_MARKER);
const QUOTED_BEGIN_LIST_TOKEN_MARKER = regexQuote(exports.BEGIN_LIST_TOKEN_MARKER);
const QUOTED_END_TOKEN_MARKER = regexQuote(exports.END_TOKEN_MARKER);
const STRING_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_STRING_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const LIST_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_LIST_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
/**
 * A string with markers in it that can be resolved to external values
 */
class TokenString {
    constructor(str, re) {
        this.str = str;
        this.re = re;
    }
    /**
     * Returns a `TokenString` for this string.
     */
    static forString(s) {
        return new TokenString(s, STRING_TOKEN_REGEX);
    }
    /**
     * Returns a `TokenString` for this string (must be the first string element of the list)
     */
    static forListToken(s) {
        return new TokenString(s, LIST_TOKEN_REGEX);
    }
    /**
     * Split string on markers, substituting markers with Tokens
     */
    split(lookup) {
        const ret = new string_fragments_1.TokenizedStringFragments();
        let rest = 0;
        this.re.lastIndex = 0; // Reset
        let m = this.re.exec(this.str);
        while (m) {
            if (m.index > rest) {
                ret.addLiteral(this.str.substring(rest, m.index));
            }
            ret.addToken(lookup(m[1]));
            rest = this.re.lastIndex;
            m = this.re.exec(this.str);
        }
        if (rest < this.str.length) {
            ret.addLiteral(this.str.substring(rest));
        }
        return ret;
    }
    /**
     * Indicates if this string includes tokens.
     */
    test() {
        this.re.lastIndex = 0; // Reset
        return this.re.test(this.str);
    }
}
exports.TokenString = TokenString;
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, "\\$&");
}
exports.regexQuote = regexQuote;
/**
 * Concatenator that disregards the input
 *
 * Can be used when traversing the tokens is important, but the
 * result isn't.
 */
class NullConcat {
    join(_left, _right) {
        return undefined;
    }
}
exports.NullConcat = NullConcat;
function containsListTokenElement(xs) {
    return xs.some((x) => typeof x === "string" && TokenString.forListToken(x).test());
}
exports.containsListTokenElement = containsListTokenElement;
/**
 * Returns true if obj is a token (i.e. has the resolve() method or is a string
 * that includes token markers), or it's a listifictaion of a Token string.
 *
 * @param obj The object to test.
 */
function unresolved(obj) {
    if (typeof obj === "string") {
        return TokenString.forString(obj).test();
    }
    else if (typeof obj === "number") {
        return extractTokenDouble(obj) !== undefined;
    }
    else if (Array.isArray(obj) && obj.length === 1) {
        return (typeof obj[0] === "string" && TokenString.forListToken(obj[0]).test());
    }
    else {
        return token_1.isResolvableObject(obj);
    }
}
exports.unresolved = unresolved;
/**
 * Bit pattern in the top 16 bits of a double to indicate a Token
 *
 * An IEEE double in LE memory order looks like this (grouped
 * into octets, then grouped into 32-bit words):
 *
 * mmmmmmmm.mmmmmmmm.mmmmmmmm.mmmmmmmm | mmmmmmmm.mmmmmmmm.EEEEmmmm.sEEEEEEE
 *
 * - m: mantissa (52 bits)
 * - E: exponent (11 bits)
 * - s: sign (1 bit)
 *
 * We put the following marker into the top 16 bits (exponent and sign), and
 * use the mantissa part to encode the token index. To save some bit twiddling
 * we use all top 16 bits for the tag. That loses us 4 mantissa bits to store
 * information in but we still have 48, which is going to be plenty for any
 * number of tokens to be created during the lifetime of any CDK application.
 *
 * Can't have all bits set because that makes a NaN, so unset the least
 * significant exponent bit.
 *
 * Currently not supporting BE architectures.
 */
// tslint:disable-next-line:no-bitwise
const DOUBLE_TOKEN_MARKER_BITS = 0xfbff << 16;
/**
 * Highest encodable number
 */
const MAX_ENCODABLE_INTEGER = Math.pow(2, 48) - 1;
/**
 * Get 2^32 as a number, so we can do multiplication and div instead of bit shifting
 *
 * Necessary because in JavaScript, bit operations implicitly convert
 * to int32 and we need them to work on "int64"s.
 *
 * So instead of x >> 32, we do Math.floor(x / 2^32), and vice versa.
 */
const BITS32 = Math.pow(2, 32);
/**
 * Return a special Double value that encodes the given nonnegative integer
 *
 * We use this to encode Token ordinals.
 */
function createTokenDouble(x) {
    if (Math.floor(x) !== x || x < 0) {
        throw new Error("Can only encode positive integers");
    }
    if (x > MAX_ENCODABLE_INTEGER) {
        throw new Error(`Got an index too large to encode: ${x}`);
    }
    const buf = new ArrayBuffer(8);
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    ints[0] = x & 0x0000ffffffff; // Bottom 32 bits of number
    // This needs an "x >> 32" but that will make it a 32-bit number instead
    // of a 64-bit number.
    ints[1] = (shr32(x) & 0xffff) | DOUBLE_TOKEN_MARKER_BITS; // Top 16 bits of number and the mask
    // tslint:enable:no-bitwise
    return new Float64Array(buf)[0];
}
exports.createTokenDouble = createTokenDouble;
/**
 * Shift a 64-bit int right 32 bits
 */
function shr32(x) {
    return Math.floor(x / BITS32);
}
/**
 * Shift a 64-bit left 32 bits
 */
function shl32(x) {
    return x * BITS32;
}
/**
 * Extract the encoded integer out of the special Double value
 *
 * Returns undefined if the float is a not an encoded token.
 */
function extractTokenDouble(encoded) {
    const buf = new ArrayBuffer(8);
    new Float64Array(buf)[0] = encoded;
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    if ((ints[1] & 0xffff0000) !== DOUBLE_TOKEN_MARKER_BITS) {
        return undefined;
    }
    // Must use + instead of | here (bitwise operations
    // will force 32-bits integer arithmetic, + will not).
    return ints[0] + shl32(ints[1] & 0xffff);
    // tslint:enable:no-bitwise
}
exports.extractTokenDouble = extractTokenDouble;
//# sourceMappingURL=data:application/json;base64,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