"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NestedStack = void 0;
const construct_compat_1 = require("./construct-compat");
const stack_1 = require("./stack");
const token_1 = require("./token");
const lazy_1 = require("./lazy");
const ros_fn_1 = require("./ros-fn");
const ros_pseudo_1 = require("./ros-pseudo");
const ros_stack_1 = require("./ros-stack");
const ros_element_1 = require("./ros-element");
const NESTED_STACK_SYMBOL = Symbol.for("ros-cdk-core.NestedStack");
class NestedStack extends stack_1.Stack {
    constructor(scope, id, props = {}) {
        super(scope, id);
        let stackProps = {
            parameters: lazy_1.Lazy.anyValue({
                produce: () => Object.keys(this.parameters).length > 0 ? this.parameters : undefined,
            }),
            timeoutMins: props.timeout,
        };
        if (props.templateUrl) {
            stackProps.templateUrl = props.templateUrl;
        }
        else {
            stackProps.templateBody = lazy_1.Lazy.stringValue({
                produce: () => this.templateBody,
            });
        }
        const parentScope = new construct_compat_1.Construct(scope, id + ".NestedStack");
        this.resource = new ros_stack_1.RosStack(parentScope, `${id}`, stackProps);
        this._contextualStackName = this.contextualAttribute(ros_pseudo_1.RosPseudo.stackName, ros_fn_1.Fn.select(1, ros_fn_1.Fn.split("/", this.resource.ref)));
        this._contextualStackId = this.contextualAttribute(ros_pseudo_1.RosPseudo.stackId, this.resource.ref);
        Object.defineProperty(this, NESTED_STACK_SYMBOL, { value: true });
        this.parameters = props.parameters || {};
    }
    static isNestedStack(x) {
        return x != null && typeof x === "object" && NESTED_STACK_SYMBOL in x;
    }
    /**
     * @internal
     */
    _toRosTemplate() {
        const template = {};
        const elements = rosElements(this);
        if (elements.length === 0)
            return;
        const fragments = elements.map((e) => this.resolve(e._toRosTemplate()));
        for (const fragment of fragments) {
            merge(template, fragment);
        }
        this.templateBody = this.resolve(template);
        return this.templateBody;
    }
    get stackName() {
        return this._contextualStackName;
    }
    get stackId() {
        return this._contextualStackId;
    }
    setParameter(name, value) {
        this.parameters[name] = value;
    }
    getAtt(attributeName) {
        return token_1.Token.asString(this.resource.getAtt("Outputs." + attributeName));
    }
    contextualAttribute(innerValue, outerValue) {
        return token_1.Token.asString({
            resolve: (context) => {
                if (stack_1.Stack.of(context.scope) === this) {
                    return innerValue;
                }
                else {
                    return outerValue;
                }
            },
        });
    }
}
exports.NestedStack = NestedStack;
function rosElements(node, into = []) {
    if (ros_element_1.RosElement.isRosElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (stack_1.Stack.isStack(child)) {
            continue;
        }
        rosElements(child, into);
    }
    return into;
}
function merge(template, fragment) {
    for (const section of Object.keys(fragment)) {
        const src = fragment[section];
        // create top-level section if it doesn't exist
        const dest = template[section];
        if (!dest) {
            template[section] = src;
        }
        else {
            template[section] = mergeSection(section, dest, src);
        }
    }
}
function mergeSection(section, val1, val2) {
    switch (section) {
        case "Description":
            return `${val1}\n${val2}`;
        case "Resources":
        case "Conditions":
        case "Parameters":
        case "Outputs":
        case "Mappings":
        case "Metadata":
            return mergeObjectsWithoutDuplicates(section, val1, val2);
        default:
            throw new Error(`CDK doesn't know how to merge two instances of the template section '${section}' - ` +
                "please remove one of them from your code");
    }
}
function mergeObjectsWithoutDuplicates(section, dest, src) {
    if (typeof dest !== "object") {
        throw new Error(`Expecting ${JSON.stringify(dest)} to be an object`);
    }
    if (typeof src !== "object") {
        throw new Error(`Expecting ${JSON.stringify(src)} to be an object`);
    }
    // add all entities from source section to destination section
    for (const id of Object.keys(src)) {
        if (id in dest) {
            throw new Error(`section '${section}' already contains '${id}'`);
        }
        dest[id] = src[id];
    }
    return dest;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmVzdGVkLXN0YWNrLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibmVzdGVkLXN0YWNrLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHlEQUEyRDtBQUMzRCxtQ0FBZ0M7QUFFaEMsbUNBQWdDO0FBQ2hDLGlDQUE4QjtBQUM5QixxQ0FBOEI7QUFDOUIsNkNBQXlDO0FBQ3pDLDJDQUF1QztBQUN2QywrQ0FBMkM7QUFHM0MsTUFBTSxtQkFBbUIsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7QUFRbkUsTUFBYSxXQUFZLFNBQVEsYUFBSztJQVdwQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQTBCLEVBQUU7UUFDcEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLFVBQVUsR0FBMkI7WUFDdkMsVUFBVSxFQUFFLFdBQUksQ0FBQyxRQUFRLENBQUM7Z0JBQ3hCLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FDWixNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQ3hFLENBQUM7WUFDRixXQUFXLEVBQUUsS0FBSyxDQUFDLE9BQU87U0FDM0IsQ0FBQztRQUVGLElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRTtZQUNyQixVQUFVLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7U0FDNUM7YUFBTTtZQUNMLFVBQVUsQ0FBQyxZQUFZLEdBQUcsV0FBSSxDQUFDLFdBQVcsQ0FBQztnQkFDekMsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxZQUFZO2FBQ2pDLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxXQUFXLEdBQUcsSUFBSSw0QkFBUyxDQUFDLEtBQUssRUFBRSxFQUFFLEdBQUcsY0FBYyxDQUFDLENBQUM7UUFDOUQsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLG9CQUFRLENBQUMsV0FBVyxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDL0QsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FDbEQsc0JBQVMsQ0FBQyxTQUFTLEVBQ25CLFdBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLFdBQUUsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FDL0MsQ0FBQztRQUNGLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQ2hELHNCQUFTLENBQUMsT0FBTyxFQUNqQixJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FDbEIsQ0FBQztRQUNGLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDbEUsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQztJQUMzQyxDQUFDO0lBeENNLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBTTtRQUNoQyxPQUFPLENBQUMsSUFBSSxJQUFJLElBQUksT0FBTyxDQUFDLEtBQUssUUFBUSxJQUFJLG1CQUFtQixJQUFJLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBd0NEOztPQUVHO0lBQ0gsY0FBYztRQUNaLE1BQU0sUUFBUSxHQUFRLEVBQUUsQ0FBQztRQUN6QixNQUFNLFFBQVEsR0FBRyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbkMsSUFBSSxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUM7WUFBRSxPQUFPO1FBRWxDLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN4RSxLQUFLLE1BQU0sUUFBUSxJQUFJLFNBQVMsRUFBRTtZQUNoQyxLQUFLLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1NBQzNCO1FBQ0QsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQztJQUMzQixDQUFDO0lBRUQsSUFBVyxTQUFTO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLG9CQUFvQixDQUFDO0lBQ25DLENBQUM7SUFFRCxJQUFXLE9BQU87UUFDaEIsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUM7SUFDakMsQ0FBQztJQUVNLFlBQVksQ0FBQyxJQUFZLEVBQUUsS0FBVTtRQUMxQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssQ0FBQztJQUNoQyxDQUFDO0lBRU0sTUFBTSxDQUFDLGFBQXFCO1FBQ2pDLE9BQU8sYUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxVQUFVLEdBQUcsYUFBYSxDQUFDLENBQUMsQ0FBQztJQUMxRSxDQUFDO0lBRU8sbUJBQW1CLENBQUMsVUFBa0IsRUFBRSxVQUFrQjtRQUNoRSxPQUFPLGFBQUssQ0FBQyxRQUFRLENBQUM7WUFDcEIsT0FBTyxFQUFFLENBQUMsT0FBd0IsRUFBRSxFQUFFO2dCQUNwQyxJQUFJLGFBQUssQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxLQUFLLElBQUksRUFBRTtvQkFDcEMsT0FBTyxVQUFVLENBQUM7aUJBQ25CO3FCQUFNO29CQUNMLE9BQU8sVUFBVSxDQUFDO2lCQUNuQjtZQUNILENBQUM7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUF0RkQsa0NBc0ZDO0FBRUQsU0FBUyxXQUFXLENBQUMsSUFBZ0IsRUFBRSxPQUFxQixFQUFFO0lBQzVELElBQUksd0JBQVUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEVBQUU7UUFDakMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztLQUNqQjtJQUNELEtBQUssTUFBTSxLQUFLLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUU7UUFDdEMsZ0NBQWdDO1FBQ2hDLElBQUksYUFBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUN4QixTQUFTO1NBQ1Y7UUFDRCxXQUFXLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO0tBQzFCO0lBQ0QsT0FBTyxJQUFJLENBQUM7QUFDZCxDQUFDO0FBRUQsU0FBUyxLQUFLLENBQUMsUUFBYSxFQUFFLFFBQWE7SUFDekMsS0FBSyxNQUFNLE9BQU8sSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQzNDLE1BQU0sR0FBRyxHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUU5QiwrQ0FBK0M7UUFDL0MsTUFBTSxJQUFJLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQy9CLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDVCxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsR0FBRyxDQUFDO1NBQ3pCO2FBQU07WUFDTCxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsWUFBWSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7U0FDdEQ7S0FDRjtBQUNILENBQUM7QUFFRCxTQUFTLFlBQVksQ0FBQyxPQUFlLEVBQUUsSUFBUyxFQUFFLElBQVM7SUFDekQsUUFBUSxPQUFPLEVBQUU7UUFDZixLQUFLLGFBQWE7WUFDaEIsT0FBTyxHQUFHLElBQUksS0FBSyxJQUFJLEVBQUUsQ0FBQztRQUM1QixLQUFLLFdBQVcsQ0FBQztRQUNqQixLQUFLLFlBQVksQ0FBQztRQUNsQixLQUFLLFlBQVksQ0FBQztRQUNsQixLQUFLLFNBQVMsQ0FBQztRQUNmLEtBQUssVUFBVSxDQUFDO1FBQ2hCLEtBQUssVUFBVTtZQUNiLE9BQU8sNkJBQTZCLENBQUMsT0FBTyxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztRQUM1RDtZQUNFLE1BQU0sSUFBSSxLQUFLLENBQ2Isd0VBQXdFLE9BQU8sTUFBTTtnQkFDbkYsMENBQTBDLENBQzdDLENBQUM7S0FDTDtBQUNILENBQUM7QUFFRCxTQUFTLDZCQUE2QixDQUNwQyxPQUFlLEVBQ2YsSUFBUyxFQUNULEdBQVE7SUFFUixJQUFJLE9BQU8sSUFBSSxLQUFLLFFBQVEsRUFBRTtRQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLGFBQWEsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsQ0FBQztLQUN0RTtJQUNELElBQUksT0FBTyxHQUFHLEtBQUssUUFBUSxFQUFFO1FBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsYUFBYSxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0tBQ3JFO0lBRUQsOERBQThEO0lBQzlELEtBQUssTUFBTSxFQUFFLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUNqQyxJQUFJLEVBQUUsSUFBSSxJQUFJLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLFlBQVksT0FBTyx1QkFBdUIsRUFBRSxHQUFHLENBQUMsQ0FBQztTQUNsRTtRQUNELElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7S0FDcEI7SUFFRCxPQUFPLElBQUksQ0FBQztBQUNkLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25zdHJ1Y3QsIElDb25zdHJ1Y3QgfSBmcm9tIFwiLi9jb25zdHJ1Y3QtY29tcGF0XCI7XHJcbmltcG9ydCB7IFN0YWNrIH0gZnJvbSBcIi4vc3RhY2tcIjtcclxuXHJcbmltcG9ydCB7IFRva2VuIH0gZnJvbSBcIi4vdG9rZW5cIjtcclxuaW1wb3J0IHsgTGF6eSB9IGZyb20gXCIuL2xhenlcIjtcclxuaW1wb3J0IHsgRm4gfSBmcm9tIFwiLi9yb3MtZm5cIjtcclxuaW1wb3J0IHsgUm9zUHNldWRvIH0gZnJvbSBcIi4vcm9zLXBzZXVkb1wiO1xyXG5pbXBvcnQgeyBSb3NTdGFjayB9IGZyb20gXCIuL3Jvcy1zdGFja1wiO1xyXG5pbXBvcnQgeyBSb3NFbGVtZW50IH0gZnJvbSBcIi4vcm9zLWVsZW1lbnRcIjtcclxuaW1wb3J0IHsgSVJlc29sdmVDb250ZXh0IH0gZnJvbSBcIi4vcmVzb2x2YWJsZVwiO1xyXG5cclxuY29uc3QgTkVTVEVEX1NUQUNLX1NZTUJPTCA9IFN5bWJvbC5mb3IoXCJyb3MtY2RrLWNvcmUuTmVzdGVkU3RhY2tcIik7XHJcbmV4cG9ydCBpbnRlcmZhY2UgTmVzdGVkU3RhY2tQcm9wcyB7XHJcbiAgcmVhZG9ubHkgcGFyYW1ldGVycz86IHsgW2tleTogc3RyaW5nXTogYW55IH07XHJcbiAgcmVhZG9ubHkgdGVtcGxhdGVVcmw/OiBzdHJpbmc7XHJcbiAgcmVhZG9ubHkgdGVtcGxhdGVCb2R5Pzogc3RyaW5nO1xyXG4gIHJlYWRvbmx5IHRpbWVvdXQ/OiBudW1iZXI7XHJcbn1cclxuXHJcbmV4cG9ydCBjbGFzcyBOZXN0ZWRTdGFjayBleHRlbmRzIFN0YWNrIHtcclxuICBwdWJsaWMgc3RhdGljIGlzTmVzdGVkU3RhY2soeDogYW55KTogeCBpcyBOZXN0ZWRTdGFjayB7XHJcbiAgICByZXR1cm4geCAhPSBudWxsICYmIHR5cGVvZiB4ID09PSBcIm9iamVjdFwiICYmIE5FU1RFRF9TVEFDS19TWU1CT0wgaW4geDtcclxuICB9XHJcblxyXG4gIHByaXZhdGUgcGFyYW1ldGVyczogeyBbbmFtZTogc3RyaW5nXTogYW55IH07XHJcbiAgcHJpdmF0ZSByZXNvdXJjZTogUm9zU3RhY2s7XHJcbiAgcHJpdmF0ZSBfY29udGV4dHVhbFN0YWNrSWQ6IHN0cmluZztcclxuICBwcml2YXRlIF9jb250ZXh0dWFsU3RhY2tOYW1lOiBzdHJpbmc7XHJcbiAgcHJpdmF0ZSB0ZW1wbGF0ZUJvZHk/OiBzdHJpbmc7XHJcblxyXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBOZXN0ZWRTdGFja1Byb3BzID0ge30pIHtcclxuICAgIHN1cGVyKHNjb3BlLCBpZCk7XHJcbiAgICBsZXQgc3RhY2tQcm9wczogeyBba2V5OiBzdHJpbmddOiBhbnkgfSA9IHtcclxuICAgICAgcGFyYW1ldGVyczogTGF6eS5hbnlWYWx1ZSh7XHJcbiAgICAgICAgcHJvZHVjZTogKCkgPT5cclxuICAgICAgICAgIE9iamVjdC5rZXlzKHRoaXMucGFyYW1ldGVycykubGVuZ3RoID4gMCA/IHRoaXMucGFyYW1ldGVycyA6IHVuZGVmaW5lZCxcclxuICAgICAgfSksXHJcbiAgICAgIHRpbWVvdXRNaW5zOiBwcm9wcy50aW1lb3V0LFxyXG4gICAgfTtcclxuXHJcbiAgICBpZiAocHJvcHMudGVtcGxhdGVVcmwpIHtcclxuICAgICAgc3RhY2tQcm9wcy50ZW1wbGF0ZVVybCA9IHByb3BzLnRlbXBsYXRlVXJsO1xyXG4gICAgfSBlbHNlIHtcclxuICAgICAgc3RhY2tQcm9wcy50ZW1wbGF0ZUJvZHkgPSBMYXp5LnN0cmluZ1ZhbHVlKHtcclxuICAgICAgICBwcm9kdWNlOiAoKSA9PiB0aGlzLnRlbXBsYXRlQm9keSxcclxuICAgICAgfSk7XHJcbiAgICB9XHJcblxyXG4gICAgY29uc3QgcGFyZW50U2NvcGUgPSBuZXcgQ29uc3RydWN0KHNjb3BlLCBpZCArIFwiLk5lc3RlZFN0YWNrXCIpO1xyXG4gICAgdGhpcy5yZXNvdXJjZSA9IG5ldyBSb3NTdGFjayhwYXJlbnRTY29wZSwgYCR7aWR9YCwgc3RhY2tQcm9wcyk7XHJcbiAgICB0aGlzLl9jb250ZXh0dWFsU3RhY2tOYW1lID0gdGhpcy5jb250ZXh0dWFsQXR0cmlidXRlKFxyXG4gICAgICBSb3NQc2V1ZG8uc3RhY2tOYW1lLFxyXG4gICAgICBGbi5zZWxlY3QoMSwgRm4uc3BsaXQoXCIvXCIsIHRoaXMucmVzb3VyY2UucmVmKSlcclxuICAgICk7XHJcbiAgICB0aGlzLl9jb250ZXh0dWFsU3RhY2tJZCA9IHRoaXMuY29udGV4dHVhbEF0dHJpYnV0ZShcclxuICAgICAgUm9zUHNldWRvLnN0YWNrSWQsXHJcbiAgICAgIHRoaXMucmVzb3VyY2UucmVmXHJcbiAgICApO1xyXG4gICAgT2JqZWN0LmRlZmluZVByb3BlcnR5KHRoaXMsIE5FU1RFRF9TVEFDS19TWU1CT0wsIHsgdmFsdWU6IHRydWUgfSk7XHJcbiAgICB0aGlzLnBhcmFtZXRlcnMgPSBwcm9wcy5wYXJhbWV0ZXJzIHx8IHt9O1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogQGludGVybmFsXHJcbiAgICovXHJcbiAgX3RvUm9zVGVtcGxhdGUoKSB7XHJcbiAgICBjb25zdCB0ZW1wbGF0ZTogYW55ID0ge307XHJcbiAgICBjb25zdCBlbGVtZW50cyA9IHJvc0VsZW1lbnRzKHRoaXMpO1xyXG4gICAgaWYgKGVsZW1lbnRzLmxlbmd0aCA9PT0gMCkgcmV0dXJuO1xyXG5cclxuICAgIGNvbnN0IGZyYWdtZW50cyA9IGVsZW1lbnRzLm1hcCgoZSkgPT4gdGhpcy5yZXNvbHZlKGUuX3RvUm9zVGVtcGxhdGUoKSkpO1xyXG4gICAgZm9yIChjb25zdCBmcmFnbWVudCBvZiBmcmFnbWVudHMpIHtcclxuICAgICAgbWVyZ2UodGVtcGxhdGUsIGZyYWdtZW50KTtcclxuICAgIH1cclxuICAgIHRoaXMudGVtcGxhdGVCb2R5ID0gdGhpcy5yZXNvbHZlKHRlbXBsYXRlKTtcclxuICAgIHJldHVybiB0aGlzLnRlbXBsYXRlQm9keTtcclxuICB9XHJcblxyXG4gIHB1YmxpYyBnZXQgc3RhY2tOYW1lKCkge1xyXG4gICAgcmV0dXJuIHRoaXMuX2NvbnRleHR1YWxTdGFja05hbWU7XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgZ2V0IHN0YWNrSWQoKSB7XHJcbiAgICByZXR1cm4gdGhpcy5fY29udGV4dHVhbFN0YWNrSWQ7XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgc2V0UGFyYW1ldGVyKG5hbWU6IHN0cmluZywgdmFsdWU6IGFueSkge1xyXG4gICAgdGhpcy5wYXJhbWV0ZXJzW25hbWVdID0gdmFsdWU7XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgZ2V0QXR0KGF0dHJpYnV0ZU5hbWU6IHN0cmluZyk6IHN0cmluZyB7XHJcbiAgICByZXR1cm4gVG9rZW4uYXNTdHJpbmcodGhpcy5yZXNvdXJjZS5nZXRBdHQoXCJPdXRwdXRzLlwiICsgYXR0cmlidXRlTmFtZSkpO1xyXG4gIH1cclxuXHJcbiAgcHJpdmF0ZSBjb250ZXh0dWFsQXR0cmlidXRlKGlubmVyVmFsdWU6IHN0cmluZywgb3V0ZXJWYWx1ZTogc3RyaW5nKSB7XHJcbiAgICByZXR1cm4gVG9rZW4uYXNTdHJpbmcoe1xyXG4gICAgICByZXNvbHZlOiAoY29udGV4dDogSVJlc29sdmVDb250ZXh0KSA9PiB7XHJcbiAgICAgICAgaWYgKFN0YWNrLm9mKGNvbnRleHQuc2NvcGUpID09PSB0aGlzKSB7XHJcbiAgICAgICAgICByZXR1cm4gaW5uZXJWYWx1ZTtcclxuICAgICAgICB9IGVsc2Uge1xyXG4gICAgICAgICAgcmV0dXJuIG91dGVyVmFsdWU7XHJcbiAgICAgICAgfVxyXG4gICAgICB9LFxyXG4gICAgfSk7XHJcbiAgfVxyXG59XHJcblxyXG5mdW5jdGlvbiByb3NFbGVtZW50cyhub2RlOiBJQ29uc3RydWN0LCBpbnRvOiBSb3NFbGVtZW50W10gPSBbXSk6IFJvc0VsZW1lbnRbXSB7XHJcbiAgaWYgKFJvc0VsZW1lbnQuaXNSb3NFbGVtZW50KG5vZGUpKSB7XHJcbiAgICBpbnRvLnB1c2gobm9kZSk7XHJcbiAgfVxyXG4gIGZvciAoY29uc3QgY2hpbGQgb2Ygbm9kZS5ub2RlLmNoaWxkcmVuKSB7XHJcbiAgICAvLyBEb24ndCByZWN1cnNlIGludG8gYSBzdWJzdGFja1xyXG4gICAgaWYgKFN0YWNrLmlzU3RhY2soY2hpbGQpKSB7XHJcbiAgICAgIGNvbnRpbnVlO1xyXG4gICAgfVxyXG4gICAgcm9zRWxlbWVudHMoY2hpbGQsIGludG8pO1xyXG4gIH1cclxuICByZXR1cm4gaW50bztcclxufVxyXG5cclxuZnVuY3Rpb24gbWVyZ2UodGVtcGxhdGU6IGFueSwgZnJhZ21lbnQ6IGFueSk6IHZvaWQge1xyXG4gIGZvciAoY29uc3Qgc2VjdGlvbiBvZiBPYmplY3Qua2V5cyhmcmFnbWVudCkpIHtcclxuICAgIGNvbnN0IHNyYyA9IGZyYWdtZW50W3NlY3Rpb25dO1xyXG5cclxuICAgIC8vIGNyZWF0ZSB0b3AtbGV2ZWwgc2VjdGlvbiBpZiBpdCBkb2Vzbid0IGV4aXN0XHJcbiAgICBjb25zdCBkZXN0ID0gdGVtcGxhdGVbc2VjdGlvbl07XHJcbiAgICBpZiAoIWRlc3QpIHtcclxuICAgICAgdGVtcGxhdGVbc2VjdGlvbl0gPSBzcmM7XHJcbiAgICB9IGVsc2Uge1xyXG4gICAgICB0ZW1wbGF0ZVtzZWN0aW9uXSA9IG1lcmdlU2VjdGlvbihzZWN0aW9uLCBkZXN0LCBzcmMpO1xyXG4gICAgfVxyXG4gIH1cclxufVxyXG5cclxuZnVuY3Rpb24gbWVyZ2VTZWN0aW9uKHNlY3Rpb246IHN0cmluZywgdmFsMTogYW55LCB2YWwyOiBhbnkpOiBhbnkge1xyXG4gIHN3aXRjaCAoc2VjdGlvbikge1xyXG4gICAgY2FzZSBcIkRlc2NyaXB0aW9uXCI6XHJcbiAgICAgIHJldHVybiBgJHt2YWwxfVxcbiR7dmFsMn1gO1xyXG4gICAgY2FzZSBcIlJlc291cmNlc1wiOlxyXG4gICAgY2FzZSBcIkNvbmRpdGlvbnNcIjpcclxuICAgIGNhc2UgXCJQYXJhbWV0ZXJzXCI6XHJcbiAgICBjYXNlIFwiT3V0cHV0c1wiOlxyXG4gICAgY2FzZSBcIk1hcHBpbmdzXCI6XHJcbiAgICBjYXNlIFwiTWV0YWRhdGFcIjpcclxuICAgICAgcmV0dXJuIG1lcmdlT2JqZWN0c1dpdGhvdXREdXBsaWNhdGVzKHNlY3Rpb24sIHZhbDEsIHZhbDIpO1xyXG4gICAgZGVmYXVsdDpcclxuICAgICAgdGhyb3cgbmV3IEVycm9yKFxyXG4gICAgICAgIGBDREsgZG9lc24ndCBrbm93IGhvdyB0byBtZXJnZSB0d28gaW5zdGFuY2VzIG9mIHRoZSB0ZW1wbGF0ZSBzZWN0aW9uICcke3NlY3Rpb259JyAtIGAgK1xyXG4gICAgICAgICAgXCJwbGVhc2UgcmVtb3ZlIG9uZSBvZiB0aGVtIGZyb20geW91ciBjb2RlXCJcclxuICAgICAgKTtcclxuICB9XHJcbn1cclxuXHJcbmZ1bmN0aW9uIG1lcmdlT2JqZWN0c1dpdGhvdXREdXBsaWNhdGVzKFxyXG4gIHNlY3Rpb246IHN0cmluZyxcclxuICBkZXN0OiBhbnksXHJcbiAgc3JjOiBhbnlcclxuKTogYW55IHtcclxuICBpZiAodHlwZW9mIGRlc3QgIT09IFwib2JqZWN0XCIpIHtcclxuICAgIHRocm93IG5ldyBFcnJvcihgRXhwZWN0aW5nICR7SlNPTi5zdHJpbmdpZnkoZGVzdCl9IHRvIGJlIGFuIG9iamVjdGApO1xyXG4gIH1cclxuICBpZiAodHlwZW9mIHNyYyAhPT0gXCJvYmplY3RcIikge1xyXG4gICAgdGhyb3cgbmV3IEVycm9yKGBFeHBlY3RpbmcgJHtKU09OLnN0cmluZ2lmeShzcmMpfSB0byBiZSBhbiBvYmplY3RgKTtcclxuICB9XHJcblxyXG4gIC8vIGFkZCBhbGwgZW50aXRpZXMgZnJvbSBzb3VyY2Ugc2VjdGlvbiB0byBkZXN0aW5hdGlvbiBzZWN0aW9uXHJcbiAgZm9yIChjb25zdCBpZCBvZiBPYmplY3Qua2V5cyhzcmMpKSB7XHJcbiAgICBpZiAoaWQgaW4gZGVzdCkge1xyXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYHNlY3Rpb24gJyR7c2VjdGlvbn0nIGFscmVhZHkgY29udGFpbnMgJyR7aWR9J2ApO1xyXG4gICAgfVxyXG4gICAgZGVzdFtpZF0gPSBzcmNbaWRdO1xyXG4gIH1cclxuXHJcbiAgcmV0dXJuIGRlc3Q7XHJcbn1cclxuIl19