"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lazy = void 0;
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * Lazily produce a value
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 */
class Lazy {
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer), options);
    }
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer));
    }
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, options), options);
    }
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, options);
    }
    constructor() { }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor() {
        this.creationStack = stack_trace_1.captureStackTrace();
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return "<unresolved-lazy>";
    }
}
class LazyString extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyNumber extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyList extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (ret !== undefined && ret.length === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return ret;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (Array.isArray(ret) && ret.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,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