"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addDependency = void 0;
const stack_1 = require("./stack");
const stage_1 = require("./stage");
const util_1 = require("./util");
/**
 * Adds a dependency between two resources or stacks, across stack and nested
 * stack boundaries.
 *
 * The algorithm consists of:
 * - Try to find the deepest common stack between the two elements
 * - If there isn't a common stack, it means the elements belong to two
 *   disjoined stack-trees and therefore we apply the dependency at the
 *   assembly/app level between the two topl-level stacks.
 * - If we did find a common stack, we apply the dependency as a template
 *   "DependsOn" between the resources that "represent" our source and target
 *   either directly or through the Stack resources that
 *   "lead" to them.
 *
 * @param source The source resource/stack (the depedent)
 * @param target The target resource/stack (the dependency)
 * @param reason Optional resource to associate with the dependency for
 * diagnostics
 */
function addDependency(source, target, reason) {
    if (source === target) {
        return;
    }
    const sourceStack = stack_1.Stack.of(source);
    const targetStack = stack_1.Stack.of(target);
    const sourceStage = stage_1.Stage.of(sourceStack);
    const targetStage = stage_1.Stage.of(targetStack);
    if (sourceStage !== targetStage) {
        throw new Error(`You cannot add a dependency from '${source.node.path}' (in ${describeStage(sourceStage)}) to '${target.node.path}' (in ${describeStage(targetStage)}): dependency cannot cross stage boundaries`);
    }
    // find the deepest common stack between the two elements
    const sourcePath = util_1.pathToTopLevelStack(sourceStack);
    const targetPath = util_1.pathToTopLevelStack(targetStack);
    const commonStack = util_1.findLastCommonElement(sourcePath, targetPath);
    // if there is no common stack, then define a assembly-level dependency
    // between the two top-level stacks
    if (!commonStack) {
        const topLevelSource = sourcePath[0]; // first path element is the top-level stack
        const topLevelTarget = targetPath[0];
        topLevelSource._addAssemblyDependency(topLevelTarget, reason);
        return;
    }
    // assertion: at this point if source and target are stacks, both are nested stacks.
    // since we have a common stack, it is impossible that both are top-level
    // stacks, so let's examine the two cases where one of them is top-level and
    // the other is nested.
    // case 1 - source is top-level and target is nested: this implies that
    // `target` is a direct or indirect nested stack of `source`, and an explicit
    // dependency is not required because nested stacks will always be deployed
    // before their parents.
    if (commonStack === source) {
        return;
    }
    // case 2 - source is nested and target is top-level: this implies that
    // `source` is a direct or indirect nested stack of `target`, and this is not
    // possible (nested stacks cannot depend on their parents).
    if (commonStack === target) {
        throw new Error(`Nested stack '${sourceStack.node.path}' cannot depend on a parent stack '${targetStack.node.path}': ${reason}`);
    }
    // we have a common stack from which we can reach both `source` and `target`
    // now we need to find two resources which are defined directly in this stack
    // and which can "lead us" to the source/target.
    const sourceResource = resourceInCommonStackFor(source);
    const targetResource = resourceInCommonStackFor(target);
    sourceResource._addResourceDependency(targetResource);
    function resourceInCommonStackFor(element) {
        const resource = stack_1.Stack.isStack(element)
            ? element.nestedStackResource
            : element;
        if (!resource) {
            throw new Error("assertion failure"); // see "assertion" above
        }
        const resourceStack = stack_1.Stack.of(resource);
        // we reached a resource defined in the common stack
        if (commonStack === resourceStack) {
            return resource;
        }
        return resourceInCommonStackFor(resourceStack);
    }
}
exports.addDependency = addDependency;
/**
 * Return a string representation of the given assembler, for use in error messages
 */
function describeStage(assembly) {
    if (!assembly) {
        return "an unrooted construct tree";
    }
    if (!assembly.parentStage) {
        return "the App";
    }
    return `Stage '${assembly.node.path}'`;
}
//# sourceMappingURL=data:application/json;base64,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