"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependableTrait = exports.ConcreteDependable = void 0;
/**
 * A set of constructs to be used as a dependable
 *
 * This class can be used when a set of constructs which are disjoint in the
 * construct tree needs to be combined to be used as a single dependable.
 *
 * @experimental
 */
class ConcreteDependable {
    constructor() {
        this._dependencyRoots = new Array();
        const self = this;
        DependableTrait.implement(this, {
            get dependencyRoots() {
                return self._dependencyRoots;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(construct) {
        this._dependencyRoots.push(construct);
    }
}
exports.ConcreteDependable = ConcreteDependable;
const DEPENDABLE_SYMBOL = Symbol.for("ros-cdk-core.DependableTrait");
/**
 * Trait for IDependable
 *
 * Traits are interfaces that are privately implemented by objects. Instead of
 * showing up in the public interface of a class, they need to be queried
 * explicitly. This is used to implement certain framework features that are
 * not intended to be used by Construct consumers, and so should be hidden
 * from accidental use.
 *
 * @example
 *
 * // Usage
 * const roots = DependableTrait.get(construct).dependencyRoots;
 *
 * // Definition
 * DependableTrait.implement(construct, {
 *   get dependencyRoots() { return []; }
 * });
 *
 * @experimental
 */
class DependableTrait {
    /**
     * Register `instance` to have the given DependableTrait
     *
     * Should be called in the class constructor.
     */
    static implement(instance, trait) {
        // I would also like to reference classes (to cut down on the list of objects
        // we need to manage), but we can't do that either since jsii doesn't have the
        // concept of a class reference.
        instance[DEPENDABLE_SYMBOL] = trait;
    }
    /**
     * Return the matching DependableTrait for the given class instance.
     */
    static get(instance) {
        const ret = instance[DEPENDABLE_SYMBOL];
        if (!ret) {
            throw new Error(`${instance} does not implement DependableTrait`);
        }
        return ret;
    }
}
exports.DependableTrait = DependableTrait;
//# sourceMappingURL=data:application/json;base64,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